#!/usr/bin/env python
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CloudHaSm
import Tac
import Tracing
import weakref

t0 = Tracing.t0
t1 = Tracing.t1

# This SM reacts to ipIntfStatus for IP/VRF changes to the given BFD interface
class IpIntfStatusReactor( Tac.Notifiee ):
   notifierTypeName = "Ip::IpIntfStatus"
   def __init__( self, parent, notifier ):
      t0( "IpIntfStatusReactor: Init" )
      self.parent = weakref.proxy( parent )
      self.ipIntfStatus = notifier
      Tac.Notifiee.__init__( self, notifier )

   @Tac.handler( 'activeAddrWithMask' )
   def handleActiveAddr( self ):
      t0( 'handleActiveddr notified' )
      assert self.parent.srcIntf == self.ipIntfStatus.intfId
      addr = self.ipIntfStatus.activeAddrWithMask.address
      vrf = self.ipIntfStatus.vrf
      if addr != '0.0.0.0':
         t0( "interface IP address %s vrf %s added for %s. "
            "Retriggering config validation." % ( \
               addr, vrf, self.parent.srcIntf ) )
         if vrf != 'default' :
            t0( 'Error in vrf config: non-default VRF is not yet supported' ) 
         else:   
            t0( 'Triggering Cloud Ha config change' )
            self.parent.cloudHaSm.sendEvent( CloudHaSm.ConfigChanged() )
      else:
         # Let BFD bring us admin down if we were up.
         # And it should avoid failover... 
         t0( 'IP address is removed. Ignoring it.' )   
         

class IpStatusReactor( Tac.Notifiee ):
   notifierTypeName = "Ip::Status"
   def __init__( self, ipStatus, cloudHaSm, srcIntf ):
      t0( "IpStatusReactor: Init: self %s (id: %x), CloudHaSm %s srcIntf %s" % \
         ( self, id( self ), cloudHaSm, srcIntf ) )
      self.ipStatus = ipStatus
      # Weakref may not be needed here as we hold reference to cloudHaSM and 
      # not the other way.
      self.cloudHaSm = cloudHaSm 
      assert self.cloudHaSm
      self.srcIntf = srcIntf
      self.ipIntfReactor = None
      Tac.Notifiee.__init__( self, self.ipStatus )
      self.createIpIntfStatusReactor()

   def createIpIntfStatusReactor( self ):
      t0( 'createIpIntfStatusReactor called' )
      if self.ipStatus.ipIntfStatus.has_key( self.srcIntf ):
         t0( 'creating ipIntStatusReactor' )
         ipIntfStatus = self.ipStatus.ipIntfStatus[ self.srcIntf ]
         self.ipIntfReactor = IpIntfStatusReactor( self, ipIntfStatus )       

   def handleAllIpIntfStatus( self ):
      t0(' handleAllIntfStatus called, self %s (id %x) , cloudHaSm %s srcIntf %s'% \
            ( self, id( self ), self.cloudHaSm, self.srcIntf ) )
      if self.ipStatus.ipIntfStatus.has_key( self.srcIntf ):
         t0( "BFD Source interface IP status handler for %s." % self.srcIntf )
         ipIntfStatus = self.ipStatus.ipIntfStatus[ self.srcIntf ]
         assert ipIntfStatus
         ipAddr = ipIntfStatus.activeAddrWithMask.address
         if ipAddr != '0.0.0.0':
            t0( 'Assigned Ip address %s (vrf %s ) to interface  %s' % (  \
               ipAddr, ipIntfStatus.vrf, self.srcIntf ) )
            self.cloudHaSm.sendEvent( CloudHaSm.ConfigChanged() )
         else:
            t0( 'IP Address is not yet assigned' )
         # spawn a reactor to handle ip address changes
         self.ipIntfReactor = IpIntfStatusReactor( self, ipIntfStatus )
      else:
         t0( 'ipIntfStatus removed. Removing reactor' ) 
         self.ipIntfReactor = None
         # Should we handle deletion case too or let the BFD bring it
         # down. Ignore for now.

   @Tac.handler( 'ipIntfStatus' )
   def handleIpIntfStatus( self, intfId ):
      t0( "IpStatusReactor:handleIpIntfStatus for %s" % intfId )
      self.handleAllIpIntfStatus()

