# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import weakref
import Tac
import Tracing
from CloudException import ConfigInvalid, MissingConfigError, InvalidJson
import exceptions
import ReversibleSecretCli
import json
import Url
import urllib

t0 = Tracing.t0
t1 = Tracing.t1
t2 = Tracing.t2
t8 = Tracing.t8

def getIpAddrVrfFromIntf( ipStatus, intfName ):
   assert ipStatus
   ip = vrf = None
   ipIntf = ipStatus.ipIntfStatus.get( intfName, None )
   if ipIntf:
      if ipIntf.activeAddrWithMask.address != '0.0.0.0':
         ip = ipIntf.activeAddrWithMask.address
         vrf = ipIntf.vrf
   return ip, vrf

def getSdkAuthCredFromFile( localFileName ):
   try:
      with open( localFileName ) as f:
         return json.load( f )    
   except IOError as e:
      t0( 'Error handling config file %s' , e )
      raise MissingConfigError( localFileName )
   except ValueError as e:
      t0( 'Error handling config file %s' , e )
      raise InvalidJson( localFileName, e.message )

def dumpCookedUpConfig( config, space=0 ):
   if not space:
      t8( "********** Cloud HA Config for backend ***********") 
   if isinstance( config, dict ):
      for key, val in config.items():
         t8( ' ' * space, key, '--->', val )
         dumpCookedUpConfig( val, space + 3 )
   elif isinstance( config, list ):
      for val in config:
         dumpCookedUpConfig( val, space + 3 )
         
# Updates the Sysdb Status object
class BaseCloudStatusHandler( object ):
   def __init__( self, agent ):
      self.agent = weakref.proxy( agent )
      self.config = None
      self.sysdbStatus = agent.haStatus_ 
      assert self.sysdbStatus
      self.cloudHaConfig = agent.haConfig_ 

   def updateStatusRoutesFromConfig( self, confCopy ):
      config = self.agent.haConfig_
      confCopy.localRoutes.clear()
      confCopy.peerRoutes.clear()
      for v in config.localRoutes.values():
         confCopy.localRoutes.addMember( v )
      for v in config.peerRoutes.values():
         confCopy.peerRoutes.addMember( v )
   
   # This handles for Aws, Azure and Gcp configs.
   def updateStatusFromConfig( self ):
      t0( 'Updating agent status from config' )
      config = self.agent.haConfig_
      confCopy = self.sysdbStatus.config
      self.copySysdbBaseConfigToStatus( confCopy, config )
      confCopy.accessConfig = config.accessConfig
      self.updateStatusRoutesFromConfig( confCopy )
 
   # Initialize ha status for CLI use. 
   def initializeSysdbStatus( self ):
      self.updateStatusFromConfig() 
      # May not need as we copy the config 
      self.sysdbStatus.updatedTimeStamp = self.agent.haConfig_.lastChangeTime
      self.sysdbStatus.status = 'validating'
      self.sysdbStatus.state = 'init'
      self.sysdbStatus.lastValidationStartTime = Tac.now()
      self.sysdbStatus.lastValidationEndTime = 0

   def updateSysdbStatus( self, status=None, state=None ):
      t0( 'updateSysdbStatus called ', status, state )
      if status:
         if self.sysdbStatus.status == 'validating' and status == 'valid':
            self.sysdbStatus.lastValidationEndTime = Tac.now()
         self.sysdbStatus.status = status
      if state:
         self.sysdbStatus.state = state

   def updateFailoverTime( self ):
      self.sysdbStatus.lastFailoverTime = Tac.now()
      self.sysdbStatus.failovers += 1
   
   def updateRecoveryTime( self ):
      self.sysdbStatus.lastRecoveryTime = Tac.now()


   # Copy Base configs common between different cloud types
   @staticmethod
   def copySysdbBaseConfigToStatus( dest, config ):
      dest.enable = config.enable
      for i in dest.peer.keys():
         del dest.peer[ i ]
      for i in config.peer.values():
         dest.peer.addMember( i )
      dest.lastChangeTime = config.lastChangeTime   


class AwsCloudStatusHandler( BaseCloudStatusHandler ):
   def __init__( self, agent ):
      super( AwsCloudStatusHandler, self).__init__( agent )
      self.sysdbStatus.config = ( 'AwsConfigInSysdbStatus' , )
   
class AzureCloudStatusHandler( BaseCloudStatusHandler ):
   def __init__( self, agent ):
      super( AzureCloudStatusHandler, self).__init__( agent )
      self.sysdbStatus.config = ( 'AzureConfigInSysdbStatus' , )

class GcpCloudStatusHandler( BaseCloudStatusHandler ):
   def __init__( self, agent ):
      super( GcpCloudStatusHandler, self).__init__( agent )
      self.sysdbStatus.config = ( 'GcpConfigInSysdbStatus', )

   def updateStatusRoutesFromConfig( self, confCopy ):
      config = self.agent.haConfig_
      confCopy.localRoutes.clear()
      confCopy.peerRoutes.clear()
      for v in config.localRoutes:
         confCopy.localRoutes.add( v )
      for v in config.peerRoutes:
         confCopy.peerRoutes.add( v )
        
class BaseCloudConfigHandler( object ):
   configValidators = {}

   def __init__( self, agent ):
      self.agent = weakref.proxy( agent )
      self.cloudHaConfig = agent.haConfig_
      self.proxyConfig = agent.proxyConfig
      self.sysdbConfigSm = None
      self.config = None

   @staticmethod
   def validateConfig( haConfig, cloudType ):
      for validate in BaseCloudConfigHandler.configValidators.itervalues():
         if not validate( haConfig, cloudType ):  
            return False
      return True
  

   # update json based config based on sysdb for backend use.
   def addGenericConfig( self, dstConfig):
      sysdbConfig = self.cloudHaConfig
      config = dstConfig[ 'generalConfig' ] = dict()
      config[ 'enable_optional' ] = "True" if sysdbConfig.enable else \
         "False"
      # Currently support only single HA peer.
      if len( sysdbConfig.peer ) > 1:
         t0( 'More than one peer are not supported, bailing out' )
         raise ConfigInvalid( 'CloudHa agent currently supports only one peer' ) 
      for i in sysdbConfig.peer.values():
         # Currently support only single HA peer.
         config[ 'hysteresis_time_optional' ] = i.recoveryWaitTime

   def addHaPeerConfig( self, dstConfig ):
      sysdbConfig = self.cloudHaConfig
      config = dstConfig[ 'bfdConfig' ] = dict()
      if sysdbConfig.peer:
         t0( 'Adding peer specific config' )
         for i in sysdbConfig.peer.values():
            config[ 'peerVeosIp' ] = str( i.peerIp )
            config[ 'bfdSourceInterface' ] = i.bfdConfig.intf
            config[ 'bfdSessionType' ] = 'multihop' if i.bfdConfig.multihop else \
               'normal'

   # This converts a CLI config into python config usable for backend etc
   def convertSysdbConfig( self ):
      t0( 'convertSysdbConfig' )
      sysdbConfig = self.cloudHaConfig
      assert sysdbConfig
      haConfig = dict()
      self.addGenericConfig( haConfig )
      self.addHaPeerConfig( haConfig )
      self.addCloudSpecificConfig( haConfig )
      return haConfig

   def interfaceExists( self, intfName ):
      isda = self.agent.intfStatusAllDir_
      lisd = self.agent.loopbackIntfStatusDir_
      return ( isda and isda.has_key( intfName ) ) or \
          ( lisd and lisd.has_key( intfName ) )  
         
   def handleSysdbConfiguration( self, cloudSm ):     
      t2( 'handleSysdbConfiguration called' )
      haConfig = self.convertSysdbConfig()
      assert haConfig
      dumpCookedUpConfig( haConfig )
      if not self.validateConfig( haConfig, self.agent.cloudType ):
         t0( 'Bad config passed. Ignoring' )
         # Return generic error unless the backend raises a 
         # more meaningful exception
         raise ConfigInvalid( "Bad Config" )
      # Make sure the config is consistent with dependent non-Cloud configs too.
      srcIntf = haConfig[ 'bfdConfig' ][ 'bfdSourceInterface' ]
      if not self.interfaceExists( srcIntf ):
         t0( 'BFD Source Interface %s doesn\'t exist' % srcIntf )
         self.agent.startIpStatusReactor( srcIntf, cloudSm )
         raise ConfigInvalid( "missing bfd source intf: %s" % str( srcIntf ) )
      srcIp, vrf = getIpAddrVrfFromIntf( self.agent.ipStatus, srcIntf )
      if not srcIp:
         t0( 'BFD Source Interface %s doesn\'t have IP address assigned' % srcIntf )
         self.agent.startIpStatusReactor( srcIntf, cloudSm )
         raise ConfigInvalid( "Ip Address is not assigned to the bfd source" \
            " interface: %s" % str( srcIntf ) )
      else:
         # No need to stop intf status reactor as ip address exists
         # as it is OK to react to ip address changes too.
         # Creation will also delete the old reactor.
         t0( 'BFD source interface IP is found. Restarting the ipStatus SM' ) 
         self.agent.startIpStatusReactor( srcIntf, cloudSm )
      haConfig[ 'bfdConfig' ][ 'bfdSourceIp' ] = srcIp
      haConfig[ 'bfdConfig' ][ 'vrf' ] = vrf
      t0( 'Using BFD source IP address %s vrf %s '% ( srcIp, vrf ) )
      # Update the config 
      self.config = haConfig
      assert  self.config

   # Apply current config for BFD. Make sure stale config is removed too.
   def applyBfdConfig( self ):
      t0( 'Apply bfd config invoked' )
      config = self.config
      agent = self.agent
      peerIp = config[ 'bfdConfig' ][ 'peerVeosIp' ]
      srcIntf = config[ 'bfdConfig' ][ 'bfdSourceInterface' ]
      bfdSessionType = config[ 'bfdConfig' ][ 'bfdSessionType' ]
      srcIp = config[ 'bfdConfig' ][ 'bfdSourceIp' ]
      vrf = config[ 'bfdConfig' ][ 'vrf' ]
      assert peerIp and srcIntf
      peer_ip = Tac.newInstance( 'Arnet::IpGenAddr' , str( peerIp ) )
      src_ip = Tac.newInstance( 'Arnet::IpGenAddr' , srcIp )
      peer = Tac.newInstance( 'Bfd::Peer' , peer_ip, self.agent.vrfName )
      isMultiHop = bool( bfdSessionType == 'multihop' )
      t0( "Adding peer ip %s srcIntf %s src_ip %s vrf %s multihop %s to BFD" % \
          ( peer_ip, srcIntf, src_ip, vrf, isMultiHop ) )
      if isMultiHop:
         peer.srcIp = src_ip 
      peer.type = bfdSessionType
      # It is possible in a race condition case that the interface
      # is deleted after we have validated it earlier statically. But by 
      # the time SDK verification is done in background in few seconds or so
      # interface was actually deleted. Protect ourselves from
      # crash. The net effect will be that we will never progress to 
      # connected state as expected since the interface is gone away. 
      # Moving forward we need to add reactors to status dir and respond if the intf
      # ever comes back up.
      try:
         if not isMultiHop:
            peer.intf = Tac.newInstance( 'Arnet::IntfId', str( srcIntf ) )
      except exceptions.IndexError as e:
         t0( "Error: %s is not found" % str( srcIntf ), e )

      if agent.bfdCloudAppConfig_.peerConfig.get( peer, None ):
         # Nothing to do
         t0( 'peer config already exists..')
      else:
         t0( 'Adding peer config' )
         agent.bfdCloudAppConfig_.newPeerConfig( peer )

      # Handle restart case by stale entries
      # A Note of caution as we cannot call self.cleanup()
      # as that will trigger failover/disruption
      # if restarted. Hence we need to clean only stale 
      # entries.
      for i in agent.bfdCloudAppConfig_.peerConfig:
         if i.ip != peer_ip or i.type != bfdSessionType or \
                ( isMultiHop and i.srcIp != src_ip ) or \
                ( not isMultiHop and i.intf != srcIntf ):
            t0( 'Removing stale BFD config entry for peer ip %s source ip %s ' \
                  'session type  %s srcIntf %s ' % ( i.ip, 
                     i.srcIp, i.type, i.intf ) )
            del agent.bfdCloudAppConfig_.peerConfig[ i ]
 
   # decorator for others to help check configuration
   @classmethod
   def haConfigValidator( cls, func ):  
      BaseCloudConfigHandler.configValidators[ func.__name__ ] = func
      return func

   # This cleans up as CloudHa Agent is exiting. 
   def cleanup( self ):
      # Remove BFD config 
      bfdPeerConfig = self.agent.bfdCloudAppConfig_.peerConfig
      for i in bfdPeerConfig:
         t0( 'Removing BFD config entry for peer ip %s ,src: %s ' % 
               ( i.ip, i.srcIp ) )
         del bfdPeerConfig[ i ]

   def addCloudSpecificConfig( self, haConfig ): 
      # Base class should implement this
      raise exceptions.NotImplementedError

   
class AwsCloudConfigHandler( BaseCloudConfigHandler ):
   def __init__( self, agent ):
      super( AwsCloudConfigHandler, self ).__init__( agent )

   def setupConfigHandler( self, haSm ):
      # Setup our SM to react to configs now
      self.sysdbConfigSm = SysdbAwsConfigSm( self.cloudHaConfig, \
                                 haSm ) 

   def addCloudSpecificConfig( self, haConfig ): 
      self.addAwsConfig( haConfig )
      self.addAwsLocalConfig( haConfig )
      self.addAwsPeerConfig( haConfig )

   def addAwsConfig( self, dstConfig ):
      t0( 'addAwsConfig' )
      sysdbConfig = self.cloudHaConfig
      sysdbAwsAccessConfig = sysdbConfig.accessConfig 
      config = dstConfig[ 'awsConfig' ] = dict()
      config[ 'region' ] = sysdbAwsAccessConfig.region
      cred = config[ 'aws_credentials_optional' ] = dict() 
      cred[ 'aws_access_key_id' ] = ReversibleSecretCli.decodeKey( 
            sysdbAwsAccessConfig.accessKey )
      cred[ 'aws_secret_access_key' ] = ReversibleSecretCli.decodeKey( 
            sysdbAwsAccessConfig.secretAccess )
      proxyName = sysdbAwsAccessConfig.proxyName
      if proxyName:
         proxy = self.proxyConfig.proxy.get( proxyName, None )
         if not proxy:
            t0( 'Proxy not yet configured, bailing out' )
            raise ConfigInvalid( 'Referenced proxy %s is not configured' \
                  % proxyName )
            
         proxyDict = config[ 'http_proxy_optional' ] = dict()
         # AWS supported only one proxy. Encode it for use with 
         # SSL/443 traffic.
         if proxy.httpsProxy:
            proxyDict[ 'http_port_optional' ] = '443'
            proxyDict[ 'http_proxy_optional' ] = str( proxy.httpsProxy )
            proxyDict[ 'http_proxy_port_optional' ] = str( proxy.httpsPort )
            proxyDict[ 'http_proxy_user_optional' ] = str( proxy.httpsUsername )
            proxyDict[ 'http_proxy_password_optional' ] = \
                  ReversibleSecretCli.decodeKey( proxy.httpsPassword ) \
                     if proxy.httpsPassword else ''
         elif proxy.httpProxy:
            # Not sure why this will be ever used and may cause confusion.
            # Adding to take care of any unforseen issues but only place it can 
            # be used( at least what I can think of is ) is for getting IAM role 
            # credentials via a proxy but that would almost always be wrong... 
            # Not sure this is a rope for users to hang them. 
            # Revisit in future releases.
            proxyDict[ 'http_port_optional' ] = '80'
            proxyDict[ 'http_proxy_optional' ] = str( proxy.httpProxy )
            proxyDict[ 'http_proxy_port_optional' ] = str( proxy.httpPort )
            proxyDict[ 'http_proxy_user_optional' ] = str( proxy.httpUsername )
            proxyDict[ 'http_proxy_password_optional' ] = \
                  ReversibleSecretCli.decodeKey( proxy.httpPassword ) \
                     if proxy.httpPassword  else ''

   def addAwsLocalConfig( self, dstConfig ):
      t0( 'addAwsLocalConfig' )
      sysdbConfig = self.cloudHaConfig
      sysdbLocalConfig = sysdbConfig.localRoutes 
      rtConfig = dstConfig[ 'awsLocalRoutingConfig' ] = dict()
      rtIdAndRNIC = rtConfig[ 'routeTableIdAndRouteNetworkInterface' ] = list()
      config = rtIdAndRNIC
      for key, val in sysdbLocalConfig.items():
         rtEntry = dict()
         rtEntry[ 'routeTableId' ] = key.routeTableId 
         rtEntry[ 'destination' ] = key.destination.stringValue
         rtEntry[ 'routeTarget' ] = val.nextHopRouteTarget
         config.append( rtEntry )

   def addAwsPeerConfig( self, dstConfig ):
      t0( 'addAwsPeerConfig' )
      sysdbConfig = self.cloudHaConfig
      sysdbPeerConfig = sysdbConfig.peerRoutes 
      rtConfig = dstConfig[ 'awsPeerRoutingConfig' ] = dict()
      rtIdAndRNIC = rtConfig[ 'routeTableIdAndRouteNetworkInterface' ] = list()
      config = rtIdAndRNIC 
      for key, val in sysdbPeerConfig.items():
         rtEntry = dict()
         rtEntry[ 'routeTableId' ] = key.routeTableId 
         rtEntry[ 'destination' ] = key.destination.stringValue
         rtEntry[ 'routeTarget' ] = val.nextHopRouteTarget
         config.append( rtEntry )

class AzureCloudConfigHandler( BaseCloudConfigHandler ):
   def __init__( self, agent ):
      super( AzureCloudConfigHandler, self ).__init__( agent )

   def setupConfigHandler( self, haSm ):
      # Setup our SM to react to configs now
      self.sysdbConfigSm = SysdbAzureConfigSm( self.cloudHaConfig, \
                                 haSm ) 

   def addCloudSpecificConfig( self, haConfig ): 
      t0(' addCloudspecificConfig' )
      self.addAzureConfig( haConfig )
      self.addAzureLocalConfig( haConfig )
      self.addAzurePeerConfig( haConfig )

   # Add common azure SDK related config
   def addAzureConfig( self, dstConfig ):
      sysdbConfig = self.cloudHaConfig
      dstAccessConfig = dstConfig[ 'azureConfig' ] = dict()
      accessConfig = sysdbConfig.accessConfig
      if accessConfig.sdkcredFileLocation:
         # pylint: disable-msg=E1103
         # localFilename is provided bu Url plugins and confuses pylint.
         dstAccessConfig[ 'azureSdkAuthCredentials' ] = getSdkAuthCredFromFile( 
            Url.parseUrl( accessConfig.sdkcredFileLocation, None ).localFilename() )
      elif accessConfig.adEmail:
         dst = dstAccessConfig[ 'azureActiveDirectoryCredentials' ] = \
               dict()
         # Fill Active Directory Credentials if there
         dst[ 'email' ] = accessConfig.adEmail
         dst[ 'password' ] = ReversibleSecretCli.decodeKey( \
               accessConfig.password )
         dst[ 'subscriptionId' ] = accessConfig.subscriptionId
      else:
         #Fill/Use MSI
         pass
      # Proxy handling.
      proxyName = accessConfig.proxyName
      if proxyName:
         proxy = self.proxyConfig.proxy.get( proxyName, None )
         if not proxy:
            t0( 'Proxy not yet configured, bailing out' )
            raise ConfigInvalid( 'Referenced proxy %s is not configured' %\
                  proxyName )

         if proxy.httpsProxy:
            password = urllib.quote( ReversibleSecretCli.decodeKey( \
                        proxy.httpsPassword ) if proxy.httpsPassword \
                           else '' )
            userPassStr = urllib.quote( proxy.httpsUsername ) + \
                  ( ( ':' + password ) if password else '' )
            httpsConfig = 'https://%s%s:%s' % ( userPassStr + ( '@' if userPassStr 
                                          else '') , proxy.httpsProxy, 
                                    proxy.httpsPort )
            dstAccessConfig[ 'https_proxy_optional' ] = httpsConfig
         if proxy.httpProxy:
            password = urllib.quote( ReversibleSecretCli.decodeKey( \
                           proxy.httpPassword ) if proxy.httpPassword \
                              else '' )
            userPassStr = urllib.quote( proxy.httpUsername ) + \
                  ( ( ':' + password ) if password else '' ) 
            httpConfig = 'http://%s%s:%s' % ( userPassStr + \
                              ( '@' if userPassStr else '' ), proxy.httpProxy,
                           proxy.httpPort )
            dstAccessConfig[ 'http_proxy_optional' ] = httpConfig

   def addAzureCommonRouteConfig( self, dstConfig, local=True ):
      sysdbConfig = self.cloudHaConfig
      if not local:
         cfgType = 'azurePeerRoutingConfig' 
         sysdbRoutes = sysdbConfig.peerRoutes
      else:
         cfgType = 'azureLocalRoutingConfig' 
         sysdbRoutes = sysdbConfig.localRoutes

      dst = dstConfig[ cfgType ] = dict()
      rts = dst[ 'routeTables' ] = list()
      newRtAdded = False
      for i, target in sysdbRoutes.items():
         routeTableEntryDict = None
         for route in rts:
            if route[ 'routeTableName' ] == i.routeTableId:
               routeTableEntryDict = route
               break
         if not routeTableEntryDict:
            routeTableEntryDict = dict()
            routeTableEntryDict[ 'routeTableName' ] = i.routeTableId
            routeTableEntryDict[ 'routes' ] = list()
            newRtAdded = True
         routeTableList = routeTableEntryDict[ 'routes' ]
         # Create new route for this entry
         route = dict()
         route[ 'prefix' ] = str( i.destination )
         route[ 'nextHopIp' ] = str( target.nextHopRouteTarget )
         routeTableList.append( route )
         rg = dst.get( 'resourceGroupName', None )  
         if not rg: 
            dst[ 'resourceGroupName' ] = target.resourceGroup
         elif rg != target.resourceGroup:
            # We currently don't support multiple resource groups for
            # each routes. reject this config.
            raise ConfigInvalid( 'multiple resource groups are not supported for '
                  'primary or backup routes: %s %s' % ( target.resourceGroup , 
                     rg ) )
         if newRtAdded:
            newRtAdded = False
            rts.append( routeTableEntryDict )

   # Add azure local routes ie. translating from sysdb config to python format
   def addAzureLocalConfig( self, dstConfig ):
      self.addAzureCommonRouteConfig( dstConfig )
      
   # Add azure peer routes
   def addAzurePeerConfig( self, dstConfig ):
      self.addAzureCommonRouteConfig( dstConfig, local=False )
      
class GcpCloudConfigHandler( BaseCloudConfigHandler ):
   def __init__( self, agent ):
      super( GcpCloudConfigHandler, self ).__init__( agent )

   def setupConfigHandler( self, haSm ):
      # Setup our SM to react to configs now
      self.sysdbConfigSm = SysdbGcpConfigSm( self.cloudHaConfig, haSm )

   def addCloudSpecificConfig( self, haConfig ):
      self.addGcpConfig( haConfig )
      self.addGcpRouteConfig( haConfig, True )
      self.addGcpRouteConfig( haConfig, False )

   def addGcpConfig( self, dstConfig ):
      t0( 'addGcpConfig' )
      sysdbConfig = self.cloudHaConfig
      sysdbGcpAccessConfig = sysdbConfig.accessConfig
      config = dstConfig[ 'gcpConfig' ] = dict()
      config[ 'project' ] = sysdbGcpAccessConfig.project
      if sysdbGcpAccessConfig.serviceFileLocation:
         config[ 'service_file_optional' ] = Url.parseUrl(
               sysdbGcpAccessConfig.serviceFileLocation, None ).localFilename()

      # Handle proxy
      proxyName = sysdbGcpAccessConfig.proxyName
      if proxyName:
         proxy = self.proxyConfig.proxy.get( proxyName, None )
         if not proxy:
            t0( 'Proxy not yet configured, bailing out' )
            raise ConfigInvalid( 'Referenced proxy %s is not configured' %
                  proxyName )

         if proxy.httpsProxy:
            password = urllib.quote(
                  ReversibleSecretCli.decodeKey( proxy.httpsPassword )
                  if proxy.httpsPassword else '' )
            userPassStr = urllib.quote( proxy.httpsUsername ) + (
                  ( ':' + password ) if password else '' )
            httpsConfig = 'https://%s%s:%s' % (
                  userPassStr + ( '@' if userPassStr else '' ),
                  proxy.httpsProxy, proxy.httpsPort )
            config[ 'https_proxy_optional' ] = httpsConfig
         if proxy.httpProxy:
            password = urllib.quote(
                  ReversibleSecretCli.decodeKey( proxy.httpPassword )
                  if proxy.httpPassword else '' )
            userPassStr = urllib.quote( proxy.httpUsername ) + (
                  ( ':' + password ) if password else '' )
            httpConfig = 'http://%s%s:%s' % (
                  userPassStr + ( '@' if userPassStr else '' ),
                  proxy.httpProxy, proxy.httpPort )
            config[ 'http_proxy_optional' ] = httpConfig

   def addGcpRouteConfig( self, dstConfig, peer=False ):
      sysdbConfig = self.cloudHaConfig
      if peer:
         t0( 'addGcpPeerConfig' )
         sysdbRouteConfig = sysdbConfig.peerRoutes
         rtConfig = dstConfig[ 'gcpPeerRoutingConfig' ] = list()
      else:
         t0( 'addGcpLocalConfig' )
         sysdbRouteConfig = sysdbConfig.localRoutes
         rtConfig = dstConfig[ 'gcpLocalRoutingConfig' ] = list()
      for key in sysdbRouteConfig:
         rtEntry = dict()
         intf = key.interface
         macAddr = ''
         if intf:
            if not self.interfaceExists( intf ):
               t0( 'Interface %s doesn\'t exist' % intf )
               raise ConfigInvalid( "Invalid GCP route intf: %s" % str( intf ) )
            macAddr = self.agent.intfStatusAllDir_.intfStatus[ intf ].burnedInAddr
         rtEntry[ 'macAddr' ] = macAddr
         rtEntry[ 'vpc' ] = key.vpc
         rtEntry[ 'destination' ] = key.destination.stringValue
         rtEntry[ 'tag' ] = key.tag
         rtConfig.append( rtEntry )


#
# Base class to handle config changes.
#
class SysdbBaseConfigSm( Tac.Notifiee ):
   notifierTypeName = "Cloud::HA::BaseConfig"
   def __init__( self, configObj, cloudSm ):
      t2(" Creating sysdb config notifiee: %s " % configObj )
      self.cloudSm = weakref.proxy( cloudSm )
      Tac.Notifiee.__init__( self, configObj )

   @Tac.handler( 'lastChangeTime' )
   def configChange( self ):
      t0(" config change called " )
      self.processChange()

   def processChange( self ):
      t0(" Sending config change event to SM" )
      self.cloudSm.sendConfigChangedEvent() 


class SysdbAwsConfigSm( SysdbBaseConfigSm ):
   def __init__( self, configObj, cloudSm ):
      assert configObj and cloudSm
      super( SysdbAwsConfigSm, self ).__init__( configObj, cloudSm ) 

class SysdbAzureConfigSm( SysdbBaseConfigSm ):
   def __init__( self, configObj, cloudSm ):
      assert configObj and cloudSm
      super( SysdbAzureConfigSm, self ).__init__( configObj, cloudSm ) 

class SysdbGcpConfigSm( SysdbBaseConfigSm ):
   def __init__( self, configObj, cloudSm ):
      assert configObj and cloudSm
      super( SysdbGcpConfigSm, self ).__init__( configObj, cloudSm )

