# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Agent
import Tracing
import CloudBfd
import CloudHaSm
import CloudHaConfig
import CloudHaIntf
from CloudUtil import getHypervisor 
t0 = Tracing.t0
t1 = Tracing.t1
traceSetting = 'Cloud*/0-3,Aws*/*,Bfd*/*,Azure*/*,Gcp*/*'

class CloudHa( Agent.Agent ):
   def __init__( self, entityManager ):
      t0( "CloudHA Agent: Init" )
      self.entityManager = entityManager
      self.vrfName = 'default'
      self.appName = 'cloudHA'
      self.bfdStatusSm_ = None
      self.bfdMonitor_ = None
      self.bfdCloudAppStatus_ = None
      self.bfdConfigAppCloudVrfDir_ = None
      self.cloudConfigHandler = None
      self.cloudStatusHandler = None
      self.bfdCloudAppConfig_ = None
      self.cloudType = getHypervisor()
      self.bfdConfigAppRootDir_ = None
      self.cloudHaSm_ = None
      self.bfdAllPeerStatus_ = None
      self.loopbackIntfStatusDir_ = None
      self.bfdConfigAppCloudDir_ = None
      self.intfStatusAllDir_ = None
      self.ipStatus = None
      self.ipStatusReactor_ = None
      self.haConfig_ = None
      #self.awsHaConfig_ = None
      self.proxyConfig = None
      self.haStatus_ = None
      Agent.Agent.__init__( self, entityManager )

   def getDefaultTrace( self ):
      return traceSetting

   def doInit( self, entManager ):
      t0( "CloudHA Agent: doInit" )
      if not self.cloudType:
         # We should not be running in any physical or unknown 
         # hardware. May be turn this on in future after forgiving
         # KVM, ESXI plaforms.
         #assert False, 
         t0( 'Unsupported platform: Bailing out' )
         return

      mg = self.entityManager.mountGroup()
      self.bfdConfigAppRootDir_ = mg.mount( 'bfd/config/app/cloudHA', 
                                    'Tac::Dir', mode='wci' )
      self.bfdAllPeerStatus_ = mg.mount( 'bfd/status/peer', 
                                    'Bfd::StatusPeer', mode='r' )
      self.loopbackIntfStatusDir_ = mg.mount( 'interface/status/loopback/intf', 
                                    'Interface::LoopbackIntfStatusDir', mode='r' )
      self.ipStatus = mg.mount( 'ip/status', 'Ip::Status', mode='r' )

      if self.cloudType == "AWS":
         self.haConfig_ = mg.mount( 'cloudha/awsconfig', 
               'Cloud::HA::AwsHaConfig', mode='r' )
         self.haStatus_ = mg.mount( 'cloudha/awsstatus', 
               'Cloud::HA::AwsStatus', mode='w' )
      elif self.cloudType == "Azure": 
         self.haConfig_ = mg.mount( 'cloudha/azureconfig', 
               'Cloud::HA::AzureHaConfig', mode='r' )
         self.haStatus_ = mg.mount( 'cloudha/azurestatus', 
               'Cloud::HA::AzureStatus', mode='w' )
      elif self.cloudType == "GCP":
         self.haConfig_ = mg.mount( 'cloudha/gcpconfig',
               'Cloud::HA::GcpHaConfig', mode='r' )
         self.haStatus_ = mg.mount( 'cloudha/gcpstatus',
               'Cloud::HA::GcpStatus', mode='w' )
      else:
         assert 0, 'Unsupported platform'

      self.proxyConfig = mg.mount( 'cloudha/proxyConfig', 
         'Cloud::HA::ProxyConfig', 'r' )

      self.intfStatusAllDir_ = self.createLocalEntity( "AllIntfStatusDir",
                                                       "Interface::AllIntfStatusDir",
                                                       "interface/status/all" )
      self.localEntitiesCreatedIs( True )

      def _finish():
         t0( 'mounts finished' )
         self.bfdConfigAppCloudVrfDir_ = self.bfdConfigAppRootDir_.newEntity( 
                                       'Tac::Dir', self.vrfName )
         self.bfdCloudAppConfig_ = self.bfdConfigAppCloudVrfDir_.newEntity( 
                                    'Bfd::AppConfig', self.appName )
         self.bfdCloudAppStatus_ = Tac.newInstance( "Bfd::AppStatus", self.appName )

         # Handle initial config
         if self.cloudType == 'AWS':
            self.cloudConfigHandler = CloudHaConfig.AwsCloudConfigHandler( self )
            self.cloudStatusHandler = CloudHaConfig.AwsCloudStatusHandler( self )
         elif self.cloudType == 'Azure':
            self.cloudConfigHandler = CloudHaConfig.AzureCloudConfigHandler( self )
            self.cloudStatusHandler = CloudHaConfig.AzureCloudStatusHandler( self )
         elif self.cloudType == 'GCP':
            self.cloudConfigHandler = CloudHaConfig.GcpCloudConfigHandler( self )
            self.cloudStatusHandler = CloudHaConfig.GcpCloudStatusHandler( self )
         else:
            assert 0, 'Unsupported platform'
            
         self.cloudHaSm_ = CloudHaSm.CloudHaStateMachine( self, self.cloudType )
         self.cloudConfigHandler.setupConfigHandler( self.cloudHaSm_ )
         
         self.bfdStatusSm_ = Tac.newInstance( 
                                    "BfdStatAgentLib::PeerStatusManager",
                                    self.appName, 
                                    self.bfdAllPeerStatus_, 
                                    self.bfdCloudAppStatus_, 
                                    self.vrfName,
                                    self.bfdCloudAppConfig_ )
         self.bfdMonitor_ = CloudBfd.BfdAppStatusMonitor( 
                                 self.bfdCloudAppStatus_, 
                                 self.cloudHaSm_ )
      mg.close( _finish )

   # CloudSm is needed as it may still be instantiating and not 
   # yet accessible via agent.
   def startIpStatusReactor( self, srcIntf, cloudSm ):
      assert cloudSm
      if self.ipStatusReactor_ and self.ipStatusReactor_.srcIntf == srcIntf:
         t0( 'Skipping creation ipStatusReactor as already exists for src intf: %s' \
            % ( srcIntf ) )
         return
      self.ipStatusReactor_ = CloudHaIntf.IpStatusReactor(
         self.ipStatus,
         cloudSm,
         srcIntf )

   def stopIpStatusReactor( self ):
      self.ipStatusReactor_ = None

def createBfdStatusSm( agent, cloudHaSm ):
   t0( 'creating Bfd app status SM %s', agent )
   # We need to explictly pass cloudHaSM as it is normally initialized in the 
   # agent itself in finish mount callback but under test environment, SM may 
   # call recreateBfdStatus synchronously in instantiation itself from 
   # finish mounts. 
   agent.bfdMonitor_ = CloudBfd.BfdAppStatusMonitor( 
                                 agent.bfdCloudAppStatus_, 
                                 cloudHaSm )


