# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# This is needed to establish explicit 
# dependency on BFD package.
# pylint: disable-msg=W0611
import BfdAgent
import Tac
import Tracing
import CloudHaSm
import CloudHaConfig 

t0 = Tracing.t0
t1 = Tracing.t1


# This uses BFD based session with the peer veos instance 

class BfdPeerStateMonitor( Tac.Notifiee ):
   notifierTypeName = "Bfd::BfdPeerStatus"
   def __init__( self, bfdPeerStatus, ha_handler ):
      t0( "bfdPeerStateMonitor: Init" )
      self.bfdPeerStatus = bfdPeerStatus
      self.peerState = self.bfdPeerStatus.peerState
      self.prevPeerState = None
      self.prevStatus = None
      self.status = self.bfdPeerStatus.status
      self.ha_handler = ha_handler
      # This derived var simply indicates whether our BFD session is in up 
      # or down state
      self.sessionUp = False
      Tac.Notifiee.__init__( self, self.bfdPeerStatus )
      self.appStatusHandler()

   @Tac.handler( 'status' )
   @Tac.handler( 'peerState' )
   def appStatusHandler( self ):
      self.prevStatus = self.status
      self.status = self.bfdPeerStatus.status
      self.prevPeerState = self.peerState
      self.peerState = self.bfdPeerStatus.peerState
      t0( 'BFD Statust PeerState: %s ->  %s, local State %s -> %s ' %\
           ( self.prevPeerState, self.peerState, self.prevStatus, self.status ) )
      # if the other peer or local BFD state is brought down due to config etc
      # Go back to init state as against failover
      if( self.sessionUp and ( self.peerState == 'adminDown' or \
               self.status == 'adminDown' ) ):
         t0( 'BFD session is admin down' ) 
         self.sessionUp = False
         self.ha_handler.sendEvent( CloudHaSm.PeerAdminDown() )
         return
      handledStates = [ 'up' , 'down' ]
      if not ( self.peerState in handledStates and \
               self.status in handledStates ):
         t0( 'Ignoring event, peerState: %s local State: %s' % \
            ( self.peerState, self.status ) )
         return
      down = self.peerState == 'down' or self.status == 'down'
      self.sessionUp = not down
      self.ha_handler.sendEvent( CloudHaSm.PeerDown() if down \
         else CloudHaSm.PeerUp() )

class BfdAppStatusMonitor( Tac.Notifiee ):
   notifierTypeName = "Bfd::AppStatus"
   def __init__( self, bfdAppStatus, ha_handler ):
      t0( 'BfdAppStatusMonitor __init__' ) 
      self.bfdAppStatus = bfdAppStatus
      self.peerStateMonitors = {}
      self.ha_handler = ha_handler
      assert ha_handler
      # Handle existing peerStatus members
      # We dont' expect more than one peer at any time
      # per current design.
      assert len( self.bfdAppStatus.peerStatus ) <= 1
      for i in self.bfdAppStatus.peerStatus:
         t0( ' Init peerstatus sm for %s' % i )
         self.appStatusHandler( i )
      Tac.Notifiee.__init__( self, self.bfdAppStatus )
      
   @Tac.handler( 'peerStatus' )
   def appStatusHandler( self, peer ):
      t0( 'Received BFD Notification for peer %s' % peer ) 
      peerStatus = self.bfdAppStatus.peerStatus.get( peer, None )
      peerSm = self.peerStateMonitors.get( peer, None )
      if peerStatus:
         t0( 'Creating peer monitor SM for peer' ) 
         assert not peerSm, "found stale peerSm"
         self.peerStateMonitors[ peer ] = BfdPeerStateMonitor(
            peerStatus, self.ha_handler )
      else:
         if peerSm:
            t0( 'Deleting peer monitor SM' )
            del self.peerStateMonitors[ peer ]
