# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import re

import Tac
from ArnetLib import U16_MAX_VALUE
from CliCommand import CliExpression, Node
from CliParserCommon import Completion, noMatch, MatchResult
from CliParser import safeInt
from CliMatcher import (
      Matcher,
      EnumMatcher,
      IntegerMatcher,
      KeywordMatcher,
      FloatMatcher
   )

#-------------------------------------------------------------------------
# Common Tokens
# In the future tokens in RouteMapCli.CommonTokens can be merged here.
#-------------------------------------------------------------------------
class CommonTokens( object ):
   '''
   This class is intended to keep common tokens that may be shared across several
   commands with the new CLI parser.
   '''
   additive = KeywordMatcher( 'additive',
                              helpdesc='Add communities to those already present' )
   delete = KeywordMatcher( 'delete', helpdesc='Delete matching communities' )
   # To be used by protocols at route-map application points.
   routeMapApplication = KeywordMatcher( 'route-map',
                                         helpdesc='Specify which route map to use' )
   nextHop = KeywordMatcher( 'next-hop', helpdesc='Route Next-hop' )


#-------------------------------------------------------------------------
# Community matching
#-------------------------------------------------------------------------
COMMUNITY_BASE10_MAX = 4294967040

# shared by both CliPlugin and CliSavePlugins
statementConfigToken = 'statement'

class CommunityAannMatcher( Matcher ):
   """Matches community in aa:nn notation"""
   def __init__( self, helpdesc, **kargs ):
      super( CommunityAannMatcher, self ).__init__( helpdesc=helpdesc, **kargs )
      self.completion_ = Completion( 'aa:nn', helpdesc, False )
      self.commFormatRe_ = re.compile( r'^(\d+):(\d+)$' )
      self.commFormatCompletionRe_ = re.compile(
         r'^(\d+)?:?(\d+)?$' )

   def match( self, mode, context, token ):
      m = self.commFormatRe_.match( token )
      if m is None:
         return noMatch
      for group in m.groups():
         if group is not None and ":" not in group and int( group ) > U16_MAX_VALUE:
            return noMatch
      return MatchResult( token, token )

   def completions( self, mode, context, token ):
      m = self.commFormatCompletionRe_.match( token )
      if m is None:
         return []
      for group in m.groups( '0' ):
         n = safeInt( group )
         if n is None or n > U16_MAX_VALUE:
            return []
      return [ self.completion_ ]

   def __str__( self ):
      return '<aa:nn>'

communityEnumMatcher = EnumMatcher( {
   'GSHUT': 'Graceful Shutdown (well-known community)',
   'internet': 'Internet (well-known community)',
   'local-as': 'Do not send outside local AS',
   'no-advertise': 'Do not advertise to any peer',
   'no-export': 'Do not export to next AS',
   } )

class CommunityConstExpr( CliExpression ):
   expression = "AANN_FORMAT | COMMUNITY_ENUM | COMMUNITY_U32"
   data = {
         "AANN_FORMAT": Node(
            matcher=CommunityAannMatcher( "Community number" ),
            alias="communityValue" ),
         "COMMUNITY_ENUM": Node(
            matcher=communityEnumMatcher,
            alias="communityValue" ),
         "COMMUNITY_U32": Node(
            matcher=IntegerMatcher( 1, COMMUNITY_BASE10_MAX,
               helpdesc='Community number' ),
            alias="communityValue" ),
      }

#-------------------------------------------------------------------------
# Extended Community Expressions
#-------------------------------------------------------------------------
class ExtCommActionExpression( CliExpression ):
   expression = '( additive | delete )'
   data = {
      'additive': CommonTokens.additive,
      'delete': CommonTokens.delete
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      action = args.get( 'additive' )
      if not action:
         action = args.get( 'delete' )
      if action:
         args[ 'ACTION' ] = [ action ]

#-------------------------------------------------------------------------
# Set Metric Matchers
#-------------------------------------------------------------------------
class MetricModifierAndNumberMatcher( Matcher ):
   def __init__( self, modifier, matcher, helpdesc=None, helpname=None ):
      self.modifier_ = modifier
      self.matcher_ = matcher
      Matcher.__init__( self, helpdesc=helpdesc, helpname=helpname )

   def match( self, mode, context, token ):
      if token.startswith( self.modifier_ ):
         numberMatch = self.matcher_.match( mode, context,
                                            token[ len( self.modifier_ ) : ] )
         if numberMatch is not noMatch:
            return MatchResult( numberMatch.result, str( numberMatch.result ) )

      return noMatch

   def completions( self, mode, context, token ):
      if ( token == '' or token == self.modifier_ or
           self.match( mode, context, token ) is not noMatch ):
         return [ Completion( self.helpname_, self.helpdesc_, False ) ]
      return []

class InputRestrictedFloatMatcher( FloatMatcher ):
   # This matcher extends FloatMatcher in order to prevent user input of
   # a float with more than N number of decimal places of precision.
   def __init__( self, lbound, ubound, precisionString, maxPrecision ):
      self.maxPrecision_ = maxPrecision
      FloatMatcher.__init__( self, lbound=lbound, ubound=ubound,
                             precisionString=precisionString )

   def precisionOutOfBounds( self, strFloat ):
      decimalPointIndex = strFloat.find( '.' )
      if ( decimalPointIndex != -1 and
           len( strFloat[ decimalPointIndex : ] ) > self.maxPrecision_ + 1 ):
         return True
      return False

   def match( self, mode, context, token ):
      if self.precisionOutOfBounds( token ):
         mode.addErrorAndStop( 'Precision of floating point value must be {} '
                               'decimal places or less'.format(
                                  self.maxPrecision_ ) )
         return noMatch
      return FloatMatcher.match( self, mode, context, token )

class MetricModifierAndFloatMatcher( MetricModifierAndNumberMatcher ):
   def completions( self, mode, context, token ):
      if ( token == '' or token == self.modifier_ or
           token == self.modifier_ + '.' or
           self.match( mode, context, token ) is not noMatch ):
         return [ Completion( self.helpname_, self.helpdesc_, False ) ]
      return []
