#!/usr/bin/env python
# Copyright (c) 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tracing
import curses.ascii
import re

__defaultTraceHandle__ = Tracing.Handle( 'CliTest' )

t0 = Tracing.trace0
t1 = Tracing.trace1
t2 = Tracing.trace2
t3 = Tracing.trace3
t4 = Tracing.trace4
t9 = Tracing.trace9

# Regular expression for matching hostnames.  According to RFC1912, "Allowable
# characters in a label for a host name are only ASCII letters, digits, and the
# `-' character".
# We require that hostnames don't start with the `-', and be at least 2
# characters, to prevent the pdb backtrace "->" from looking like a prompt.
hostnameRe = '(?:[a-zA-Z0-9][a-zA-Z0-9-_]+)'    # ie "localhost" or "deer"
# ie "deer.arastra.com"
# Hostnames must not start with "(diag", "bash", or "Aboot".
# If they do, it becomes tricky to tell what mode we are in.
fullHostnameRe = '(?!\(diag)(?!-?bash)(?!Aboot)(?:[a-zA-Z0-9][.a-zA-Z0-9-_]+)'
redsupRe = '(|\(s\d(|-\w+)\))'
vrfRe = '(\(vrf:[a-zA-Z0-9-]+\))?'  # optional vrf name in prompt
timeRe = '(?:[0-9:]+)?'      # optional time in prompt
secureMonitorRe = '(\(secure\))?' # secure-monitor user indicator
newlineRe = '(?:\n)'  # Simple, because of CliTest's VT100 emulation.
moreRe = '(?: \-\-More\-\- )'
# Pagination on the show command is piped through less, which will generate
# this escape sequence at the end of each page of output.
moreReCompiled = re.compile( moreRe )

# The password prompt can be different depending on which PAM module was
# invoked to generate it.
passwdPromptRe = '[Pp]assword: '

TIMEOUT = 600
# The timeout above might be very long, but when you are running on a
# Celeron with two virtual machines and ssh'ing in and starting the
# Cli for the first time in a pre-Cli-daemon world, or maybe you are
# booting the VMs and Aboot is uncompressing filesystem images, things
# can be slow.

DEFAULT_ROOT_PASSWORD = 'arastra'

class NoParamsSpecified( object ):
   pass

class Mode( object ):
   allModes_ = []
   allPrompts = []
   modeMap_ = {}
   needFindPath_ = False
   def __init__( self, name, prompt, parent, cmdFromParent,
                 sessionCmdFromParent=None,
                 challengeStr=None, responseStr=None,
                 promptContainsHostname=True, sync=True,
                 cmdToParent='exit',
                 runsBash=False,
                 checkReturnCodeCmd=None,
                 prettyMode=None,
                 fragile=False,
                 ambiguousPrompt=False,
                 groupChange=False,
                 testParams=NoParamsSpecified,
                 testCmds=None,
                 testParentParams=None,
                 rangeLen=1,
                 cohabPrompt=None,
                 singleInstance=False,
                 noNewline=False,
                 modeChangeFailure=None,
                 skipIfEmpty=None,
                 cleanupCmd=None,
                 enabled=True,
                 osType=None,
                 challengeCb=None ):
      """'ambiguousPrompt' should be True if we cannot determine what mode
      we're in purely by looking at the prompt."""
      assert not ambiguousPrompt or ( cmdToParent and parent ), \
            "Modes with an ambiguous prompt should have a parent and a way to " \
            "get there."
      self.name = name
      # This callback returns response string for challenge other than password,
      # default value assigned is none
      self.challengeCb = challengeCb
      prompt = prompt.replace( r'\(config', r'\(config(?:-as|-s-[^)]+)?' )
      self.cohabPrompt = cohabPrompt if cohabPrompt else prompt
      if name == 'login':
         # The hostname is optional - When you first try to login,
         # the hostname is printed before the login prompt. However,
         # if you try to login and fail, you then get into a retry loop
         # where you are prompted to login again. However, in this loop,
         # the login prompt is not preceeded with the hostname
         self.prompt_ = '(?!Last )(%s )?' % (fullHostnameRe) + prompt
      elif promptContainsHostname:
         # The default prompt is %H%R%p, which may include "." characters.
         # System Test / TAC duts add the time.
         self.prompt_ = ( secureMonitorRe + fullHostnameRe + redsupRe + vrfRe +
                          timeRe + prompt )
      else:
         self.prompt_ = prompt
      self.compiledPrompt_ = None
      self.parent = parent
      self.cmdFromParent = cmdFromParent
      if self.cmdFromParent:
         assert type( self.cmdFromParent ) is str, "cmdFromParent must be a string"
      # cmd used to enter mode when in config session
      self.sessionCmdFromParent = sessionCmdFromParent
      self.toMode = {}
      self.toModeCmd = {}
      self.challengeStr = challengeStr
      self.responseStr = responseStr
      self.sync = sync
      self.cmdToParent = cmdToParent
      if self.cmdToParent:
         assert type( self.cmdToParent ) is str, "cmdToParent must be a string"
      self.checkReturnCodeCmd = checkReturnCodeCmd
      self.prettyMode = prettyMode
      self.runsBash = runsBash
      if runsBash:
         # It's redundant to specify runsBash and a checkReturnCodeCmd or prettyMode
         assert not checkReturnCodeCmd
         assert not prettyMode
         self.checkReturnCodeCmd = "echo $?"
         self.prettyMode = "Bash"
      self.fragile = fragile
      self.ambiguousPrompt = ambiguousPrompt
      self.singleInstance = singleInstance
      self.noNewline = noNewline
      self.modeChangeFailure = modeChangeFailure

      # Whether this mode is group-change, where the config will be saved only
      # when exiting this mode.
      self.groupChange = groupChange

      # Enumerable parameters to enter this mode when testing.
      # These parameters are used in auto-generated test cases for each config-mode.
      # If none, the auto-test will skip this mode.
      # If [], it means that this mode can be entered without additional parameter
      # Note: The mode will still be used in the Umbrella/NoOrDefaultCliTest.py
      #       stest.  To avoid using it in any test, use "enabled=False".
      assert testParams is not NoParamsSpecified, \
            "You must explicitly supply testParams"
      self.testParams = testParams
      if skipIfEmpty is None:
         # by default, singleton modes should have skipIfEmpty=True
         skipIfEmpty = not testParams
      self.skipIfEmpty = skipIfEmpty
      self.cleanupCmd_ = cleanupCmd

      # For auto-generated test cases, if enabled is false, the auto-tests will
      # skip this mode completely.
      self.enabled_ = enabled

      # If this mode only applies to a certain product, pass the osType.
      # If specified, this mode will only be used by test clients from
      # the same operating system.
      self.osType_ = osType

      # All commands to execute when we run the automatic show active
      # tests. Then the test checks whether the commands do appear in
      # the output of 'show active'.
      self.testCmds = testCmds

      # Enumerable parameters to enter the parent before entering this
      # mode. If testParentParams is None, it means that this child
      # mode can support all its parent's params, so we will use the
      # parent.testParams. Otherwise, it means that this child mode
      # can only support a subset of its parent's params. These
      # parameters are used in auto-generated test cases for each
      # config-mode. If [], it means that this mode can be entered
      # without additional parameter
      if testParentParams is None and self.parent is not None:
         self.testParentParams = self.parent.testParams
      else:
         self.testParentParams = testParentParams

      # Whether this is a range config mode.
      # If it is, rangeLen > 1, and tells the number of individual modes that
      # the parameters in testParams will cover. For example, in configIfRangeMode's
      # testParams, it is 'test1-9', so the rangeLen is 9 correspondingly.
      # If it is not, rangeLen is 1 by default.
      self.rangeLen = rangeLen

      # findPath() may not work if 2 instances with the same name are
      # in allModes_.  assert rather than remove old version first, because this
      # is probably unintentional, and better to catch this now then let it 
      # fail silently later.
      msg = "Attempting to redefine mode %s" % self.name
      assert not Mode.modeMap_.has_key( self.name ), msg
      if enabled:
         Mode.allModes_.append( self )
         Mode.allPrompts.append( self.prompt )
         Mode.modeMap_[ self.name ] = self
      # Mark that we need to recalculate paths between modes.
      # Needed if an external module adds modes beyond the ones
      # we know about here.
      Mode.needFindPath_ = True

   def __str__( self ):
      return "<CliTestMode '%s'>" % self.name

   def __repr__( self ):
      return "<{} object at 0x{:x} '{}'>".format(
            self.__class__.__name__, id( self ), self.name )

   @staticmethod
   def _maybeFindPath():
      if Mode.needFindPath_:
         t9( "Recalculating mode paths" )
         findPath()
         Mode.needFindPath_ = False
   @staticmethod
   def resetAllModes():
      Mode.allModes_ = []
      Mode.allPrompts = []
      Mode.modeMap_ = {}
      Mode.needFindPath_ = False

   # Precompile the prompt and newline.
   def _compilePrompt( self ):
      fullPrompt = ''
      if not self.noNewline:
         fullPrompt += newlineRe
      fullPrompt += self.prompt_
      # using re.DOTALL copied from pexpect.compile_pattern_list()
      self.compiledPrompt_ = re.compile( fullPrompt, re.DOTALL )

   @property
   def compiledPrompt( self ):
      if self.compiledPrompt_ is None:
         self._compilePrompt()
         assert self.compiledPrompt_ is not None
      return self.compiledPrompt_

   # To auto-compile the prompt regexp when it changes, prompt becomes
   # a property accessing the local attribute prompt_
   @property
   def prompt( self ):
      assert self.enabled_
      return self.prompt_
   @prompt.setter
   def prompt( self, prompt ):
      self.prompt_ = prompt
      # Force the prompt to be re-compiled next time self.compiledPrompt is accessed.
      self.compiledPrompt_ = None

   def cleanupCmd( self, param ):
      cmd = self.cleanupCmd_ or 'default ' + self.cmdFromParent
      return cmd if param is None else cmd % param

   @property
   def enabled( self ):
      return self.enabled_

   @property
   def osType( self ):
      return self.osType_

# Warning: please try not to add more modes in this file! It should
# reside in the CliTestMode directory of your package. Many of the following
# modes shouldn't be here either, and they will be cleaned up.

# This won't always catch falling into the debugger, because the
# backtrace may contain things that look like prompts for other
# modes.  See resync() for another attempt at catching this.
pdbMode = Mode( name='pdb', prompt='\(Pdb\) ',
                parent=None, cmdFromParent=None,
                sync=False, promptContainsHostname=False,
                testParams=None )

abootPromptMode = Mode(
   name='abootPrompt',
   prompt='Press Control-C now to enter Aboot shell',
   parent=None, cmdFromParent=None,
   sync=False,
   promptContainsHostname=False,
   testParams=None
   )

abootMode = Mode(
   name='aboot',
   prompt='Aboot#\s?',
   parent=abootPromptMode, cmdFromParent=curses.ascii.ctrl( 'c' ),
   fragile=True, # because of timing, hitting Ctrl-C from
                 # abootPromptMode does not always work.
   promptContainsHostname=False,
   runsBash=True,
   testParams=None
   )

loginMode = Mode( name="login", prompt="login: ",
                  parent=abootMode, cmdFromParent="exit", sync=False,
                  cmdToParent=None, testParams=None,
                  modeChangeFailure='Login incorrect' )

unprivModePrompt = '>'
unprivMode = Mode( name='unpriv', prompt=unprivModePrompt,
                   parent=loginMode,
                   cmdFromParent='admin', challengeStr=passwdPromptRe,
                   cmdToParent='exit', testParams=None )

# Hostnames must not start with "(diag", "bash", or "Aboot".
# If they do, it becomes tricky to tell what mode we are in.
enableModePrompt = '#'
enableMode = Mode( name='enable',
                   prompt=enableModePrompt,
                   parent=unprivMode,
                   cmdFromParent='enable', challengeStr=passwdPromptRe,
                   cmdToParent='disable', testParams=None
                   )
# The (?!<) in the regexp below is key --- it prevents the prompt from
# matching when the input buffer is scrolling (read-line-style scrolling with
# a "<" in the left column to indicate that characters "scrolled off" the left
# edge of the screen.)  It turns out that this style of scrolling is used
# only when the prompt is a large fraction of the screen width.  -kduda 2009-07-14
bashMode = Mode( name='bash',
                 prompt=( '([-]?bash[ \S]*|(?!<)([ \S]*@[\w\.\-]+'
                          '( \S[ \S]*)?))[>$%] ?' ),
                 parent=enableMode, cmdFromParent='bash',
                 cmdToParent='exit;exit', # double exit needed if sth running in b/g
                 promptContainsHostname=False, runsBash=True,
                 testParams=None )
bashSuPrompt = 'bashsu# '
bashSuPromptRe = r'%s\s?' % bashSuPrompt.strip()
bashSuMode = Mode( name='bashSu', prompt=bashSuPromptRe,
                   parent=bashMode, cmdFromParent="PS1='%s' sudo su" % bashSuPrompt,
                   # clear previous errors (BUG33681)
                   # and double exit to get out if something is running in background
                   cmdToParent='exit 0;exit 0',
                   promptContainsHostname=False, runsBash=True, testParams=None )

# This pseudo-mode allows discoverMode() to discover our current
# mode when we're in a configuration mode that's not defined here,
# e.g., config-router-ospf.  If we're calling discoverMode(),
# and we're in any config mode, we'll go via enableMode to any other
# config mode.  We can't get into this mode, since it's not real.
unknownConfigMode = Mode( name='unknown-config', prompt='\(config[^)]*\)#',
                          parent=enableMode, cmdFromParent=None,
                          cmdToParent='end', testParams=None )

# Now the real configuration modes.
configMode = Mode( name='config', prompt='\(config\)#',
                   parent=enableMode, cmdFromParent='configure terminal',
                   sessionCmdFromParent='configure session %s',
                   testParams=None )

configRouterGeneralMode = Mode( name='router general',
                                prompt='\(config-router-general\)#',
                                parent=configMode, cmdFromParent='router general',
                                testParams=None )

configIfMode = Mode( name='config-if',
                     prompt='\(config-if-(?P<modeParam>[^\)]*)\)#',
                     parent=configMode, cmdFromParent='interface %s',
                     testParams=[ 'eth1', 'lo1', 'ma1', 'po1', 
                                  'test1', 'vlan1', 'vxlan1' ] )

configRoleMode = Mode( name='config-role',
                           prompt='\(config-role[^\)]*\)#',
                           parent=configMode,
                           cmdFromParent='role %s',
                           groupChange=True,
                           testParams=None )

configVlanMode = Mode( name='config-vlan', prompt='\(config-vlan[^\)]*\)#',
                       parent=configMode, cmdFromParent='vlan %s',
                       testParams=[ '1', '2' ],
                       skipIfEmpty=lambda param : param == '1' )
configMstMode = Mode( name='config-mst', prompt='\(config-mst\)#',
                      parent=configMode,
                      cmdFromParent='spanning-tree mst configuration',
                      testParams=[],
                      # We set this to False so we do not test "skipIfEmpty"
                      # in automated test, since mst has its own implementation
                      # of 'show active'
                      skipIfEmpty=False )

configMgmtMode = Mode( name='config-mgmt', prompt='\(config-mgmt[^\)]*\)#',
                       parent=configMode, cmdFromParent='management %s',
                       testParams=[ 'console', 'ssh', 'telnet', 'xmpp', 
                                    'accounts' ], skipIfEmpty=True )
configAclMode = Mode( name='config-acl', prompt='\(config-acl[^\)]*\)#',
                      parent=configMode, cmdFromParent='ip access-list %s',
                      groupChange=True, testParams=[ 'test-ip-acl',
                                                     'test-ip-acl2' ] )
configIp6AclMode = Mode( name='config-ipv6-acl', prompt='\(config-ipv6-acl[^\)]*\)#',
                      parent=configMode, cmdFromParent='ipv6 access-list %s',
                      groupChange=True, testParams=[ 'test-ipv6-acl',
                                                     'test-ip6-acl2' ] )
configMacAclMode = Mode( name='config-mac-acl', 
                         prompt='\(config-mac-acl[^\)]*\)#',
                         parent=configMode, cmdFromParent='mac access-list %s',
                         groupChange=True, testParams=[ 'test-mac-acl',
                                                        'test-mac-acl2' ] )
configStdAclMode = Mode( name='config-std-acl',
                         prompt='\(config-std-acl[^\)]*\)#',
                         parent=configMode,
                         cmdFromParent='ip access-list standard %s',
                         groupChange=True, testParams=[ 'test-standard-acl',
                                                        'test-standard-acl2' ] )
configStdIp6AclMode = Mode( name='config-std-ipv6-acl',
                         prompt='\(config-std-ipv6-acl[^\)]*\)#',
                         parent=configMode,
                         cmdFromParent='ipv6 access-list standard %s',
                         groupChange=True, testParams=[ 'test-standard-ipv6-acl',
                                                        'test-standard-ipv6-acl2' ] )
configDynAclMode = Mode( name='config-dyn-acl', prompt='\(config-dyn-acl[^\)]*\)#',
                         parent=configMode,
                         cmdFromParent='ip access-list %s dynamic',
                         groupChange=True, testParams=None )
configDynIp6AclMode = Mode( name='config-dyn-ipv6-acl',
                            prompt='\(config-dyn-ipv6-acl[^\)]*\)#',
                            parent=configMode,
                            cmdFromParent='ipv6 access-list %s dynamic',
                            groupChange=True, testParams=None )
configDynMacAclMode = Mode( name='config-dyn-mac-acl',
                            prompt='\(config-dyn-mac-acl[^\)]*\)#',
                            parent=configMode,
                            cmdFromParent='mac access-list %s dynamic',
                            groupChange=True, testParams=None )
configDynStdAclMode = Mode( name='config-dyn-std-acl',
                            prompt='\(config-dyn-std-acl[^\)]*\)#',
                            parent=configMode,
                            cmdFromParent='ip access-list standard %s dynamic',
                            groupChange=True, testParams=None )
configDynStdIp6AclMode = Mode( name='config-dyn-std-ipv6-acl',
                               prompt='\(config-dyn-std-ipv6-acl[^\)]*\)#',
                               parent=configMode,
                               cmdFromParent='ipv6 access-list standard %s dynamic',
                               groupChange=True, testParams=None )
configCpMode = Mode( name='config-system-cp', prompt='\(config-system-cp\)#',
                           parent=configMode, cmdFromParent='system control-plane',
                           testParams=[],
                           # required since the copp config always appears
                           skipIfEmpty = False )

configStrataMmuQueueMode = Mode( name='config-queue',
   prompt='\(config-queue[^\)]*\)#', parent=configMode, 
   cmdFromParent='platform trident mmu queue profile %s', groupChange=True,
   testParams=[ 'test-mmu-queue', 'test-mmu-queue2' ] )

# We can't tell if we're in rootLoginMode or bashSuMode because people don't
# set PS1 when entering bashSuMode manually.
rootLoginPrompt = '([-]?bash[ \S]*|(?!<)([ \S]*@[\w\.\-]+( \S[ \S]*)?))[#>] '
rootLoginMode = Mode( name='rootLogin',
                      prompt=rootLoginPrompt,
                      parent=loginMode, cmdFromParent='root',
                      challengeStr=passwdPromptRe,
                      responseStr=DEFAULT_ROOT_PASSWORD,
                      promptContainsHostname=False,
                      cmdToParent='exit;exit',
                      runsBash=True, ambiguousPrompt=True,
                      testParams=None )
configMlagMode = Mode( name='config-mlag',
                       prompt='\(config-mlag\)#',
                       parent=configMode,
                       cmdFromParent='mlag',
                       testParams=[] )

configRedundancyMode = Mode( name   ='redundancy',
                             prompt='\(config-redundancy\)#',
                             parent=configMode,
                             cmdFromParent='redundancy',
                             testParams=None )
# Since right now redundancy mode is still in progress, the CliSave plugin
# does not save configurations for redundancy mode, so we skip testing this mode.
consoleDownMode = Mode( name='consoleDown',
                        prompt='\[down -- use \^\^ \^\^ \? for help\]',
                        parent=None,
                        cmdFromParent=None,
                        promptContainsHostname=False,
                        testParams=None)
consoleReadOnlyMode = Mode( name='consoleReadOnly',
                        prompt='\[read-only -- use \^\^ \^\^ \? for help\]',
                        parent=None,
                        cmdFromParent=None,
                        promptContainsHostname=False,
                        testParams=None,
                        noNewline=True )

configRouteMapMode = Mode( name='config-route-map',
                           prompt='\(config-route-map[^\)]*\)#',
                           parent=configMode,
                           cmdFromParent='route-map %s',
                           groupChange=True,
                           testParams=[ 'test-test', 'test permit 20' ] )

configIpPrefixMode = Mode( name='config-ip-pfx',
                           prompt='\(config-ip-pfx\)#',
                           parent=configMode,
                           cmdFromParent='ip prefix-list %s',
                           groupChange=True,
                           testParams=[ 'test-test', 'test2' ] )

configIpv6PrefixMode = Mode( name='config-ipv6-pfx',
                           prompt='\(config-ipv6-pfx\)#',
                           parent=configMode,
                           cmdFromParent='ipv6 prefix-list %s',
                           groupChange=True,
                           testParams=[ 'test-test', 'test2' ] )

configDgMode = Mode( name='config-dg', 
                     prompt='\(config-dg-[^\)]*\)#',
                     parent=configMode,
                     cmdFromParent='ip decap-group %s',
                     testParams=[ 'foo', 'bar' ] )

pythonMode = Mode( name="python",
                   prompt=">" ">> ", # defeat the conflict-marker check
                   parent=enableMode, cmdFromParent="python",
                   cmdToParent=curses.ascii.ctrl( 'd' ),
                   sync=False,
                   promptContainsHostname=False, runsBash=False,
                   testParams=None )

aconsMode = Mode( name='acons',
                  prompt='\$ ',
                  promptContainsHostname=False,
                  parent=enableMode,
                  cmdFromParent='bash python -m Acons %s',
                  cmdToParent=curses.ascii.ctrl( 'd' ),
                  sync=False,
                  testParams=None )

qosProfileMode = Mode( name='config-qos-profile', 
                       prompt='\(config-qos-profile-[^\)]*\)#',
                       parent=configMode,
                       cmdFromParent='qos profile %s',
                       testParams=None )

configP4RuntimeMode = Mode( name='p4-runtime',
                            prompt=r'\(config-p4-runtime\)#',
                            parent=configMode,
                            cmdFromParent='p4-runtime',
                            testParams=None )

# Create a cycle so that we can reach the aboot modes from the other modes.
abootPromptMode.parent = rootLoginMode
abootPromptMode.cmdFromParent = 'reboot; read' # read inhibits shell prompt
abootPromptMode.cmdToParent = None

# Figure out how we go from one mode to another.  This function must be called
# again after defining any more Artest.Mode objects.
def findPath():

   # Build an adjacency matrix.
   neighbors = { x: [] for x in Mode.allModes_ }
   for x in Mode.allModes_:
      if x.parent:
         neighbors[ x.parent ].append( ( x, x.cmdFromParent ) )
      if x.parent and x.cmdToParent:
         neighbors[ x ].append( ( x.parent, x.cmdToParent ) )

   for x in Mode.allModes_:
      # Do a breadth-first search over the set of modes, starting at mode x, to find
      # the path to each other reachable mode.
      cmdPath = {}
      modePath = {}
      queue = []

      cmdPath[ x ] = []
      modePath[ x ] = []
      queue.append( x )

      while queue:
         current = queue.pop( 0 )
         for y, cmd in neighbors[ current ]:
            if y not in modePath:
               cmdPath[ y ] = cmdPath[ current ] + [ cmd ]
               modePath[ y ] = modePath[ current ] + [ y ]
               queue.append( y )

      # Now store the path (or None if there isn't any path) between each pair of
      # modes.
      for y in Mode.allModes_:
         if y.name in x.toMode:
            # An earlier call to this function already found a path between these
            # modes.
            continue

         x.toModeCmd[ y.name ] = cmdPath.get( y )
         x.toMode[ y.name ] = modePath.get( y )

         t9( 'Mode', x.name, '--> Mode', y.name, ':', x.toModeCmd[ y.name ] )
