# Copyright (c) 2006-2011, 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os, signal, sys, errno
import CliInputInterface
import LoggingDefs
from _TacUtils import setpdeathsig, threadsigmask

# Logging to a terminal is implemented by tailing the output
# of the file /var/log/eos-console to the stdout of the current
# terminal. The tail command runs in a child which
# is created for the console Cli session and by the 'terminal monitor'
# command, and is killed with a SIGTERM whenever the Cli process exits,
# courtesy of setpdeathsig.
loggingSubprocess = None

def tailLogSubprocess( logFilename ):
   pid = os.fork()
   if pid == 0:
      # this saves a lot of memory, compared to ManagedSubprocess
      setpdeathsig( signal.SIGTERM )
      # Needed as this is masked in CliShell's CliInputThread
      threadsigmask( signal.SIGTERM, False )
      # avoid being killed by Ctrl-C
      signal.signal( signal.SIGINT, signal.SIG_IGN )
      fd = os.open( '/dev/null', os.O_RDWR )
      os.dup2( fd, 0 )
      os.dup2( fd, 2 ) # no error about the file not there
      os.execvp( 'tail', ['tail', '-n 0', '--retry',
                          '--follow=name', logFilename ] )
   else:
      return pid

def enableLoggingMonitor( ):
   global loggingSubprocess
   if not loggingSubprocess:
      logfile = LoggingDefs.monitorLogFilename
      loggingSubprocess = tailLogSubprocess( logfile )

def sendSignalToPid( pid, sig ):
   if not pid:
      return
   try:
      os.kill( pid, sig )
      os.waitpid( pid, 0 )
   except OSError:
      # If pid doesn't exist, then do nothing.
      pass

# Kill the process we started to run tail by sending it a SIGTERM.  We don't
# bother to wait for it, because we don't want to get stuck here if something
# goes wrong.
def disableLoggingMonitor():
   global loggingSubprocess
   sendSignalToPid( loggingSubprocess, signal.SIGTERM )
   loggingSubprocess = None

class LoggingSynchronousMonitor( object ):
   # When logging synchronous is enabled, this is a callback object from CliInput
   # to synchronous logging vs input.

   def __init__( self, filename ):
      self.maxlines = 100
      self.filename = filename
      try:
         self.open( True )
      except IOError:
         self.file = None
         self.offset = 0

   def open( self, seekToEnd=False ):
      self.file = open( self.filename, "r" )
      if seekToEnd:
         self.file.seek( 0, os.SEEK_END )
      self.offset = self.file.tell()

   def __call__( self, reopen ):
      note = ''
      try:
         if self.file:
            if reopen:
               # reopen
               self.open()
               note = LoggingDefs.consoleSyncLogRotatedMsg + '\n\r'
            else:
               # We have to do this to reset the iterator
               self.file.seek( self.offset, os.SEEK_SET )
         else:
            self.open( True )
      except IOError as e:
         # logrotate creates file with 0600 and then chmod()s 0644.
         # Ignore EACCES during open() between above two operations
         if not ( reopen and e.errno == errno.EACCES ):
            self.file = None
            self.offset = 0
         return 0

      lines = 0
      for line in self.file:
         if lines == 0:
            # first line
            sys.stdout.write( '\n\r' )
         elif lines == self.maxlines:
            # we need to break out
            note += LoggingDefs.consoleSyncLogSkippedMsg + '\n\r'
            self.file.seek( 0, os.SEEK_END )
            break
         lines += 1
         sys.stdout.write( line )
         if not line.endswith( '\n' ):
            sys.stdout.write( '\n' )
         # needed for raw mode
         sys.stdout.write( '\r' )

      if not lines:
         # nothing read, did the file shrink? Just set to the end of the file
         # so we can fix our offset.
         self.file.seek( 0, os.SEEK_END )
      self.offset = self.file.tell()

      if note:
         note += LoggingDefs.consoleSyncLogRunCmdMsg + '\n\r'
         if not lines:
            sys.stdout.write( '\n\r' )
         sys.stdout.write( note )
         lines += 1
      sys.stdout.flush()
      return lines

loggingSynchronousMonitor = None

def enableLoggingSync( console ):
   global loggingSynchronousMonitor
   if not loggingSynchronousMonitor:
      if console:
         logfile = LoggingDefs.consoleSyncLogFilename
      else:
         logfile = LoggingDefs.monitorSyncLogFilename
      loggingSynchronousMonitor = LoggingSynchronousMonitor( logfile )
      CliInputInterface.loggingSyncEnabledIs( logfile,
                                              loggingSynchronousMonitor )

def disableLoggingSync():
   global loggingSynchronousMonitor
   CliInputInterface.loggingSyncEnabledIs( "" )
   loggingSynchronousMonitor = None

def Plugin( context ):
   CliInputInterface.registerMethod( 'enableLoggingSync',
                                     enableLoggingSync )
   CliInputInterface.registerMethod( 'disableLoggingSync',
                                     disableLoggingSync )
   CliInputInterface.registerMethod( 'enableLoggingMonitor',
                                     enableLoggingMonitor )
   CliInputInterface.registerMethod( 'disableLoggingMonitor',
                                     disableLoggingMonitor )
