# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import threading

import Cell
import CliSessionDataStore
import EntityManager
import Plugins
from Syscall import gettid
import TableOutput
import Tac
import Tracing

t0 = Tracing.trace0
t1 = Tracing.trace1
t5 = Tracing.trace5
t8 = Tracing.trace8
t9 = Tracing.trace9

WARN_WAIT_TIME = 30

ancestorDir = None
cliConfig = None
pendingSession = None
sessionConfig = None
sessionConfigDir = None
sessionDir = None
sessionInputDir = None
sessionStatus = None

sessionEm = None

redundancyReactor = None

# threadLocalData object needs to be created just once and not per thread. This
# object grabs the locks when attributes are accessed. The attributes are local
# to the thread.
threadLocalData = threading.local()

# This mplements a registration mechanism that allows the plugins to register
# a callback function when the current session is committed.
#
# The typical usage is to use BasicCli.maybeCallConfigSessionOnCommitHandler()
# which will either call the handler directly or register it as an onCommit
# handler based on whether it's inside a session right now.
#
# The handler takes one 'onCommit' parameter - it's True if the handler
# is invoked after a CLI session commit, and False if it's called without
# a session being involved (it's really a convenience for using the
# BasicCli.maybeCallConfigSessionOnCommitHandler() API).

# Maps each session to a dictionary of post-commit handlers.
class OnCommitHandler( object ):
   def __init__( self ):
      self.orderList = []
      self.handlerMap = {}

sessionOnCommitHandlers = {}

def registerSessionOnCommitHandler( em, key, handler ):
   """ Adds the handler to the list for the current session.

   A handler must be a callable which takes mode as a parameter. It also
   takes an optional second parameter onSessionCommit=True which tells
   from which context the handler is invoked. This is useful in case the
   handler is invoked from both config session and other cases by using
   BasicCli.maybeCallConfigSessionOnCommitHandler().

   See CliSessionOnCommit.invokeSessionOnCommitHandlers() for how the handler
   is invoked.

   Only the last handler of the same key will be registered. This is useful
   since we only need to have one callback no matter how many times a
   particular configuration is changed.
   """

   sessionName = currentSession( em )
   assert sessionName, "Not in Cli session"
   handlers = sessionOnCommitHandlers.setdefault( sessionName,
                                                  OnCommitHandler() )
   newKey = key not in handlers.handlerMap
   handlers.handlerMap[ key ] = handler
   if newKey:
      # so we can invoke handlers based on registration time
      handlers.orderList.append( key )

def discardSessionOnCommitHandlers( sessionName ):
   """ Releases the registered session commit handlers. """
   sessionOnCommitHandlers.pop( sessionName, None )
   CliSessionDataStore.cleanup( sessionName )

def configRootInitialized( em ):
   h = handlerDir( em, create=False )
   return ( h and h.configRoot and h.configRoot.rootsComplete )

def waitForConfigRootInitialized( em ):
   sleep = not Tac.activityManager.inExecTime.isZero
   Tac.waitFor( lambda: configRootInitialized( em ),
                sleep=sleep, warnAfter=0,
                description="configRoot to be initialized" )

def sessionStatusInitialized( em ):
   # this test is needed for tests using Simple EntityManager
   if not sessionStatus:
      return False
   h = handlerDir( em, create=False )
   return ( h and h.status and
            h.status.initialized and
            configRootInitialized( em ) )

def waitForSessionStatusInitialized( em ):
   sleep = not Tac.activityManager.inExecTime.isZero
   Tac.waitFor( lambda: sessionStatusInitialized( em or sessionEm ),
                sleep=sleep, warnAfter=0,
                description="session status to be initialized" )

def registerCopyHandlerDir( em, name, typeName ):
   # register an entry uner /ar/CliSessionMgr/inputDir/<name>
   inputDir = getAgentDir( em ).newEntity( "Tac::Dir", "inputDir" )
   return inputDir.newEntity( typeName, name )

def registerConfigGroupCopyHandlerDir( em, name, typeName ):
   # register an entry uner /ar/CliSessionMgr/configGroupInputDir/<name>
   inputDir = getAgentDir( em ).newEntity( "Tac::Dir", "configGroupInputDir" )
   return inputDir.newEntity( typeName, name )

def handlerDir( em, create=True ):
   # /ar/CliSessionMgr/handlerDir
   agentDir = getAgentDir( em, create )
   if not agentDir:
      return None
   h = agentDir.get( 'handlerDir' )
   if h or not create:
      return h
   return agentDir.newEntity( "Cli::Session::HandlerDir", "handlerDir" )

def configRoot( em ):
   return handlerDir( em ).configRoot

def attributeEditBypassDir( em ):
   return handlerDir( em ).attributeEditBypassTable.defaultEntityDir

def cachedSessionName():
   return getattr( threadLocalData, 'cachedSessionName', None )

def cachedSessionNameIs( name ):
   threadLocalData.cachedSessionName = name

def currentSession( em ):
   """Returns name of current session, or None"""

   # cached currentSession is based on the assumption that *nothing* will
   # change our session without going through either CliSession.enterSession or
   # CliSession.exitSession.  This assumption will break if we add any cases
   # that enter or exit sessions through, say, tacc.
   # If we do make a change, then we need to create a reactor to
   # openSessionStatus (or status) and react to oss.session.name, and reset
   # the cachedSessionName whenever that changes.  It is about less than
   # 1 microsecond to check the python variable, but more than 200 microseconds
   # to go through genericIf to tacc.  This is called by every configMount,
   # and we have some tight loops where Cli code de-references a configMount
   # thousands of times.  So this can save *minutes* off of the time.

   sName = cachedSessionName()
   if sName is not None:
      if not sName: # should be == "" if not in session
         return None
      else:
         return sName
   # Make sure everything else is mounted, too.
   if not sessionStatusInitialized( em ):
      return None
   pid = gettid()
   ossDir = sessionDir
   oss = ossDir.openSession.get( pid )
   if oss and oss.session:
      sessionName = oss.session.name
      pcs = sessionStatus.pendingChangeStatusDir.status.get( sessionName )
      if( pcs ): # check if pcs.status == "pending"  ?
         cachedSessionNameIs( sessionName )
   cachedSessionNameIs( "" )
   return None

def prefixIs( entityManager, path, flags, typename ):
   t9( "prefixIs: path", path, "flags", flags, "type", typename )
   waitForConfigRootInitialized( entityManager )
   cleanPath = EntityManager.cleanPath( path )
   rootTrie = configRoot( entityManager ).rootTrie
   assert rootTrie.isPrefix( cleanPath ), \
          "cannot ConfigMount path %s: " \
          "must also be registered using <em>.registerConfigMount" \
          % cleanPath
   return rootTrie.prefixFlags( cleanPath )

class RedundancyReactor( Tac.Notifiee ):
   ''' The CliSessionMgr state machines should be created only on the active.
   '''

   notifierTypeName = 'Redundancy::RedundancyStatus'

   def __init__( self, redundancyStatus, entityManager, callback ):
      super( RedundancyReactor, self ).__init__( redundancyStatus )
      self.redundancyStatus_ = redundancyStatus
      self.entityManager_ = entityManager
      self.callback_ = callback

      if( redundancyStatus.protocol == "sso" and
          redundancyStatus.mode != "active" ):
         t0( "Not creating creating CliSessionMgr on standby" )
         self.handleStage( '' )
      else:
         self.callback_()

   @Tac.handler( 'switchoverStage' )
   def handleStage( self, stage ):
      switchoverReady = 'SwitchoverReady'
      if( switchoverReady in self.redundancyStatus_.switchoverStage and
          self.redundancyStatus_.switchoverStage[ switchoverReady ] ):
         t8( "Creating CliSessionMgr state machines" )
         self.callback_()

# register for ConfigSession deletion notification
sessionCleanupHandlers_ = []

def registerSessionCleanupHandler( h ):
   sessionCleanupHandlers_.append( h )

class PendingChangeConfigReactor( Tac.Notifiee ):
   notifierTypeName = 'Cli::Session::PendingChangeConfigDir'

   @Tac.handler( 'config' )
   def handleConfig( self, name ):
      if name not in self.notifier_.config:
         t0( "notifying session cleanup handlers" )
         # config session deleted, notify
         for h in sessionCleanupHandlers_:
            h( name )

def getAgentDir( em, create=True ):
   parent = em.root().parent
   mgr = parent.get( 'CliSessionMgr' )
   if mgr or not create:
      return mgr
   return parent.newEntity( "Tac::Dir", "CliSessionMgr" )

def getAgent( em, create=True ):
   agentDir = getAgentDir( em, create )
   if not agentDir:
      return None
   agent = agentDir.get( "root" )
   if agent or not create:
      return agent
   return agentDir.newEntity( "Cli::Session::Agent::SessionSmCreator",
                              "root" )

def _registerBootstrapHandlers( h, em ):
   dirHandler = h.entityFilterDir.dirHandler
   registerCustomCopyHandler( em,
                              "", "Tac::Dir",
                              dirHandler,
                              direction="both" )

def doMounts( em, block=True, callback=None, bootstrap=True ):
   global ancestorDir, cliConfig, pendingSession, sessionConfig
   global sessionConfigDir, sessionDir, sessionInputDir, sessionStatus
   global sessionEm

   t0( "doMounts(", em.sysname(), ", block", block, ")" )
   assert not ( block and not bootstrap ), "block=True requires bootstrap=True"

   h = handlerDir( em )

   if sessionEm == em and sessionStatusInitialized( em ):
      t0( "sysname", em.sysname(), "already mounted for sysname" )
      if bootstrap:
         _registerBootstrapHandlers( h, em )
      if callback:
         callback()
      return

   mg = em.mountGroup()
   # clear previous data
   h.status = h.configRoot = None

   _cleanConfig = mg.mount( "session/cleanConfig", "Tac::Dir", "wi" )
   _cleanConfigStatus = mg.mount( "cli/session/cleanConfigStatus",
                                 "Cli::Session::CleanConfigStatus", "w" )
   _configRoot = mg.mount( "Sysdb/configRoot", "Sysdb::ConfigRoot", "ri" )

   ancestorDir = mg.mount( "session/ancestorDir", "Tac::Dir", "wi" )
   cliConfig = mg.mount( "cli/session/input/config",
                         "Cli::Session::CliConfig", "wi" )
   pendingSession = mg.mount( "session/sessionDir", "Tac::Dir", "wi" )
   sessionConfig = mg.mount( "cli/session/config",
                             "Cli::Session::Config", "wi" )
   sessionConfigDir = mg.mount( Cell.mountpath( "cli/sessionMap/config" ),
                                "Cli::Session::OpenSessionConfigDir", "wif" )
   sessionDir = mg.mount( Cell.mountpath( "cli/sessionMap/status" ),
                          "Cli::Session::OpenSessionStatusDir", "wi" )
   sessionInputDir = mg.mount( Cell.mountpath( "cli/sessionMap/input/config" ),
                               "Tac::Dir", "wi" )
   sessionStatus = mg.mount( "cli/session/status",
                             "Cli::Session::Status", "wi" )
   sessionEm = em

   def _init():

      h.status = sessionStatus
      h.configRoot = _configRoot

      if bootstrap:
         t0( "initialize sessionConfig and sessionStatus" )
         sessionConfig.pendingChangeConfigDir = ( "pendingChangeConfig", )
         sessionStatus.ancestorDir = ancestorDir
         sessionStatus.cleanConfig = _cleanConfig
         sessionStatus.cleanConfigStatus = _cleanConfigStatus
         sessionStatus.sessionDir = pendingSession
         sessionStatus.rootDependency = ( "rootDependency", )
         sessionStatus.pendingChangeStatusDir = ( "pendingChangeStatus", )

         _registerBootstrapHandlers( h, em )
         sessionStatus.initialized = True

      if callback:
         callback()

   mg.close( callback=_init, blocking=False )
   if block:
      waitForSessionStatusInitialized( em )

pendingChangeConfigReactor = None

def startCohabitingAgent( em, redundancyStatus=None, callback=None,
                          block=True, noPlugins=False ):
   t0( "startCohabitingAgent( sysname", em.sysname(), ")" )
   def createSm():
      t0( "Agent mounts complete" )
      # Load all ConfigSessionPlugins
      if not noPlugins:
         loadPlugins( em )

      global pendingChangeConfigReactor
      pendingChangeConfigReactor = PendingChangeConfigReactor(
         sessionConfig.pendingChangeConfigDir )

      t0( "Cohabiting CliSessionMgr starting for", em.sysname() )
      sessionConfigDir.openSessionDir = sessionInputDir
      sessionConfig.userConfig = cliConfig
      h = handlerDir( em )

      t1( "Maybe create a cleanConfig" )
      if not sessionStatus.cleanConfigStatus.cleanConfigComplete:
         t0( "Creating clean config" )
         sysname = em.sysname()
         cleanSysname = "cleanConfig-%s-%d" % ( sysname, os.getpid() )
         emClean = EntityManager.Local( sysname=cleanSysname )
         configRootClean = emClean.root().entity[ 'Sysdb/configRoot' ]
         t0( "rootsComplete:", configRootClean.rootsComplete )
         assert configRootClean.rootsComplete

         t0( "Creating clean config" )
         h.doCreateCleanConfig( emClean.root(),
                                em.root() )
         t0( "Created clean config successfully" )

      agent = getAgent( em )
      agent.cliSessionSm = ( sessionConfig,
                             sessionStatus,
                             sessionConfigDir,
                             sessionDir,
                             h,
                             em.root(),
                             pendingSession,
                             em.cEntityManager(),
                             # Won't work if we do a cohabiting switchover
                             # breadth test ...
                             None, None )

      if callback:
         callback()

   def mountsComplete():
      global redundancyReactor
      if redundancyStatus:
         redundancyReactor = RedundancyReactor( redundancyStatus, em,
                                                createSm )
      else:
         createSm()

   doMounts( em, callback=mountsComplete, block=block )

def cohabitingAgentReady( em ):
   agent = getAgent( em, create=False )
   return bool( agent and agent.cliSessionSm and
                agent.cliSessionSm.sessionRootDir )

# In a breadth test, if you are killing Sysdb, too, pass in cleanupSysdb=True
# in order to be able to start up a cohabitingAgent using the new Sysdb.
# It would be best if you used a distinct sysname, too.  Even so, this
# is tricky to try.
def stopCohabitingAgent( em, cleanupSysdb=False ):
   agentDir = getAgentDir( em )
   agent = agentDir.get( 'root' )
   if not agent:
      return
   agent.cliSessionSm.sessionConfigReactor.clear()
   agent.cliSessionSm.openSessionDirReactor = None
   agent.cliSessionSm.sessionConfigDirReactor = None
   if agent.cliSessionSm.sessionCleanupReactor:
      agent.cliSessionSm.sessionCleanupReactor.idleSessionReactor = None

   agent.cliSessionSm.sessionCleanupReactor = None
   agent.cliSessionSm = None
   agent = None
   agentDir.deleteEntity( "root" )
   if cleanupSysdb:
      cleanupSessionMgrState()

def cleanupSessionMgrState():
   t0( "cleanupSessionMgrState" )
   global sessionEm
   global ancestorDir, cliConfig, pendingSession, sessionConfig
   global sessionConfigDir, sessionDir, sessionInputDir, sessionStatus
   global sessionOnCommitHandlers

   sessionEm = None

   ancestorDir = None
   cliConfig = None
   pendingSession = None
   sessionConfig = None
   sessionConfigDir = None
   sessionDir = None
   sessionInputDir = None
   sessionStatus = None

   sessionOnCommitHandlers = {}

def cleanupCohabitingAgentIfNeeded( em ):
   agent = getAgent( em )
   if not agent:
      return
   stopCohabitingAgent( em, cleanupSysdb=True )

def sessionNames( entityManager=None ):
   waitForSessionStatusInitialized( entityManager )
   statusSet = sessionStatus.pendingChangeStatusDir.status
   namesToReturn = []
   for sessionName, pcs in statusSet.iteritems():
      if pcs.state != "aborted":
         namesToReturn.append( sessionName )
   return namesToReturn

def uniqueSessionNameCounter():
   return getattr( threadLocalData, 'uniqueSessionNameCounter', 0 )

def uniqueSessionNameCounterIs( value ):
   threadLocalData.uniqueSessionNameCounter = value

def uniqueSessionName( entityManager=None, prefix="gensym", okToReuse=False ):
   waitForSessionStatusInitialized( entityManager )
   if not validSessionName( prefix ):
      prefix = "invalidName"
   newSessionName = None
   counter = 0
   if not okToReuse:
      counter = uniqueSessionNameCounter()
      uniqueSessionNameCounterIs( uniqueSessionNameCounter() + 1 )
   tid = threading.current_thread().ident
   while not newSessionName:

      candidate = "%s-%d-%d-%d" % ( prefix, os.getpid(), tid, counter )
      counter += 1
      if sessionStatus.pendingChangeStatusDir and \
         sessionConfig.pendingChangeConfigDir and \
         not ( sessionStatus.pendingChangeStatusDir.status.get( candidate ) or
               sessionConfig.pendingChangeConfigDir.config.get( candidate )):
         newSessionName = candidate
   if counter > uniqueSessionNameCounter():
      uniqueSessionNameCounterIs( counter )
   return newSessionName

def getSessionUserName():
   username = ( os.environ.get( "LOGNAME" ) or
                os.environ.get( "USER", "[unknown]" ))
   return username

def getPrettyTtyName():
   terminal = os.getenv( "REALTTY" )
   if not terminal:
      for fd in ( 0, 1, 2 ):
         try:
            terminal = os.ttyname( fd )
            break
         except OSError:
            pass
   if not terminal:
      terminal = ( os.environ.get( "SSH_TTY" ) or
                   os.environ.get( "STY" ) or
                   os.ctermid() )
   # make terminal name prettier (patterned after Cli/UtmpDump)
   if terminal:
      # Strip off a leading '/dev/', if any
      terminal = terminal.replace( '/dev/', '' )
      terminal = terminal.replace( 'pts/', 'vty' ).replace( 'ttyS', 'con' )
   return terminal

def validSessionName( sessionName ):
   for c in sessionName:
      if not ( c.isalnum() or c in "_-" ):
         return False
   return True

class SessionAlreadyCompletedError( Exception ):
   def __init__( self, sessionName ):
      # NOTE: this message is displayed by the CLI.
      msg = "Session %s is already completed." % sessionName
      Exception.__init__( self, msg )

def isSessionPendingCommitTimer( sessionName ):
   return isCommitTimerInProgress() and \
          sessionStatus.commitTimerSessionName == sessionName

def commitTimerSessionName():
   return sessionStatus.commitTimerSessionName

def isSessionPresent( sessionName ):
   """Return True if the specified session is present, False otherwise. """
   if ( sessionConfig.pendingChangeConfigDir.config.get( sessionName ) or
        sessionStatus.pendingChangeStatusDir.status.get( sessionName ) ):
      return True
   else:
      return False

def canCommitSession( em, sessionName ):
   """Return "" if we can successfully commit a  session, or return an
      error message."""
   if sessionName is None:
      # We are already in config session mode.
      sessionName = currentSession( em )
   elif not isSessionPresent( sessionName ):
      return "Cannot commit non-existent session %s." % sessionName
   pcs = sessionStatus.pendingChangeStatusDir.status.get( sessionName )
   if pcs and pcs.completed:
      return "Cannot commit session %s, it is already complete." % sessionName
   elif pcs and isCommitTimerInProgress() and pcs.state != "pendingCommitTimer":
      return  "Cannot commit because another session, %s is pending commit" \
               " timer." % sessionStatus.commitTimerSessionName
   elif pcs and pcs.noCommit:
      return  "Cannot commit no-commit session %s" % sessionName
   return ""

def canEnterSession( sessionName, entityManager, noCommit ):
   """Return "" if we can successfully enter the specified session, or
   return an error message otherwise."""
   waitForSessionStatusInitialized( entityManager )
   if sessionName and not validSessionName( sessionName ):
      return "Session name %s is invalid." % sessionName

   pcs = sessionStatus.pendingChangeStatusDir.status.get( sessionName )
   if pcs and pcs.completed:
      return "Cannot enter session %s, it is already complete." % sessionName
   if pcs and noCommit and not pcs.noCommit:
      return ( "Cannot change an already existing session %s to no-commit"
               % sessionName )
   if pcs and pcs.noCommit and pcs.noCommitUser != getSessionUserName():
      return ( "User %s cannot enter no-commit session %s created by %s"
               % ( getSessionUserName(), sessionName, pcs.noCommitUser ) )
   if pcs and pcs.state == "pendingCommitTimer":
      return ( "Cannot enter session %s, it is in "
             "pendingCommitTimer state." % sessionName )
   return ""

def enterSession( sessionName, entityManager=None, noCommit=False ):
   """This process will encapsulate all Cli config commands in a session,
   and the commands won't be applied until the session is committed. If
   we are unable to enter the session, this will signal Tac.Timeout, or
   return a non-empty string explaining why we can't."""

   t0( 'Entering session:', sessionName )
   if not entityManager:
      entityManager = sessionEm
   response = canEnterSession( sessionName, entityManager, noCommit )
   if response != "":
      return response

   if not sessionName:
      sessionName = uniqueSessionName( entityManager, "sess" )

   configSet = sessionConfig.pendingChangeConfigDir.config
   statusSet = sessionStatus.pendingChangeStatusDir.status
   if not sessionName in configSet:
      t0( "Creating session config for", sessionName )
      pcc = configSet.newMember( sessionName, False, False, True, Tac.now() )
   sleep = not Tac.activityManager.inExecTime.isZero
   Tac.waitFor( lambda: statusSet.get( sessionName ), sleep=sleep,
                warnAfter=WARN_WAIT_TIME,
                description="session to be created [pcs]" )

   # exit in case we were already in one
   exitSession( entityManager )

   pcc = configSet.get( sessionName )
   if noCommit and not pcc.noCommit:
      pcc.noCommit = noCommit
      pcc.noCommitUser = getSessionUserName()
   pid = gettid()
   osc = sessionInputDir.get( str( pid ) )
   if( not osc ):
      osc = sessionInputDir.newEntity( "Cli::Session::OpenSessionConfig",
                                       str( pid ))
   oss = Tac.waitFor( lambda: sessionDir.openSession.get( pid ), sleep=sleep,
                      warnAfter=WARN_WAIT_TIME,
                      description="session to be created [sessionStatus]" )
   osc.session = sessionName
   osc.user = getSessionUserName()
   osc.terminal = getPrettyTtyName()

   Tac.waitFor( lambda: oss.session == pcc, sleep=sleep,
                warnAfter=WARN_WAIT_TIME,
                description="process to be attached to session" )
   if pcc:
      cachedSessionNameIs( sessionName )
   else:
      cachedSessionNameIs( "" )
   t0( "Entered CliSession", sessionName )
   return ""

def _cleanSessionConfig( em, sessionName ):
   assert em
   pid = gettid()
   osc = sessionInputDir.get( str( pid ))
   oss = sessionDir.openSession.get( pid )
   if osc and osc.session == sessionName:
      osc.session = ""
   if oss:
      Tac.waitFor( lambda: oss.session != sessionName,
                   sleep=not Tac.activityManager.inExecTime.isZero,
                   warnAfter=WARN_WAIT_TIME,
                   description="oss.session to be None" )
   configSet = sessionConfig.pendingChangeConfigDir.config
   pcc = configSet.get( sessionName )
   if( pcc and pcc.commandDir.get( pid )):
      del pcc.commandDir[ pid ]

def exitSession( entityManager=None, forceSessionName=None ):
   waitForSessionStatusInitialized( entityManager )
   pid = gettid()
   osc = sessionInputDir.get( str( pid ))
   oss = sessionDir.openSession.get( pid )
   oldSessionName = oss and oss.session and osc and osc.session
   needToExitCurrentSession = ((forceSessionName is None ) or
                               ( osc and osc.session == forceSessionName ))
   if needToExitCurrentSession:
      cachedSessionNameIs( "" )
      if osc:
         osc.session = ""
   if( not osc or not oss ): # couldn't be in a session
      assert not oss, "session status existed without session config."
      return
   if needToExitCurrentSession:
      Tac.waitFor( lambda: oss.session == None,
                   sleep=not Tac.activityManager.inExecTime.isZero,
                   warnAfter=WARN_WAIT_TIME,
                   description="oss.session to be None" )
   if forceSessionName:
      oldSessionName = forceSessionName
   if oldSessionName:
      t0( "Exiting CliSession", oldSessionName )
      configSet = sessionConfig.pendingChangeConfigDir.config
      pcc = configSet.get( oldSessionName )
      if( pcc and pcc.commandDir.get( pid )):
         del pcc.commandDir[ pid ]

def addSessionRoot( em, root, sessionName=None ):
   if sessionName is None:
      sessionName = currentSession( em )
   assert sessionName, "Cannot addRoot when not in Cli session"
   waitForSessionStatusInitialized( em )
   pcc = sessionConfig.pendingChangeConfigDir.config.get( sessionName )
   pcs = sessionStatus.pendingChangeStatusDir.status.get( sessionName )

   if not pcs or pcs.completed or pcs.state == 'pendingCommitTimer':
      raise SessionAlreadyCompletedError( sessionName )

   # Note that currentSession already called ensureSessionMounts()
   root = EntityManager.cleanPath( root )
   if root and root[-1] == "/":
      root = root[ 0: -1 ]
   if not root in pcc.target:
      # Note, may have been copied into status because of a ptr, so
      # may already be in status.local/ancestor
      pcc.target[ root ] = True
   waitStr = "path %s to appear in session" % root
   t5( "Adding root %s to session %s" % ( root, sessionName ))
   Tac.waitFor( lambda: ( pcs.completed or
                          ( root in pcs.local ) or
                          ( root in pcs.error ) ),
                sleep=not Tac.activityManager.inExecTime.isZero,
                warnAfter=WARN_WAIT_TIME,
                description=waitStr )
   if not pcs.completed:
      if root in pcs.local:
         return
      error = pcs.error.get( root )
      if error:
         raise Exception( " Path: " + root + " not mounted because " + error )
   # Someone must have cleaned up the session status
   raise SessionAlreadyCompletedError( sessionName )

def sessionConfigRoots( em, sessionName=None ):
   if sessionName is None:
      sessionName = currentSession( em )
   assert sessionName, "No relevant session"
   waitForSessionStatusInitialized( em )

   pcs = sessionStatus.pendingChangeStatusDir.status.get( sessionName )
   if not pcs:
      assert sessionName, "Session %s is no longer available" % sessionName

   return pcs.local.keys()

# For tests, only
def _sessionCommandComplete( commandDir, requestResponse ):
   cid, _sleep, _oldCommand = requestResponse
   return cid in commandDir.response

def sessionCommand( em, command, forceSessionName=None,
                    warnAfter=WARN_WAIT_TIME,
                    # for debugging only
                    requestOnly=False, responseOnly=False,
                    cmdArg="" ):
   """For debugging we allow caller to split sessionCommand into two parts.
   The value returned by requestOnly=True must be passed as the value of
   the responseOnly keyword arg in order to complete the original command.
   The caller can tamper with state between the 2 calls.
   If requestOnly is True, we initiate a new call, and return after we
   see that the mgr agent has gotten the request.
   If requestOnly is the string 'startOnly' we don't wait to see if agent
   has gotten the request, but return immediately"""
   assert not ( requestOnly and responseOnly )
   if requestOnly:
      assert requestOnly == True or requestOnly == "startOnly"
   if forceSessionName:
      sessionName = forceSessionName
   else:
      sessionName = currentSession( em )
      assert sessionName, "Not in Cli session"
   pcs = sessionStatus.pendingChangeStatusDir.status.get( sessionName )
   pcc = sessionConfig.pendingChangeConfigDir.config.get( sessionName )
   pid = gettid()
   commandRequestDir = pcc.commandDir.get( pid )
   responseErrMsg = 'response for PID %d to appear in PendingChangeStatus'
   if responseOnly:
      cid, sleep, oldCommand = responseOnly
      assert oldCommand == command
      if( not commandRequestDir ):
         # Really, this should not usually happen unless we exited the session
         # between the request and response.
         commandRequestDir = pcc.commandDir.newMember( pid )
      commandResponseDir = Tac.waitFor( lambda : pcs.commandResponse.get( pid ),
            sleep=sleep, warnAfter=warnAfter,
            description=( responseErrMsg % pid ))
   else:
      if( not commandRequestDir ):
         commandRequestDir = pcc.commandDir.newMember( pid )
      commandRequestDir.commandId += 1
      cid = commandRequestDir.commandId
      commandRequestDir.command.addMember(
         Tac.Value( "Cli::Session::CommandRequest", id=cid, cmd=command,
                    cmdArg=cmdArg ))
      sleep = not Tac.activityManager.inExecTime.isZero
      commandResponseDir = Tac.waitFor( lambda : pcs.commandResponse.get( pid ),
            sleep=sleep, warnAfter=warnAfter,
            description=( responseErrMsg % pid ))
      if not requestOnly == "startOnly":
         Tac.waitFor( lambda : commandResponseDir.command.get( cid ),
                      sleep=sleep, warnAfter=warnAfter,
                      description='command id %d to appear in command' %  cid )

   if requestOnly:
      return ( cid, sleep, command, )
   Tac.waitFor( lambda : cid in commandResponseDir.response, sleep=sleep,
                warnAfter=warnAfter,
                description='command id %d to appear in response' % cid )
   response = commandResponseDir.response.get( cid )
   del commandRequestDir.command[ cid ]
   if( response ):
      # first char of response is "+" for success or "-" for failure.
      # If "+", then rest of string should be ""
      response = response[1:]
   return response

def deleteSession( em, sessionName ):
   t0( 'Deleting session:', sessionName )
   configSet = sessionConfig.pendingChangeConfigDir.config
   statusSet = sessionStatus.pendingChangeStatusDir.status
   pcc = configSet.get( sessionName )
   if not pcc:
      t0( "Session", sessionName, "not removed: config not found" )
      return "Cannot delete non-existent session " + sessionName
   # Make sure we're not in the session, and clean any command queue we may
   # have put there.
   response = sessionCommand( em, "remove", forceSessionName=sessionName )
   if response:
      t0( "Session", sessionName, "not removed:", response )
      return response
   sleep = not Tac.activityManager.inExecTime.isZero
   def sessionDeleted():
      pcs = statusSet.get( sessionName )
      return not pcs
   Tac.waitFor( sessionDeleted, sleep=sleep,
                warnAfter=WARN_WAIT_TIME,
                description="session to be marked for deletion" )
   discardSessionOnCommitHandlers( sessionName )
   _cleanSessionConfig( em, sessionName )
   return ""

def abortSession( em, sessionName=None ):
   if sessionName is None:
      sessionName = currentSession( em )
   t0( 'Aborting session:', sessionName )
   if not sessionName:
      return "Not in Cli session"

   discardSessionOnCommitHandlers( sessionName )
   pcsDir = sessionStatus.pendingChangeStatusDir.status
   # If pcc deleted, then pcs deleted.
   pcc = sessionConfig.pendingChangeConfigDir.config.get( sessionName )
   if not pcc:
      return ""
   sleep = not Tac.activityManager.inExecTime.isZero
   # should never wait for very long
   pcs = Tac.waitFor( lambda : pcsDir.get( sessionName ), sleep=sleep,
                      warnAfter=WARN_WAIT_TIME,
                      description="CliSessionMgr to create the session.")

   if pcs.completed and pcs.success:
      return "Too late to abort; session already committed."

   response = sessionCommand( em, "abort", forceSessionName=sessionName )
   if response:
      print response
      return response
   try:
      sleep = not Tac.activityManager.inExecTime.isCurrent
      Tac.waitFor( lambda: ( pcs.state == "aborted" and
                             not pcs.success and
                             pcs.completed ),
                   sleep=sleep,
                   warnAfter=WARN_WAIT_TIME,
                   description="session to abort" )
   except Tac.Timeout:
      return "Abort timed out"

   return ""

def isCommitTimerInProgress():
   return sessionStatus.commitTimerInProgress

def commitSession( em, debugAttrChanges=None, timerValue=None, sessionName=None ):
   '''Returns an empty string on success, an error string if the session
   could not be committed, or None if the session no longer exists.'''
   t0( "commitSession()", 'debugAttrChanges', debugAttrChanges,
       'sessionName', sessionName, 'timerValue', timerValue )
   if sessionName is None:
      sessionName = currentSession( em )
   assert sessionName, "Not in Cli session"
   pcs = sessionStatus.pendingChangeStatusDir.status.get( sessionName )
   pcc = sessionConfig.pendingChangeConfigDir.config.get( sessionName )
   if not pcc:
      t0( "Session", sessionName, "not committed: config not found" )
      discardSessionOnCommitHandlers( sessionName )
      return None
   if debugAttrChanges is None:
      # there is a test command that sets this flag
      debugAttrChanges = pcc.logAttributeEdits
   else:
      pcc.logAttributeEdits = bool( debugAttrChanges )
   pcc.commitTimeout = timerValue or Tac.endOfTime

   response = ""
   # If timerValue is None, this is due to a "commit".
   # If timerValue is not None, this is due to a "commit timer". In that case,
   # do not commit if it is only a timerValue change(not the first commit
   # timer command).
   if timerValue is None or not isCommitTimerInProgress():
      t0( 'Committing session:', sessionName )
      response = sessionCommand( em, "commit", forceSessionName=sessionName )
   if response:
      pcc.logAttributeEdits = False
      return response
   try:
      if isCommitTimerInProgress():
         Tac.waitFor( lambda: ( pcs.success ),
                      warnAfter=WARN_WAIT_TIME,
                      description="session to commit" )
      if debugAttrChanges is not None:
         output = []
         for attrEditEntry in pcs.attributeEditLog.attributeEditEntry.itervalues():
            attrEdit = attrEditEntry.attributeEdit
            for path in attrEditEntry.pathCounter:
               if not output:
                  tableHeadings = ( 'ParentName', 'AttrName', 'AttrType',
                                    'Op' )
                  # set tableWidth to unlimited so that we do not wrap around.
                  # this is easier to create logres.
                  table = TableOutput.createTable( tableHeadings, tableWidth=9999 )
                  f1 = TableOutput.Format( justify='left' )
                  f1.noPadLeftIs( True )
                  f2 = TableOutput.Format( justify='left' )
                  f2.noPadLeftIs( True )
                  f3 = TableOutput.Format( justify='left' )
                  table.formatColumns( f1, f2, f2, f3 )

               line = ( path, attrEdit.attributeName,
                        attrEdit.attributeType.split('::')[ -1 ],
                        attrEdit.editType )
               if not line in output:
                  table.newRow( *line )
                  output.append( line )
         if output:
            print table.output()
   except Tac.Timeout:
      return "Commit timed out"
   finally:
      pcc.logAttributeEdits = False

   return ""

def rollbackSession( em, configGroups=None ):

   def runSessionCommand( em, command, cmdArg="" ):
      clearCount = pcs.clears
      response = sessionCommand( em, command, cmdArg=cmdArg )
      if response:
         print response
         return response
      try:
         Tac.waitFor( lambda: pcs.clears != clearCount, warnAfter=WARN_WAIT_TIME,
                      description="session config state to be cleared" )
      except Tac.Timeout:
         return "rollback timed out"
      return ""

   sessionName = currentSession( em )
   assert sessionName, "Not in Cli session"
   # Should only happen during breadth tests:
   if not sessionStatus.cleanConfigStatus.cleanConfigComplete:
      return "Cannot rollback if no cleanConfig exists. Use useCleanConfig=True."
   pcs = sessionStatus.pendingChangeStatusDir.status.get( sessionName )
   if pcs.state == 'pendingCommitTimer':
      raise SessionAlreadyCompletedError( sessionName )

   command = "clearConfig"
   if configGroups:
      pcs.configGroupRollback = True
      for group in configGroups:
         response = runSessionCommand( em, command, cmdArg=group )
         if response:
            return response
      return ""
   else:
      pcs.configGroupRollback = False
      return runSessionCommand( em, command )

# You can rollback from a saved, completed, session.
# It is more efficient than using a saved config file because
# there is no Cli parsing needed.
def rollbackFromSession( em, fromSession, configGroups=None ):
   sessionName = currentSession( em )
   if not sessionName:
      return "Not in Cli session"
   pcs = sessionStatus.pendingChangeStatusDir.status.get( sessionName )
   if( not pcs ):
      msg = "Cannot rollback session %s; it does not exist"
      return msg % sessionName
   # pcs exists
   if( pcs.completed ):
      msg = "Cannot rollback session %s; it is already completed"
      return msg % sessionName

   fromPcs = sessionStatus.pendingChangeStatusDir.status.get( fromSession )
   if not ( fromPcs and fromPcs.completed and fromPcs.success and
            fromPcs.state == "completed" ):
      msg = "Cannot rollback from session %s;"
      if not fromPcs:
         msg += " the session does not exist"
      elif not ( fromPcs.completed and fromPcs.success and
                 fromPcs.state == "completed" ):
         msg += " the session is not successfully completed"
      return  msg % fromSession
   # This is not a supported feature, so this path needs more testing.
   fromRoot = pendingSession.get( fromSession )
   if not fromRoot.keys():
      msg = "configuration state of session %s has been reclaimed"
      return msg % fromSession

   # do clean config
   # Really, all we need to do is do an entityCopy for each root in clean-config
   # that is *not* in fromPcs
   # Copy fromPcs first, the session has the transitive closure of roots.
   # Anything *not* in that session is now left over, and we can copy
   # that root from clean-config.
   result = rollbackSession( em )
   if result:
      return result

   # ConfigGroups are not handled here
   assert configGroups is None
   h = handlerDir( em )
   if( h.doReplaceConfig( fromRoot, sessionName,
                          h.configRoot ) ):
      pcs.rollbackSession = fromSession
      pcs.cleared = True
      return ""
   else:
      return "replace config did not succeed."

def sessionSize( em, sessionName=None, warnAfter=WARN_WAIT_TIME ):
   """Return the amount of memory used by the given sessionName or current session
   if sessionName is None.
   Note that it's computed when this function is called, so it is an expensive
   operation."""
   if not sessionName:
      sessionName = currentSession( em )
   before = Tac.now()
   size = sessionCommand( em, "size", sessionName, warnAfter )
   t0( 'Computing size of session', sessionName, 'took', Tac.now() - before,
       'seconds' )
   return int( size )

def validFilterAttribute( typeName, attr ):
   """True if attribute 'attr' is a logging attribute of type 'typeName'"""

   t = Tac.typeNode( typeName )
   return t.attr( attr ) is not None and t.attr( attr ).isLogging

def validAttribute( typeName, attr ):
   """True if attribute 'attr' is an attribute of type 'typeName'"""

   t = Tac.typeNode( typeName )
   return t.attr( attr ) is not None

def registerCustomCopyHandler( em, path, typeName, handler,
                               direction="out", onlyConfigGroupTable=False ):
   """
   We can register a special handler for entityCopy either by path, or
   by type, or by both.

   path-only is most specific --- it applies only to the specific entity at
   that path.

   If you pass path and typename, then it applies to all entities of
   exactly type == typename that are in the subtree rooted at path.
   Finally, if you pass in only the typename, then the handler applies
   to entities of type typename, anywhere they are encountered (assuming
   not overridden by a path).

   So a handler for type Foo::Bar is overridden, for entities under the
   x/y subtree by a handler for type Foo::Bar scoped to x/y, and *that*
   is overridden, for the entity x/y/a/b/c by a handler registered
   specifically for x/y/a/b/c

   (From comments in CliSession.tac)

   direction can be either "in", "out", or "both"
   """
   assert handler
   h = handlerDir( em )
   if onlyConfigGroupTable:
      assert direction == "out"

      # assert that input handler has "filteredAttribute" attribute
      assert hasattr( handler, "filteredAttribute")

      # check for existing filter defined for input path and type
      typeHandlers = h.configGroupHandlerTable.typeSpecificHandlers[ typeName ]
      if path in typeHandlers.pathSpecific:
         existingHandler = typeHandlers.pathSpecific[ path ]
         # assert that existing handler has "filteredAttribute" attribute
         assert hasattr( existingHandler, "filteredAttribute" )

         # do filtering as intended by existing handler
         for attr in existingHandler.filteredAttribute.attributeName:
            handler.filteredAttribute.attributeName[ attr ] = True
         t1( "registering configGroup CustomCopyHandler for type:", typeName,
             "path:", path, "with filtering attributes:",
             handler.filteredAttribute.attributeName.keys() )
      else:
         t1( "no existing handler found for type:", typeName, "path:", path,
             "while registering handler for onlyConfigGroupTable")

   if direction == "in" or direction == "both":
      h.inCopyHandlerTable.registerCustomCopyHandler( path, typeName, handler )
   if direction == "out" or direction == "both":
      if not onlyConfigGroupTable:
         h.copyHandlerTable.registerCustomCopyHandler( path, typeName, handler )
      h.configGroupHandlerTable.registerCustomCopyHandler( path, typeName, handler )

def registerEntityCopyAllowedAttrs( em, path, typeName, allowedAttrs ):
   """Only copy the allowedAttrs during EntityCopy. allowedAttrs are used at
   commit stage of Cli Session, but only when the Session involves rolling back
   ConfigGroups"""
   t = Tac.typeNode( typeName )
   # verify that allowedAttrs are valid
   for attr in allowedAttrs:
      assert validAttribute( typeName, attr )
   boilerPlateAttrs = [ "name", "fullName", "parent", "parentAttrName", "entity" ]
   allAttrs = [ attr.name for attr in t.attributeQ if attr.name not in
                boilerPlateAttrs ]
   filteredAttrs = [ attr for attr in allAttrs if attr not in allowedAttrs and
                     validFilterAttribute( typeName, attr ) ]
   t1( "filtered Attributes:", filteredAttrs, "for type:", typeName, "path:", path )

   # verify that filteredAttrs doesnt already exists as part of preAttribute or
   # postAttribute list
   h = handlerDir( em )
   key = Tac.Value( "Cli::Session::HandlerContext", path=path, typeName=typeName )
   if key in h.configGroupFilterDir.ecf:
      copyFilter = h.configGroupFilterDir.ecf[ key ]
      if copyFilter.preAttribute:
         preList = copyFilter.preAttribute.attributeName.values()
         for attr in filteredAttrs:
            assert attr not in preList
      if copyFilter.postAttribute:
         postList = copyFilter.postAttribute.attributeName.values()
         for attr in filteredAttrs:
            assert attr not in postList

   registerEntityCopyFilter( em, path, typeName, filteredAttrs,
                             direction="out", OnlyFilterConfigGroups=True )

def registerEntityCopyFilter( em, path, typeName, filteredAttrs,
                              orderedAttrsBefore=None,
                              orderedAttrsAfter=None,
                              direction="out",
                              OnlyFilterConfigGroups=False ):
   """Skip copying the attributes in filteredAttrs.  Optionally impose an order
   on the copying: first copy orderedAttrsBefore, in the order of the list.
   Then copy all other attrs not in filteredAttrs, orderedAttrsBefore, or
   orderedAttrsAfter.  And, finally, optionally copy (in the order of the list)
   copy the attributes named in the list orderedAttrsAfter

   See help string from registerCustomCopyHandler for more info"""
   if OnlyFilterConfigGroups:
      assert direction == "out"
   h = handlerDir( em )
   if not ( orderedAttrsBefore or orderedAttrsAfter ):
      attrFilter = Tac.newInstance( "Cli::Session::AttributeFilter" )
      for attr in filteredAttrs:
         assert validFilterAttribute( typeName, attr )
         attrFilter.attributeName[ attr ] = True
      if direction == "in" or direction == "both":
         h.inCopyHandlerTable.registerEntityCopyFilter( h.entityFilterDir, path,
                                                        typeName, attrFilter )
      if direction == "out" or direction == "both":
         if not OnlyFilterConfigGroups:
            h.copyHandlerTable.registerEntityCopyFilter( h.entityFilterDir, path,
                                                         typeName, attrFilter )
         h.configGroupHandlerTable.registerEntityCopyFilter( h.configGroupFilterDir,
            path, typeName, attrFilter )

   else:
      def buildFilter( ecFilterDir ):
         key = Tac.Value( "Cli::Session::HandlerContext", path=path,
                          typeName=typeName )
         ecf = ecFilterDir.ecf.newMember( key )
         ecf.filteredAttribute = ()
         for attr in filteredAttrs:
            assert validFilterAttribute( typeName, attr )
            ecf.filteredAttribute.attributeName[ attr ] = True

         if orderedAttrsBefore:
            ecf.preAttribute = ()
            for i, attr in enumerate( orderedAttrsBefore ):
               assert validFilterAttribute( typeName, attr )
               ecf.preAttribute.attributeName[ i ] = attr
               ecf.filteredAttribute.attributeName[ attr ] = True

         if orderedAttrsAfter:
            ecf.postAttribute = ()
            for i, attr in enumerate( orderedAttrsAfter ):
               assert validFilterAttribute( typeName, attr )
               ecf.postAttribute.attributeName[ i ] = attr
               ecf.filteredAttribute.attributeName[ attr ] = True
         return ecf

      if not OnlyFilterConfigGroups:
         ecf = buildFilter( h.entityFilterDir )
      if direction == "in" or direction == "both":
         assert OnlyFilterConfigGroups is False
         h.inCopyHandlerTable.registerCustomCopyHandler( path, typeName, ecf )
      if direction == "out" or direction == "both":
         if not OnlyFilterConfigGroups:
            h.copyHandlerTable.registerCustomCopyHandler( path, typeName, ecf )
         cgEcf = buildFilter( h.configGroupFilterDir )
         h.configGroupHandlerTable.registerCustomCopyHandler( path, typeName, cgEcf )

def registerAttributeEditLogBypass( em, parentPath, attrName, attrType,
                                    attrEditType, defaultEnt=None ):
   """ Ideally, no attributes should be changed during "config replace
   running-config force". Some exceptions are, however, found to be acceptable.
   Agents can use this API to register exceptions. Registered attributes will
   not be shown in "debug-attribute".
   We find that some agents do not delete a config entity even when all its
   configurable fields are default, but it is legal for config replace to delete
   this entity. To ensure that the entity deleted by config replace has default
   setting, agents can further provide a defaultEnt to which all deleted entities
   (of the same type) are compared during entity copy. Note that the comparison
   excludes construtor parameters and hence the defaultEnt can be instantiated
   using dummy construtors.

   parentPath: the path to the bypassed entity relative to the root. If an
               absolute path is passed in (with a leading '/'), it will be
               automatically converted to a relative path.
   attrName: the attribute name as reported in debug-attribute output. "" can
             be specified to apply to all attribute names.
   attrType: the attribute type as reported in debug-attribute output. "" can
             be specified to apply to all attribute types.
   attrEditType: 'set' or 'del'.
   defaultEnt: when specified, EntityCopy compares it against the entity that
               is in the destination but not the source. The entity is bypassed
               by debug-attribute if all non-constructor attributes are the same.
   """
   h = handlerDir( em )
   ae = Tac.Value( "Cli::Session::AttributeEdit", attrName, attrType,
                   attrEditType )
   if parentPath.startswith( '/' ):
      # strip the first two components
      parentPath = '/'.join( parentPath.split( '/' )[ 3: ] )
   aeBypassEntry = h.attributeEditBypassTable.attributeEditBypassEntry.\
                   newMember( ae )
   aeBypassEntry.parentPath[ parentPath ] = True
   if defaultEnt:
      aeBypassEntry.defaultEnt = defaultEnt

def registerEntityCopyDebug( entityManager, attr, entityType="", pathname="",
                             editType="set", action=None ):
   assert sessionConfig.userConfig
   assert editType in [ "set", "del" ]
   assert attr
   assert action in [ None, "actionAssert", "actionTrace" ]
   assert cliConfig == sessionConfig.userConfig

   key = Tac.Value( "Cli::Session::AttributeEdit",
                    attributeName=attr, attributeType=entityType,
                    editType=editType )
   if action: # add an entry
      if entityType:
         ecDebugConfig = cliConfig.debugPerTypeEntityCopy.get( key )
         if not ecDebugConfig:
            ecDebugConfig = cliConfig.debugPerTypeEntityCopy.newMember( key )
         ecDebugConfig.action[ pathname ] = action
      else:
         ecDebugConfig = cliConfig.debugEntityCopy.get( pathname )
         if not ecDebugConfig:
            ecDebugConfig = cliConfig.debugEntityCopy.newMember( pathname )
         entry = ecDebugConfig.debugConfig.get( key )
         if not entry:
            entry = ecDebugConfig.debugConfig.newMember( key )
         entry.action = action
   else: # delete the entry for this description
      if entityType:
         ecDebugConfig = cliConfig.debugPerTypeEntityCopy.get( key )
         if ecDebugConfig:
            if ecDebugConfig.action.get( pathname ):
               del ecDebugConfig.action[ pathname ]
            if len( ecDebugConfig.action ) == 0:
               del cliConfig.debugPerTypeEntityCopy[ key ]
      else:
         ecDebugConfig = cliConfig.debugEntityCopy.get( pathname )
         if ecDebugConfig:
            entry = ecDebugConfig.debugConfig.get( key )
            if entry:
               del ecDebugConfig.debugConfig[ key ]
            if not entityType and not len( ecDebugConfig.debugConfig ):
               del cliConfig.debugEntityCopy[ pathname ]

def registerConfigGroup( em, cfgGroupName, mountPath ):
   """ Register 'mountPath' as a member of config-group 'cfgGroupName'. Adds to
   any previously registered mounts
   """
   h = handlerDir( em )
   assert mountPath in h.configRoot.root, 'Not a config root: %s' % mountPath

   mountPath = EntityManager.cleanPath( mountPath )

   configGroup = sessionStatus.configGroup
   if cfgGroupName not in configGroup:
      configGroup.newMember( cfgGroupName )
   t9( "Adding mountpoint:", mountPath, "CONFIG_GROUP:", cfgGroupName )
   configGroup[ cfgGroupName ].root[ mountPath ] = True

def registerConfigRootDependency( em, path, dependent ):
   """Register that dependent depends on path (the former having a Ptr to
   the latter). This causes EntityCopy to copy them together.

   To keep things simple:
   1. No chain dependency is allowed. A path being dependent on cannot depend
      on another path.
   2. A path can at most depend on one other path.
   """
   assert '*' not in path
   assert '*' not in dependent
   assert dependent not in sessionStatus.rootDependency.root
   for p in sessionStatus.rootDependency.root.itervalues():
      assert path not in p.dependent
      if p.name != path:
         assert dependent not in p.dependent

   root = sessionStatus.rootDependency.newRoot( path )
   root.dependent[ dependent ] = True

def unregisterConfigRootDependency( em, path, dependent ):
   # This is just for testing so we can try different dependencies
   root = sessionStatus.rootDependency.root.get( path )
   if root:
      del root.dependent[ dependent ]
      if not root.dependent:
         del sessionStatus.rootDependency.root[ path ]

def registerConfigRootFilter( em, name, mountPath ):
   """ Register 'mountPath' to be filtered from config root.
   """
   t0( "registerConfigRootFilter", name, mountPath )
   h = handlerDir( em )
   assert mountPath in h.configRoot.root, 'Not a config root: %s' % mountPath
   mountPath = EntityManager.cleanPath( mountPath )
   t9( "Adding mountpoint:", mountPath, "to filter", name )
   table = h.configRootFilterTable.newFilter( name )
   table.root[ mountPath ] = True
   state = Tac.newInstance( "Ark::PathTrieState" )
   table.rootTrie.prefixIs( mountPath, state )

def activeConfigRootFilterIs( em, name ):
   """ Set the active config root filter table. Note, the API currently doesn't
   work (very well) for the existence of multiple filters. This should be addressed
   in case the need arises. """
   h = handlerDir( em )
   if name:
      t0( "set active config root filter to", name )
      h.activeConfigRootFilter = h.configRootFilterTable.filter.get( name )
   else:
      t8( "restore active config root filter to default" )
      h.activeConfigRootFilter = None

def activeConfigRootFilter( em ):
   h = handlerDir( em )
   return h.activeConfigRootFilter

def loadPlugins( entityManager ):
   t0( "load ConfigSessionPlugins" )
   Plugins.loadPlugins( 'ConfigSessionPlugin', entityManager )
