# Copyright (c) 2011, 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from array import array
from socket import gethostname
from datetime import datetime, timedelta
import re, os, time
import Logging
import Tac

SchedCliType = Tac.Type( "System::CliScheduler::ScheduledCli" )
intervalMin = 2
# Maximum scheduling interval is 1 day. That transalates to 24X3600 seconds
intervalMax = ( 24 * 60 )

# Timeout in minutes
timeoutMin = SchedCliType.timeoutMin / 60
timeoutMax = SchedCliType.timeoutMax / 60
timeoutDefault = SchedCliType.timeoutDefault / 60

maxLogFilesMin = 0
maxLogFilesMax = 10000

jobsInProgressMin = 1
jobsInProgressMax = 4

# pkgdeps: import CliPlugin.CliCli
eosCliShell = os.getenv( 'CLI_SCHED_EOS_CLISHELL', "/usr/bin/CliShell" )
showTechJobName = os.getenv( 'CLI_SCHED_DEFAULT_JOBNAME', "tech-support" )
showTechCliCommand = os.getenv( 'CLI_SCHED_DEFAULT_COMMAND', "show tech-support" )
showTechIntervalDefault = 60
showTechMaxLogFilesDefault = 100
showTechVerboseDefault = False if 'CLI_SCHED_DEFAULT_JOBNAME' not in os.environ \
                         else True

scheduleNow = 0
scheduleNowStr = "now"

scheduleOnce = 0
scheduleOnceStr = "once"

gzStart = -22
gzEnd = -7
regStart = -19
regEnd = -4
DATEFMT_ERROR = { "INVALID": -1, "PAST": -2 }

logPrefixDefault = "flash:schedule/"

SYS_CLI_SCHEDULER_JOB_COMPLETED = None
Logging.logD(
   id="SYS_CLI_SCHEDULER_JOB_COMPLETED",
   severity=Logging.logDebug,
   format="The scheduled CLI execution job \'%s\' completed successfully.%s",
   explanation="A scheduled CLI command successfully executed. The output "\
      "of that command is stored in a log file if max-log-files > 0. The default "\
      "log file location is flash:/schedule/..., but may be in another location "\
      "if the \'loglocation\' option was specified.",
   recommendedAction=Logging.NO_ACTION_REQUIRED )

SYS_CLI_SCHEDULER_ABORT = None
Logging.logD(
   id="SYS_CLI_SCHEDULER_ABORT",
   severity=Logging.logWarning,
   format="Execution of scheduled CLI execution job \'%s\' was aborted due "\
          "to an error: %s",
   explanation="Scheduled execution of a CLI command was aborted. The output "\
               "of that command has been stored in a log file if "\
               "max-log-files > 0. The default log file location is flash, "\
               "but may be in another location if the \'loglocation\' option was "\
               "specified.",
   recommendedAction="Please try to execute the CLI command interactively to "\
         "make sure it works fine." )

SYS_CLI_SCHEDULER_FILESYSTEM_FULL = None
Logging.logD(
   id="SYS_CLI_SCHEDULER_FILESYSTEM_FULL",
   severity=Logging.logWarning,
   format="Execution of scheduled CLI execution job \'%s\' was aborted due "\
         "to target filesystem being full",
   explanation="Scheduled execution of a CLI command was aborted due to lack "\
         "of space in target filesystem.",
   recommendedAction="Please delete unused files to free up space." )

SYS_CLI_SCHEDULER_SKIP = None
Logging.logD(
   id="SYS_CLI_SCHEDULER_SKIP",
   severity=Logging.logNotice,
   format="Execution of scheduled CLI execution job \'%s\' was skipped",
   explanation="Scheduled execution of a CLI command was skipped "\
         "since previous execution for this job is yet to complete.",
   recommendedAction="This may be due to CLI command taking long to complete. "\
         "Please try to increase execution interval to a larger value." )

SYS_MEMORY_EXHAUSTION_CLI_SCHEDULER_DISABLED = None
Logging.logD(
   id="SYS_MEMORY_EXHAUSTION_CLI_SCHEDULER_DISABLED",
   severity=Logging.logNotice,
   format="CliScheduler is disabled.  All subsequent scheduled CLI execution jobs "\
         "will be skipped.",
   explanation="CliScheduler is disabled due to memory exhaustion on the system.",
   recommendedAction="Please try to disable features, then run "\
         "'reset system memory exhaustion'." ) 

SYS_CLI_SCHEDULER_DISABLED_SKIP = None
Logging.logD(
   id="SYS_CLI_SCHEDULER_DISABLED_SKIP",
   severity=Logging.logNotice,
   format="Execution of scheduled CLI execution job \'%s\' was skipped because "\
         "CliScheduler is currently disabled.",
   explanation="Scheduled execution of a CLI command was skipped "\
         "since CliScheduler is currently disabled.",
   recommendedAction="See the previous log why CliScheduler is disabled." )

SYS_CLI_SCHEDULER_ENABLED = None
Logging.logD(
   id="SYS_CLI_SCHEDULER_ENABLED",
   severity=Logging.logNotice,
   format="CliScheduler is enabled, continuing its execution of scheduled CLI jobs.",
   explanation="CliScheduler is enabled, resuming its execution of scheduled "\
         "CLI jobs.",
   recommendedAction=Logging.NO_ACTION_REQUIRED )

def replaceToken( tokenString ):
   ''' This function takes a string as input and replaces the following
   tokens with their respective values :
   1)  %h           :       Hostname until the first '.' 
   2)  %H           :       Hostname
   3)  %D           :       Date and Time ( YYYY-MM-DD.HH-MM )
   4)  %D{format}   :       Date and Time using strftime() utility'''

   fqdnHostName = gethostname()
   localtime = time.localtime()

   # The following block searches for '%D{formatString}' type of tokens and 
   # replaces them by the date-time in the desired format. 
   regexObj = re.compile( '(%D{[^}]*})' )
   match = regexObj.search( tokenString )
   while match:
      matchedToken = match.group()
      formatString = matchedToken[ 3:-1 ]
      formatTime = time.strftime( formatString, localtime )
      tokenString = tokenString.replace( matchedToken, formatTime )
      match = regexObj.search( tokenString )

   # Replace %h and %H
   hostname = fqdnHostName.split( ".", 1 )[ 0 ]
   tokenString = tokenString.replace( '%h', hostname )
   tokenString = tokenString.replace( '%H', fqdnHostName )

   # Replace %D
   formatString = '%Y-%m-%d.%H%M'
   tokenString = tokenString.replace( '%D',
   time.strftime( formatString, localtime ) )

   return tokenString

def extractTsFromFilename( fname ):
   '''
   Filename will either be 
      <jobName>_YYYY-MM-DD.HHMM.log if its yet to be gziped or
      <jobName>_YYYY-MM-DD.HHMM.log.gz
   e.g. tech-support_2011-02-28.0027.log or
        tech-support_2011-02-28.0027.log.gz
   If filename is invalid return None
   '''
   try:
      if fname.endswith(".gz"):
         ts = fname[ gzStart:gzEnd ]
      else:
         ts = fname[ regStart:regEnd ]
   except IndexError:
      return None
   try:
      return time.mktime( time.strptime( ts, "%Y-%m-%d.%H%M" ) )
   except ValueError:
      return None

def extractAtFromDateTime( dateTime ):
   if dateTime == scheduleNow:
      return scheduleNowStr
   else:
      atTime = time.localtime( dateTime )
      return "%02d:%02d:%02d %02d/%02d/%04d" % ( atTime.tm_hour, atTime.tm_min,
            atTime.tm_sec, atTime.tm_mon, atTime.tm_mday, atTime.tm_year )

# Returns the count of "Job under progress" from "show schedule summary" output      
def jobUnderProgressCountInSummary( summary ):
   return len( re.findall( 'under.+?progress', summary, flags=re.DOTALL ) )

# Returns the number of seconds since epoch or 0 (scheduleNow) or -1 (on error)
# Input format is {"now" | 0 | [hh, mm, ss] | [hh, mm, ss, MM, dd, yyyy]}
def createDateTimefromAt( at, interval ):   
   result = array( 'd' )
   if at == scheduleNowStr:
      result.append( scheduleNow )
   elif at == scheduleNow:
      result.append( at )
   else:
      strAt = " ".join( map( str, at ) )
      try:
         if len( at ) == 3:
            curTime = datetime.now()
            # Check if we have already crossed that time for today
            if ( curTime.hour > at[ 0 ] or ( curTime.hour == at[ 0 ] and
                                             ( curTime.minute > at[ 1 ] or
                                               ( curTime.minute == at[ 1 ] and
                                                 curTime.second > at[ 2 ] ) ) ) ):
               # Schedule it on the next day
               curTime = curTime + timedelta( days=1 )
            strAt = strAt + time.strftime( " %m %d %Y", curTime.timetuple() )
         epochSched = time.mktime( time.strptime( strAt, "%H %M %S %m %d %Y" ) )
         epochCur = time.time()
         result.append( epochSched )
         if epochCur > epochSched and interval:
            nextSched = datetime.strptime( strAt, "%H %M %S %m %d %Y" )
            nextSched = nextSched + timedelta( minutes = interval )
            now = datetime.now()
            if nextSched < now:
               diff = now - nextSched
               toAdd = divmod( diff.total_seconds(), interval * 60 )
               nextSched = nextSched + \
                           timedelta( minutes = toAdd[0] * interval )
               if toAdd[1]:
                  nextSched = nextSched + timedelta( minutes = interval )
            result.append( time.mktime( nextSched.timetuple() ) )
      except ValueError:
         result.append( DATEFMT_ERROR[ "INVALID" ] )
   return result

def inPast( epoch ):
   # scheduleNow (0) should not be treated as past
   return ( epoch != scheduleNow ) and ( time.time() > epoch )

class CliSchedLogfileList( object ):
   def __init__( self, root ):
      self.root = root
      self.fileList = []
      self.refresh()

   def refresh( self ):
      self.fileList = []
      for root, _, files in os.walk( self.root ):
         self.fileList += [ os.path.join( root, f ) for f in files \
                          if extractTsFromFilename( f ) ]
      self.fileList.sort( key=extractTsFromFilename )

   def deleteFiles( self, flist ):
      for fname in flist:
         os.unlink( fname )
      self.refresh()

   def __getitem__( self, idx ):
      if type( idx ) is int:
         return self.fileList[ idx ]

      if type( idx ) is slice:
         return self.fileList[ idx.start:idx.stop ]

class CliSchedLogMgr:
   def __init__( self, root, maxLogFiles ):
      self.root = root
      self.maxLogFiles = maxLogFiles
      self.fileset = CliSchedLogfileList( self.root )

   def getSnapshots( self ):
      return self.fileset[:]

   def rotateSnapshots( self ):
      files = self.fileset[:]
      files.reverse()
      files = files[ self.maxLogFiles: ]
      self.fileset.deleteFiles( files )

