# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import CliSave, Tac
from CliSavePlugin.ServiceCliSave import serviceConfigCmdSeqName
from CliSavePlugin.IntfCliSave import IntfConfigMode
from XcvrLib import mediaTypeEnumToClis
from CliMode.Layer1Monitor import Layer1MonitorBaseMode
import Toggles.XcvrToggleLib
IntfConfigMode.addCommandSequence( 'Xcvr.config' )

sfpHighPowerEnable = Toggles.XcvrToggleLib.toggleSfpHighPowerEnabled()

class Layer1MonitorConfigMode( Layer1MonitorBaseMode, CliSave.Mode ):
   def __init__( self, param ):
      Layer1MonitorBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( Layer1MonitorConfigMode )
Layer1MonitorConfigMode.addCommandSequence( 'Layer1Monitor.config' )

@CliSave.saver( 'Xcvr::Xgc', 'hardware/xcvr/xgc' )
def saveXcvrXgcConfig( xgc, root, sysdbRoot, options ):
   cmds = root[ serviceConfigCmdSeqName ]

   # NOTE: we ignore "options.saveAll" and "options.saveAllDetail"
   #       here, because the "service unsupported-transceiver" command
   #       is hidden.
   #
   #       We DO make sure to use the sanitized version of the key
   #       when the user requests a sanitized config to be shown. This
   #       is implemented quite nicely by CliSave.sanitizedOutput().
   if xgc.licensee:
      # We avoid formatting the key using "hex( xgc.key )" because
      # "hex()" generates a leading "0x", which I want to avoid in
      # order to simplify the CliParser.PatternRule ever so slightly.
      # This lets us avoid one extra test case in the CliSaveTest.
      cmds.addCommand( 'service unsupported-transceiver %s %s' % \
                          ( xgc.licensee,
                            CliSave.sanitizedOutput( options, "%x" % xgc.key ) ) )


   if xgc.ethComplianceBit7IsPsm4:
      cmds.addCommand( 'transceiver qsfp ethernet-compliance bit7-is-psm4' )

   pm = Tac.Value( 'Xcvr::Sff8436PerformanceMonitoring' )
   if xgc.performanceMonitoringPeriod.valueInSeconds != pm.defaultPMPeriod:
      # Convert period to the orginally configured units
      val = xgc.performanceMonitoringPeriod.valueInSeconds
      if( xgc.performanceMonitoringPeriod.unit == 'm' ):
         val = val / 60         # minutes
      elif( xgc.performanceMonitoringPeriod.unit == 'h' ):
         val = val / ( 60 * 60 )        # hours
      elif( xgc.performanceMonitoringPeriod.unit == 'd' ):
         val = val / ( 60 * 60 * 24 )   # days
      cmds.addCommand( 'performance-monitoring period %d%s' % 
            ( val, xgc.performanceMonitoringPeriod.unit ) )
   elif options.saveAll:
      cmds.addCommand( 'no performance-monitoring period' )
  
   if xgc.performanceMonitoringGlobal:
      cmds.addCommand( 'performance-monitoring transceiver default' )
   elif options.saveAll:
      cmds.addCommand( 'no performance-monitoring transceiver default' )
   
   defaultTargetTemp = Tac.Value( 'Units::Celsius' )
   if xgc.targetTemperature100GEDwdm2 != defaultTargetTemp.value:
      cmds.addCommand( 'transceiver 100GE-DWDM2 temperature cooling-target %d ' \
            'celsius' % int( xgc.targetTemperature100GEDwdm2 ) )
   elif options.saveAll:
      cmds.addCommand( 'no transceiver 100GE-DWDM2 temperature cooling-target' )

   for config in xgc.tuningConfig.values():
      intf = config.name
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intf )
      cmds = mode[ 'Xcvr.config' ]
      tuningConstants = Tac.Value( 'Xcvr::TuningConstants' )
      if config.frequency == 0:
         channelStr = 'transceiver channel %d' % config.channel
         gridStr = ''
         if config.grid != tuningConstants.defaultGrid:
            if config.grid % 1e3:
               gridStr = ' grid-spacing %.2f' % ( float( config.grid ) / 1e3 )
            else:
               gridStr = ' grid-spacing %d' % ( config.grid / 1e3 )
         cmds.addCommand( channelStr + gridStr )
      else:
         cmds.addCommand( 'transceiver frequency %.3f%s' %
                          ( ( float( config.frequency ) / 1e3 ),
                            ' ghz' if config.displayUnits else '' ) )
   for rxConfig in xgc.rxTuningConfig.values():
      intf = rxConfig.name
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intf )
      cmds = mode[ 'Xcvr.config' ]
      if rxConfig.frequency:
         cmds.addCommand( 'transceiver receiver frequency %.3f ghz' %
                          ( float( rxConfig.frequency ) / 1e3 ) )
      if rxConfig.fineFrequency:
         cmds.addCommand( 'transceiver receiver frequency fine %.3f ghz' %
                          ( float( rxConfig.fineFrequency ) / 1e3 ) )

   for config in xgc.txPowerConfig.itervalues():
      intf = config.name
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intf )
      cmds = mode[ 'Xcvr.config' ]
      powerConstants = Tac.Value( 'Xcvr::PowerConstants' )
      if config.signalPower != powerConstants.defaultTxPower:
         cmds.addCommand( 'transceiver transmitter signal-power %.2f' 
                          % config.signalPower )
      elif options.saveAll:
         cmds.addCommand( 'no transceiver transmitter signal-power' )

   for config in xgc.rxPowerConfig.itervalues():
      intf = config.name
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intf )
      cmds = mode[ 'Xcvr.config' ]
      powerConstants = Tac.Value( 'Xcvr::PowerConstants' )
      if config.signalPower != powerConstants.defaultRxPower:
         cmds.addCommand( 'transceiver receiver signal-power %.2f' 
                          % config.signalPower )
      elif options.saveAll:
         cmds.addCommand( 'no transceiver receiver signal-power' )

   for intf in xgc.txDisabled:
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intf )
      cmds = mode[ 'Xcvr.config' ]
      cmds.addCommand( 'transceiver transmitter disabled' )

   for config in xgc.performanceMonitoringConfig.itervalues():
      intf = config.name
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intf )
      cmds = mode[ 'Xcvr.config' ]
      if config.performanceMonitoringEnabled:
         cmds.addCommand( 'performance-monitoring transceiver' )
      elif options.saveAll:
         cmds.addCommand( 'no performance-monitoring transceiver' )

   if xgc.domThresholdOverrideEnabled:
      cmds.addCommand( 'transceiver dom-threshold file %s' %
                       xgc.domThresholdOverrideSrc )
   elif options.saveAll:
      cmds.addCommand( 'no transceiver dom-threshold file' )

   if xgc.domThresholdLogging:
      mode = root[ Layer1MonitorConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'Layer1Monitor.config' ]
      cmds.addCommand( 'logging transceiver' )


@CliSave.saver( 'Tac::Dir', 'hardware/xcvr/cli/config/slice' )
def saveXcvrConfigCli( baseDir, root, sysdbRoot, options ):
   for xcvrConfigCliDir in baseDir.values():
      for xcvrConfigCli in xcvrConfigCliDir.xcvrConfigCli.values():
         intf = xcvrConfigCli.name
         mode = root[ IntfConfigMode ].getOrCreateModeInstance( intf )
         cmds = mode[ 'Xcvr.config' ]
         if xcvrConfigCli.overrideMediaType != "xcvrUnknown":
            mediaType = mediaTypeEnumToClis[ xcvrConfigCli.overrideMediaType ]
            cmds.addCommand( 'transceiver media override %s' % mediaType )
         elif options.saveAll:
            cmds.addCommand( 'no transceiver media override' )

         if xcvrConfigCli.forceModuleRemoved:
            cmds.addCommand( 'transceiver diag simulate removed' )
         elif options.saveAll:
            cmds.addCommand( 'no transceiver diag simulate removed' )

         if sfpHighPowerEnable:
            if xcvrConfigCli.preventSfpHighPowerEnable:
               cmds.addCommand( 'transceiver sfp power low')
            elif options.saveAll:
               cmds.addCommand( 'no transceiver sfp power' )

         if xcvrConfigCli.forceQsfpManagementMode:
            cmds.addCommand( 'transceiver diag management' )
         elif options.saveAll:
            cmds.addCommand( 'no transceiver diag management' )

         for laneId in xcvrConfigCli.slotTuningParams.keys():
            cliTuningParam = ''
            laneTuningParam = xcvrConfigCli.slotTuningParams[ laneId ]
            if laneTuningParam.txInputEqualization.valid:
               cliTuningParam += ' tx-input-equalization ' + \
                  str( laneTuningParam.txInputEqualization.val )
            elif laneTuningParam.txInputEqualization.moduleDefault:
               cliTuningParam += ' tx-input-equalization module-default'
            elif laneTuningParam.txInputAdaptiveEqEnable.valid:
               cliTuningParam += ' tx-input-equalization auto'

            if laneTuningParam.rxOutputPreEmphasis.valid:
               cliTuningParam += ' rx-output-pre-emphasis ' + \
                  str( laneTuningParam.rxOutputPreEmphasis.val )
            elif laneTuningParam.rxOutputPreEmphasis.moduleDefault:
               cliTuningParam += ' rx-output-pre-emphasis module-default'

            if laneTuningParam.rxOutputPostEmphasis.valid:
               cliTuningParam += ' rx-output-post-emphasis ' + \
                  str( laneTuningParam.rxOutputPostEmphasis.val )

            if laneTuningParam.rxOutputAmplitude.valid:
               cliTuningParam += ' rx-output-amplitude ' + \
                  str( laneTuningParam.rxOutputAmplitude.val ) 
            elif laneTuningParam.rxOutputAmplitude.moduleDefault:
               cliTuningParam += ' rx-output-amplitude module-default'

            if cliTuningParam:
               cmds.addCommand( 'transceiver electrical lane %s%s' % (
                                laneId + 1, cliTuningParam ) )
            elif options.saveAll:
               cmds.addCommand( 'no transceiver electrical lane %s' % (
                                laneId + 1 ) )

         if xcvrConfigCli.overrideTuningParams:
            cmds.addCommand( 'transceiver electrical tuning override' )
         elif options.saveAll:
            cmds.addCommand( 'no transceiver electrical tuning override' )
