#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import Tac
import CliSave
import CliSavePlugin.IntfCliSave as IntfCliSave
import Toggles.McastVpnLibToggleLib

IntfCliSave.IntfConfigMode.addCommandSequence( 'McastVpn.VxlanIntf',
                                               after=[ 'VxlanIntf.config' ] )

def getVlanFloodCmds( intfConfig ):
   vlanFloodCmds = []
   for vlan in sorted( intfConfig.vlanToFloodGroup ):
      configEntry = intfConfig.vlanToFloodGroup[ vlan ]
      vlanFloodCmds.append( 'vxlan vlan %d flood group %s'
                            % ( vlan, configEntry.underlayGroup.stringValue ) )
   return vlanFloodCmds

def getVlanMulticastCmds( intfConfig ):
   vlanMulticastCmds = []
   for vlan in sorted( intfConfig.vlanToMulticastGroup ):
      configEntry = intfConfig.vlanToMulticastGroup[ vlan ]
      vlanMulticastCmds.append( 'vxlan vlan %d multicast group %s'
                            % ( vlan, configEntry.underlayGroup.stringValue ) )
   return vlanMulticastCmds

def getVrfMulticastCmds( intfConfig ):
   vrfMulticastCmds = []
   for vrf in sorted( intfConfig.vrfToMulticastGroup ):
      configEntry = intfConfig.vrfToMulticastGroup[ vrf ]
      vrfMulticastCmds.append( 'vxlan vrf %s multicast group %s'
                            % ( vrf, configEntry.underlayGroup.stringValue ) )
   return vrfMulticastCmds

def getUnderlayRouteTypeCmds( intfConfig ):
   underlayRouteTypeCmds = []

   if not Toggles.McastVpnLibToggleLib.toggleMcastVpnUnderlayAsmEnabled():
      return underlayRouteTypeCmds

   UnderlayRouteType = \
         Tac.Type( "Routing::Multicast::UnderlayRouteType::RouteType" )
   if intfConfig.mcastUnderlayRouteType == UnderlayRouteType.pimasm:
      underlayRouteTypeCmds.append( 'vxlan multicast protocol pim asm' )
   if intfConfig.floodUnderlayRouteType == UnderlayRouteType.pimasm:
      underlayRouteTypeCmds.append( 'vxlan flood protocol pim asm' )
   return underlayRouteTypeCmds

def getRemoteVniToVrfCmds( intfConfig ):
   remoteVniToVrfCmds = []
   for vni in sorted( intfConfig.remoteVniToVrf ):
      remoteVniToVrfCmds.append( 'vxlan remote vni %d vrf %s'
                                 % ( vni, intfConfig.remoteVniToVrf[ vni ] ) )
   return remoteVniToVrfCmds

@CliSave.saver( 'Routing::Multicast::IpTunnelGroupConfig',
                'multicast/tunnel/group/config',
                requireMounts=( 'interface/config/eth/vxlan', ) )
def saveMcastVpnConfig( ipTunnelGroupConfig, root, sysdbRoot, options,
                        requireMounts ):
   vtiConfigDir = requireMounts[ 'interface/config/eth/vxlan' ]
   for vti in vtiConfigDir.vtiConfig:
      intfConfig = ipTunnelGroupConfig.get( vti )
      if not intfConfig or intfConfig.isConfigEmpty:
         continue
      mcastVpnCmds = getVlanFloodCmds( intfConfig ) + \
                     getVlanMulticastCmds( intfConfig ) + \
                     getVrfMulticastCmds( intfConfig ) + \
                     getUnderlayRouteTypeCmds( intfConfig ) + \
                     getRemoteVniToVrfCmds( intfConfig )
      mode = root[ IntfCliSave.IntfConfigMode ].getOrCreateModeInstance( vti )
      cmds = mode[ 'McastVpn.VxlanIntf' ]
      for cmd in mcastVpnCmds:
         cmds.addCommand( cmd )
