# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac, CliSave
import Arnet
from CliSavePlugin.Controllerdb import ( CvxConfigMode,
                                         controllerConfigPath,
                                         getClusterName )
from VxlanControllerAgent import vxlanControllerConfigSysdbPath
from CliMode.Vxlan import VxlanMode
import Toggles.VxlanControllerToggleLib
 
class VxlanConfigSaveMode( VxlanMode, CliSave.Mode ):
   def __init__( self, param ):
      VxlanMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CvxConfigMode.addChildMode( VxlanConfigSaveMode,
                            after=[ 'cvx' ] )
VxlanConfigSaveMode.addCommandSequence( 'VxlanControllerCVX.config' )
CliSave.GlobalConfigMode.addCommandSequence( 'VxlanControllerGlobal.config',
                                             after=[ CvxConfigMode ] )

def addCommandsToVxlanMode( root, cmds, cvxClusterName ):
   if cmds:
      mode = root[ CvxConfigMode ].getOrCreateModeInstance(
         CvxConfigMode.modeName( cvxClusterName ) )
      vxlanMode = mode[ VxlanConfigSaveMode ]
      vxlanMode = vxlanMode.getOrCreateModeInstance( 'cvx-vxlan' )
      vxlanMode = vxlanMode[ 'VxlanControllerCVX.config' ]
      for cmd in cmds:
         vxlanMode.addCommand( cmd )
 
def saveVxlcConfigVniNotation( entity, root, sysdbRoot, options ):
   cmds = root[ 'VxlanControllerGlobal.config' ]
   if entity.vniInDottedNotation:
      cmds.addCommand( 'vxlan vni notation dotted' )
   elif options.saveAll:
      cmds.addCommand( 'no vxlan vni notation dotted' )

def saveVxlcConfigEnable( entity, root, sysdbRoot, options, cvxClusterName ):
   cmds = []
   if entity.enable or options.saveAll:
      cmds.append( "%sshutdown" % ( entity.enable and "no " or "" ) )

   addCommandsToVxlanMode( root, cmds, cvxClusterName )

def saveVxlcConfigDataPathLearning( entity, root, sysdbRoot, options,
                                    cvxClusterName ):
   cmds = []
   if entity.dataPathLearning:
      cmds.append( 'vtep mac-learning data-plane' )
   elif options.saveAll:
      cmds.append( 'vtep mac-learning control-plane' )
 
   addCommandsToVxlanMode( root, cmds, cvxClusterName )
 
def saveVxlcConfigPortDot1qVniMapping( entity, root, sysdbRoot, options,
                                       cvxClusterName ):
   cmds = []
   if entity.portDot1qVniMapping:
      cmds.append( 'vtep vni mapping port-dot1q-tag' )
   elif options.saveAll:
      cmds.append( 'vtep vni mapping vlan' )

   addCommandsToVxlanMode( root, cmds, cvxClusterName )

def saveVxlcConfigRestartSyncPeriod( entity, root, sysdbRoot, options,
                                     cvxClusterName ):
   cmds = []
   if entity.restartSyncPeriod != entity.defaultRestartSyncPeriod:
      cmd = ( 'resync-period ' + str( entity.restartSyncPeriod ) if
              entity.restartSyncPeriod else 'no resync-period' )
      cmds.append( cmd )
   elif options.saveAll:
      cmds.append( 'resync-period %d' % entity.defaultRestartSyncPeriod )
 
   addCommandsToVxlanMode( root, cmds, cvxClusterName )
   
def saveVxlcConfigBgpEvpnRedist( entity, root, sysdbRoot, options,
                                     cvxClusterName ):
   cmds = []
   if entity.bgpEvpnRedist:
      cmds.append( "redistribute bgp evpn vxlan" )
   elif options.saveAll:
      cmds.append( "redistribute bgp evpn vxlan" )

   addCommandsToVxlanMode( root, cmds, cvxClusterName )

def saveVxlcConfigMulticastMembershipList( entity, root, sysdbRoot, options,
                                       cvxClusterName ):
   cmds = []
   if entity.multicastMembershipList:
      cmds.append( 'multicast membership-list' )
   elif options.saveAll:
      cmds.append( 'no multicast membership-list' )

   addCommandsToVxlanMode( root, cmds, cvxClusterName )

def saveVxlcConfigDefaultFloodList( entity, root, sysdbRoot, options,
                                    cvxClusterName ):
   cmds = []
   if entity.floodVtepList:
      vteps = ' '.join( sorted( entity.floodVtepList.iterkeys(), 
                                key=lambda x: Arnet.IpAddress( x ).value ) )
      cmd = 'flood vtep %s' % vteps
      cmds.append( cmd )
   elif options.saveAll:
      cmd = 'no flood vtep'
      cmds.append( cmd )

   addCommandsToVxlanMode( root, cmds, cvxClusterName )
   

def saveVxlcConfigVniFloodList( entity, root, sysdbRoot, options, cvxClusterName ):
   cmds = []
   macAddr = Tac.Type( "Arnet::EthAddr" ).ethAddrZero
   if entity.vniToVtepList:
      for ( vni, vtepList ) in sorted( entity.vniToVtepList.iteritems(),
                                key=lambda (k,_): k):
         bumMacVtepList = vtepList.remoteVtep[ macAddr ]
         vteps = ' '.join(sorted( bumMacVtepList.vtepIpFloodList.iterkeys(),
                                  key=lambda x: Arnet.IpAddress( x ).value ) )
         cmds.append( 'vni %s flood vtep %s' % ( vni, vteps ) )
   elif options.saveAll:
      cmds.append( 'no vni flood vtep' )

   addCommandsToVxlanMode( root, cmds, cvxClusterName )

def saveArpReplyRelayVtepList( entity, root, options, cvxClusterName ):
   cmds = []
   cmd = "arp reply relay vtep "
   if entity.arpReplyRelayVtep:
      cmd += " ".join( [ v.stringValue \
                         for v in sorted( entity.arpReplyRelayVtep.keys(),
                                          key=lambda x: x.sortKey ) ] )
      cmds.append( cmd )
   elif options.saveAll:
      cmds.append( "no " + cmd )

   addCommandsToVxlanMode( root, cmds, cvxClusterName )

# pylint: disable-msg=C0322
@CliSave.saver( 'VxlanController::Config', vxlanControllerConfigSysdbPath,
                requireMounts=( controllerConfigPath, ) )
def saveVxlanControllerConfig( entity, root, sysdbRoot, options, requireMounts ):
   cvxClusterName = getClusterName( requireMounts[ controllerConfigPath ] )
   saveVxlcConfigEnable( entity, root, sysdbRoot, options, cvxClusterName )
   saveVxlcConfigDataPathLearning( entity, root, sysdbRoot, options, cvxClusterName )
   saveVxlcConfigPortDot1qVniMapping( entity, root, sysdbRoot, options,
                                      cvxClusterName )
   saveVxlcConfigVniNotation( entity, root, sysdbRoot, options )
   saveVxlcConfigRestartSyncPeriod( entity, root, sysdbRoot, options,
                                    cvxClusterName )
   saveVxlcConfigBgpEvpnRedist( entity, root, sysdbRoot, options, cvxClusterName )
   if Toggles.VxlanControllerToggleLib.toggleVCSMulticastKnobEnabled():
      saveVxlcConfigMulticastMembershipList( entity, root, sysdbRoot, options,
                                          cvxClusterName )
   saveVxlcConfigDefaultFloodList( entity, root, sysdbRoot, options, cvxClusterName )
   saveVxlcConfigVniFloodList( entity, root, sysdbRoot, options, cvxClusterName )
   saveArpReplyRelayVtepList( entity, root, options, cvxClusterName )

