#!/usr/bin/env python
# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac, CliSave, IntfCliSave, Arnet
import Toggles.VxlanToggleLib
import collections
from IntfCliSave import IntfConfigMode
from Ethernet import convertMacAddrCanonicalToDisplay
from VxlanVniLib import VniFormat

IntfConfigMode.addCommandSequence( 'VxlanIntf.config' )
CliSave.GlobalConfigMode.addCommandSequence( 'VxlanGlobal.config' )
CliSave.GlobalConfigMode.addCommandSequence( 'Vxlan.staticMac' )

#-------------------------------------------------------------------------------
# Saves the state of an Interface::VtiConfig object.
#-------------------------------------------------------------------------------
# pylint: disable-msg=C0322
# XXX Workaround for a pylint bug with decorators.
@CliSave.saver( 'Vxlan::VtiConfig', 'interface/config/eth/vxlan',
                attrName = 'vtiConfig',
                requireMounts = ( 'vxlancontroller/config',
                                  'bridging/hwcapabilities',
                                  'interface/status/all',
                                  'vxlan/config',
                                  'interface/config/eth/vxlan' ) )
def saveVtiConfig( entity, root, sysdbRoot, options,
                   requireMounts ):
   # Cleanup and bail out on inconsistent state where only one of VxlanConfig and
   # VtiConfig exists
   vxlanConfigDir = requireMounts[ 'vxlan/config' ]
   if entity.intfId not in vxlanConfigDir.vxlanConfig:
      vtiConfigDir = requireMounts[ 'interface/config/eth/vxlan' ]
      del vtiConfigDir.vtiConfig[ entity.intfId ]
      return

   # handle baseclass (Arnet::IntfConfig) first
   IntfCliSave.saveIntfConfig( entity, root, sysdbRoot, options,
                               requireMounts )

   saveAll = options.saveAll

   mode = root[ IntfConfigMode ].getOrCreateModeInstance( entity.intfId )
   cmds = mode[ 'VxlanIntf.config' ]

   if entity.floodMcastGrp != '0.0.0.0':
      cmds.addCommand( 'vxlan multicast-group %s' %
                       entity.floodMcastGrp )
   elif saveAll:
      cmds.addCommand( 'no vxlan multicast-group' )

   if entity.srcIpIntf != '':
      cmds.addCommand( 'vxlan source-interface %s' %
                       entity.srcIpIntf )
   elif saveAll:
      cmds.addCommand( 'no vxlan source-interface' )

   if entity.cliVccImportVlans == '':
      cmds.addCommand( 'vxlan controller-client import vlan none' )
   elif entity.cliVccImportVlans != '1-4094':
      cmds.addCommand( 'vxlan controller-client import vlan %s' %
                       entity.cliVccImportVlans )
   elif saveAll:
      cmds.addCommand( 'vxlan controller-client import vlan 1-4094' )

   if entity.controllerClientMode:
      cmds.addCommand( 'vxlan controller-client' )
   elif saveAll:
      cmds.addCommand( 'no vxlan controller-client' )

   if entity.arpReplyRelay:
      cmds.addCommand( 'arp reply relay' )
   elif saveAll:
      cmds.addCommand( 'no arp reply relay' )

   if entity.mlagSharedRouterMacConfig == 'autoGenerated':
      cmds.addCommand( 'vxlan virtual-router encapsulation mac-address '
                       'mlag-system-id' )
   elif entity.mlagSharedRouterMacConfig == 'explicitConfig':
      cmds.addCommand( 'vxlan virtual-router encapsulation mac-address %s' %
                       entity.mlagSharedRouterMacAddr )
   elif saveAll:
      cmds.addCommand( 'no vxlan virtual-router encapsulation mac-address' )

   if entity.arpLocalAddress:
      cmds.addCommand( 'arp source ip address local' )
   elif saveAll:
      cmds.addCommand( 'no arp source ip address local' )

   #BUG48350
   #if entity.ttl != entity.defaultTtl:
   #   cmds.addCommand( 'vxlan ttl %d' % entity.ttl )
   #elif saveAll:
   #   cmds.addCommand( 'no vxlan ttl' )

   # if entity.udpPort != entity.vxlanWellKnownPort:
   #    cmds.addCommand( 'vxlan udp-port %d' % entity.udpPort )
   # elif saveAll:
   #    cmds.addCommand( 'no vxlan udp-port' )

   # As per discussion, we decided to generate udp-port running config always 
   # till a consensus reached for default port in the industry.
   cmds.addCommand( 'vxlan udp-port %d' % entity.udpPort )

   if entity.srcPortRange != Tac.Value( "Vxlan::VxlanSrcPortRange" ):
      cmds.addCommand( 'vxlan udp-port source offset %d length %d' % 
            ( entity.srcPortRange.offset, entity.srcPortRange.length ) )
   elif saveAll:
      cmds.addCommand( 'no vxlan udp-port source' )

   # Generate secure udp-port running config only if any vtep is configured
   # as secure or if the secure udp-port is configured.
   if entity.vtepToSecProfile or \
         entity.secUdpPort != entity.vxlanSecWellKnownPort:
      cmds.addCommand( 'vxlan security udp-port %d' % entity.secUdpPort )

   if entity.use32BitVni:
      cmds.addCommand( 'vxlan header vni 32-bit' )
   elif saveAll:
      cmds.addCommand( 'no vxlan header vni 32-bit' )

   if entity.vtepSourcePruningAll:
      cmds.addCommand( 'vxlan bridging vtep-to-vtep source-vtep tx disabled' )
   elif entity.vtepSetForSourcePruning:
      vteps = ' '.join( sorted( entity.vtepSetForSourcePruning.iterkeys(),
                                key=lambda x: Arnet.IpAddress( x ).value ) )
      cmd = 'vxlan bridging vtep-to-vtep source-vtep tx disabled %s' % vteps
      cmds.addCommand( cmd )
   elif entity.vtepToVtepBridging:
      cmds.addCommand( 'vxlan bridging vtep-to-vtep' )
      if saveAll:
         cmds.addCommand( 'no vxlan bridging vtep-to-vtep source-vtep tx disabled' )
   elif saveAll:
      cmds.addCommand( 'no vxlan bridging vtep-to-vtep' )
      cmds.addCommand( 'no vxlan bridging vtep-to-vtep source-vtep tx disabled' )

   if entity.floodLearnedAll:
      cmds.addCommand( 'vxlan flood vtep learned data-plane' )
   elif saveAll:
      cmds.addCommand( 'no vxlan flood vtep learned data-plane' )

   if entity.mcastRouting:
      cmds.addCommand( 'vxlan multicast routing ipv4' )
   elif saveAll:
      cmds.addCommand( 'no vxlan multicast routing ipv4' )

   vxlanEncapType = Tac.Type( "Vxlan::VxlanEncap" )
   brHwCap = requireMounts[ 'bridging/hwcapabilities' ]
   if Toggles.VxlanToggleLib.toggleVxlanEncapsulationTypeEnabled() and \
      brHwCap.vxlanEncapIpv6ConfigSupported:

      dualEncapEnabled = \
            Toggles.VxlanToggleLib.toggleVxlanEncapsulationTypeDualEnabled()
      if entity.vxlanEncap == vxlanEncapType.vxlanEncapIp6:
         cmds.addCommand( 'vxlan encapsulation ipv6' )
      elif dualEncapEnabled and entity.vxlanEncap == vxlanEncapType.vxlanEncapDual:
         cmds.addCommand( 'vxlan encapsulation ipv4 ipv6' )
      elif saveAll:
         cmds.addCommand( 'vxlan encapsulation ipv4' )

   vxlanCntrlConfig = requireMounts[ 'vxlancontroller/config' ]

   sortedVlans = sorted( entity.vlanToVniMap.iterkeys(),
                         key=lambda x: "%04d" % x )
   # Save VLAN-VNI map using range when new syntax is enabled
   if entity.parent.vlanVniRangeSyntax and sortedVlans:
      vlanStr, vniStr = "", ""
      prevVlan = startVlan = None
      for vlan in sortedVlans:
         vlan = int( vlan )
         # Skip first iteration
         if prevVlan is None:
            prevVlan = startVlan = vlan
            continue
         vni = entity.vlanToVniMap[ vlan ]
         prevVni = entity.vlanToVniMap[ prevVlan ]
         # VLANs and VNIs must be adjacent to collapse into range
         if vlan != prevVlan + 1 or vni != prevVni + 1:
            startVni = VniFormat( entity.vlanToVniMap[ startVlan ], 
                                  vxlanCntrlConfig.vniInDottedNotation )
            if startVlan == prevVlan:
               vlanStr += "%d," % startVlan
               vniStr += "%s," % startVni
            else:
               endVni = VniFormat( entity.vlanToVniMap[ prevVlan ], 
                                   vxlanCntrlConfig.vniInDottedNotation )
               vlanStr += "%d-%d," % ( startVlan, prevVlan )
               vniStr += "%s-%s," % ( startVni, endVni )
            startVlan = vlan
         prevVlan = vlan
      # Add final mapping
      startVni = VniFormat( entity.vlanToVniMap[ startVlan ], 
                            vxlanCntrlConfig.vniInDottedNotation )
      if prevVlan == startVlan:
         vlanStr += "%d" % startVlan
         vniStr += "%s" % startVni
      else:
         endVni = VniFormat( entity.vlanToVniMap[ prevVlan ], 
                             vxlanCntrlConfig.vniInDottedNotation )
         vlanStr += "%d-%d" % ( startVlan, prevVlan )
         vniStr += "%s-%s" % ( startVni, endVni )
      cmds.addCommand( 'vxlan vlan %s vni %s' % ( vlanStr, vniStr ) )
   else:
      for vlan in sortedVlans:
         vniStr = VniFormat( entity.vlanToVniMap[ vlan ],
               vxlanCntrlConfig.vniInDottedNotation )
         cmds.addCommand( 'vxlan vlan %s vni %s' %
               ( vlan, vniStr ) )

   if entity.mcastGrpDecap:
      grps = ' '.join( sorted( entity.mcastGrpDecap.iterkeys(),
                        key=lambda x: Arnet.IpAddress( x ).value ) )
      cmds.addCommand( 'vxlan multicast-group decap %s' % grps )
   elif saveAll:
      cmds.addCommand( 'no vxlan multicast-group decap' )
   vrfSet = set( entity.vrfToVniMap.iterkeys() )
   # vrfToDecapVniMap is reverse map of entity.alternateDecapVniToVrfMap
   vrfToDecapVniMap = collections.defaultdict( set )
   for vni,vrf in entity.alternateDecapVniToVrfMap.iteritems():
      vrfToDecapVniMap[ vrf ].add( vni )

   def displayDecapVniForVrf( vrf ):
      # Decap VNIs
      if vrf not in vrfToDecapVniMap:
         return
      dotN = vxlanCntrlConfig.vniInDottedNotation
      decapVniStr = ",".join( sorted( VniFormat( vni, dotN ).__str__()
                                           for vni in vrfToDecapVniMap[ vrf ] ) )
      if decapVniStr:
         cmds.addCommand( 'vxlan vrf %s vni decapsulation %s' %
                          ( vrf, decapVniStr ) )

   decapVrfs = set( entity.alternateDecapVniToVrfMap.values() )
   for vrf in sorted( vrfSet | decapVrfs ):
      if vrf in vrfSet:
         vniStr = VniFormat( entity.vrfToVniMap[ vrf ],
                                       vxlanCntrlConfig.vniInDottedNotation )
         cmds.addCommand( 'vxlan vrf %s vni %s' % ( vrf, vniStr ) )
      displayDecapVniForVrf( vrf )

   if entity.vtepAddrMask != '255.255.255.255':
      cmds.addCommand( 'vxlan vtep ipv4 address-mask %s' %
                       entity.vtepAddrMask )
   elif saveAll:
      cmds.addCommand( 'no vxlan vtep ipv4 address-mask' )

   if entity.mlagSrcIpIntf != '':
      cmds.addCommand( 'vxlan mlag source-interface %s' %
                       entity.mlagSrcIpIntf )
   elif saveAll:
      cmds.addCommand( 'no vxlan mlag source-interface' )
      
   if entity.varpVtepSrcIpIntf != '':
      cmds.addCommand( 'vxlan virtual-vtep local-interface %s' %
                       entity.varpVtepSrcIpIntf )
   elif saveAll:
      cmds.addCommand( 'no vxlan virtual-vtep local-interface' )

   if entity.decapFilterMode != 'filterEnabled':
      cmdStr = 'vxlan decapsulation filter'
      if entity.decapFilterMode == 'filterDisabled':
         cmdStr += ' disabled'
         cmds.addCommand( cmdStr )
      else:
         cmdStr += ' interface multiple-vrf disabled'
         if entity.decapFilterMode == 'filterRelaxedIntf':
            for intf in Arnet.sortIntf( entity.decapFilterIntf ):
               cmdStr += ( ' %s' % intf )
         cmds.addCommand( cmdStr )
   elif saveAll:
      cmds.addCommand( 'no vxlan decapsulation filter' )
   
   if Toggles.VxlanToggleLib.toggleVxlanSecEnabled():
      profileToVtepMap = dict()
      for ( vtep, profile ) in entity.vtepToSecProfile.iteritems():
         profileToVtepMap.setdefault( profile, list() ).append( vtep )
      for profile, vteps in sorted( profileToVtepMap.iteritems() ):
         secVteps = ' '.join( sorted( ip.v4Addr for ip in vteps ) )
         cmds.addCommand( 'vxlan vtep %s ip security profile %s' %
                           ( secVteps, profile ) )

   if Toggles.VxlanToggleLib.toggleRemoteTunnelPICEnabled():
      if entity.bfdEnabled and entity.intfId == 'Vxlan1':
         cmds.addCommand( 'bfd vtep evpn interval %d min-rx %d multiplier %d' %
                           ( entity.bfdIntervalParams.minTx,
                             entity.bfdIntervalParams.minRx,
                             entity.bfdIntervalParams.mult ) )
      elif saveAll:
         if entity.intfId == 'Vxlan1':
            cmds.addCommand( 'no bfd vtep evpn' )

@CliSave.saver( 'Vxlan::VxlanConfigDir', 'vxlan/config' )
def saveVxlanFdbConfig( entity, root, sysdbRoot, options ):
   cmds = root[ 'Vxlan.staticMac' ]
   configuredHosts = entity.fdbConfig.configuredHost
   sortedConfiguredHosts = sorted( configuredHosts.itervalues(),
                                   key=lambda x: "%04d" % x.macVlanPair.vlanId +
                                   x.macVlanPair.macAddr )
   for host in sortedConfiguredHosts:
      macAddr = convertMacAddrCanonicalToDisplay( host.macVlanPair.macAddr )
      cmds.addCommand(
         'mac address-table static %s vlan %s interface %s vtep %s' %
         ( macAddr, host.macVlanPair.vlanId, host.vtiIntfId, host.remoteVtepAddr ) )

@CliSave.saver( 'Vxlan::VxlanConfig', 'vxlan/config',
                attrName='vxlanConfig',
                requireMounts = ( 'interface/config/eth/vxlan',
                                  'vxlan/config' ) )
def saveVxlanConfig( entity, root, sysdbRoot, options, requireMounts ):
   # Cleanup and bail out on inconsistent state where only one of VxlanConfig and
   # VtiConfig exists
   vtiConfigDir = requireMounts[ 'interface/config/eth/vxlan' ]
   if entity.intfId not in vtiConfigDir.vtiConfig:
      vxlanConfigDir = requireMounts[ 'vxlan/config' ]
      del vxlanConfigDir.vxlanConfig[ entity.intfId ]
      return

   saveAll = options.saveAll
   mode = root[ IntfConfigMode ].getOrCreateModeInstance( entity.intfId )
   cmds = mode[ 'VxlanIntf.config' ]

   if entity.floodVtepList or entity.floodVtepList6:
      vteps = ' '.join( sorted( entity.floodVtepList.iterkeys(),
                                key=lambda x: Arnet.IpAddress( x ).value ) )
      vteps6 = sorted( Arnet.IpGenAddr( str( ip ) )
                         for ip in entity.floodVtepList6 )
      separator = ' ' if vteps else ''
      vteps += separator + ' '.join( str( ip ) for ip in vteps6 )
      cmd = 'vxlan flood vtep %s' % vteps
      cmd = cmd.strip()
      cmds.addCommand( cmd )
   elif saveAll:
      cmd = 'no vxlan flood vtep'
      cmds.addCommand( cmd )

   if entity.vlanToVtepList:
      for ( vlan, vl ) in sorted( entity.vlanToVtepList.iteritems(),
            key=lambda (k,_): k ):
         vteps = ' '.join( sorted( vl.remoteVtepAddr.iterkeys(),
                                   key=lambda x: Arnet.IpAddress( x ).value ) )
         vteps6 = sorted( Arnet.IpGenAddr( str( ip ) )
                            for ip in vl.remoteVtepAddr6 )
         separator = ' ' if vteps else ''
         vteps += separator + ' '.join( [ str( ip ) for ip in vteps6 ] )
         cmd = 'vxlan vlan %d flood vtep %s' % ( vlan, vteps )
         cmd = cmd.strip()
         cmds.addCommand( cmd )
   elif saveAll:
      cmd = 'no vxlan vlan flood vtep'
      cmds.addCommand( cmd )

   if entity.learnFrom == 'learnFromFloodList':
      cmd = 'vxlan learn-restrict flood'
      cmds.addCommand( cmd )
   elif entity.learnFrom == 'learnFromAny':
      cmd = 'vxlan learn-restrict any'
      cmds.addCommand( cmd )
   elif entity.learnFrom == 'learnFromList':
      if entity.learnPrefixList:
         prefs = sorted( entity.learnPrefixList.iterkeys(),
                         key=lambda x: x.ipPrefix.sortKey )
         prefixes = ' '.join( [ x.ipPrefix.stringValue for x in prefs ] )
         cmd = 'vxlan learn-restrict vtep %s' % prefixes
      else:
         # Learn from nobody
         cmd = 'vxlan learn-restrict vtep'
      cmds.addCommand( cmd )
   elif saveAll:
      cmd = 'no vxlan learn-restrict'
      cmds.addCommand( cmd )

   if entity.vlanToLearnRestrict:
      for ( vlan, vl ) in sorted( entity.vlanToLearnRestrict.iteritems(),
            key=lambda (k,_): k ):
         if vl.learnFrom == 'learnFromFloodList':
            cmd = 'vxlan vlan %d learn-restrict flood' % vlan
            cmds.addCommand( cmd )
         elif vl.learnFrom == 'learnFromAny':
            cmd = 'vxlan vlan %d learn-restrict any' % vlan
            cmds.addCommand( cmd )
         elif vl.learnFrom == 'learnFromList':
            if vl.prefixList:
               prefs = sorted( vl.prefixList.iterkeys(),
                               key=lambda x: x.ipPrefix.sortKey )
               prefixes = ' '.join( [ x.ipPrefix.stringValue for x in prefs ] )
               cmd = 'vxlan vlan %d learn-restrict vtep %s' % ( vlan, prefixes )
            else:
               # Learn from nobody
               cmd = 'vxlan vlan %d learn-restrict vtep' % vlan
            cmds.addCommand( cmd )
         elif saveAll:
            cmd = 'no vxlan vlan %d learn-restrict' % vlan
            cmds.addCommand( cmd )
   elif saveAll:
      cmd = 'no vxlan vlan learn-restrict vtep'
      cmds.addCommand( cmd )

   for vni in sorted( entity.vniToIpAclMap.keys() ):
      aclName = entity.vniToIpAclMap[ vni ]
      cmds.addCommand( 'vxlan vni %d routed ip access-list %s in' % (
         vni, aclName ) )

   if entity.parent.ecnPropagation:
      cmds.addCommand( 'vxlan qos ecn propagation' )
   elif options.saveAll:
      cmds.addCommand( 'no vxlan qos ecn propagation' )
