# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave
import Tac
from CliMode.TunnelRib import TunnelRibsBaseMode, TunnelRibBaseMode, \
                              tunnelTableIdToCliToken

ProtoPrefAction = Tac.Type( "Tunnel::TunnelTable::ProtoPrefAction" )

class TunnelRibsConfigMode( TunnelRibsBaseMode, CliSave.Mode ):
   def __init__( self, param ):
      TunnelRibsBaseMode.__init__( self )
      CliSave.Mode.__init__( self, param )

class TunnelRibConfigMode( TunnelRibBaseMode, CliSave.Mode ):
   def __init__( self, tunnelRibName ):
      TunnelRibBaseMode.__init__( self, tunnelRibName )
      CliSave.Mode.__init__( self, tunnelRibName )

CliSave.GlobalConfigMode.addChildMode( TunnelRibsConfigMode )
TunnelRibsConfigMode.addChildMode( TunnelRibConfigMode )
TunnelRibConfigMode.addCommandSequence( "TunnelRib.config" )

def saveTunnelRib( tunnelRib, prefStringDefault,
                   igpCostStringDefault, trsMode ):
   mode = trsMode[ TunnelRibConfigMode ].getOrCreateModeInstance( tunnelRib.name )
   cmds = mode[ "TunnelRib.config" ]
   for entry in tunnelRib.entry.itervalues():
      prefString = prefStringDefault
      igpCostPrefString = igpCostStringDefault
      if entry.prefMapping.action == ProtoPrefAction.prefStatic:
         prefString = " preference %d" % entry.prefMapping.staticPref
      if entry.igpCostPrefMapping.action == ProtoPrefAction.prefStatic:
         igpCostPrefString = " igp-cost preference %d" \
               % entry.igpCostPrefMapping.staticPref
      cmds.addCommand( "source-protocol %s%s%s" % ( tunnelTableIdToCliToken(
                       entry.tunnelTableId ), prefString,
                       igpCostPrefString ) )

@CliSave.saver( "Tunnel::TunnelTable::TunnelRibConfigDir",
                "tunnel/tunnelRibs/config" )
def saveTunnelRibCliConfig( entity, root, sysdbRoot, options ):
   trsMode = None
   for tunnelRib in sorted( entity.config.itervalues(),
                            key=lambda config: config.name ):
      if tunnelRib.name == "system-tunnel-rib":
         # For system-tunnel-rib we only print the o/p in show running-config
         # if some preference or igp-cost is overridden
         # But we always print the defaults in the show running-config all
         isAnyEntryOverriden = False
         for entry in tunnelRib.entry.itervalues():
            if entry.prefMapping.action == ProtoPrefAction.prefStatic or \
                  entry.igpCostPrefMapping.action == ProtoPrefAction.prefStatic:
               isAnyEntryOverriden = True
               break
         if isAnyEntryOverriden or options.saveAll:
            # At this point either we are printing show running-config and some
            # attribute of the system-tunnel-rib has been overridden, or we
            # are printing the show running-config all
            if not trsMode:
               trsMode = root[ TunnelRibsConfigMode ].getOrCreateModeInstance( None )
            saveTunnelRib( tunnelRib, "", "", trsMode )
      elif tunnelRib.name == "system-colored-tunnel-rib":
         # system-colored-tunnel-rib is currently not configurable
         continue
      elif tunnelRib.name == "system-tunneling-ldp-tunnel-rib":
         # system-tunneling-ldp-tunnel-rib is currently not configurable
         continue
      elif tunnelRib.name == "system-igp-shortcut-tunnel-rib":
         # system-igp-shortcut-tunnel-rib is currently not configurable
         continue
      else:
         # For custom tunnel ribs we print the same o/p in both
         # show running-config and show running-config all
         if not trsMode:
            trsMode = root[ TunnelRibsConfigMode ].getOrCreateModeInstance( None )
         saveTunnelRib( tunnelRib, "", "", trsMode )
