# Copyright (c) 2006-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements saving the TunnelIntf CLI.
#-------------------------------------------------------------------------------
import CliSave, IntfCliSave, TunnelIntfUtil
import Arnet
import Intf.IntfRange
from IntfCliSave import IntfConfigMode
from IpLibConsts import DEFAULT_VRF
from Toggles.MirroringToggleLib import toggleTapAggGreTunnelTerminationEnabled

IntfConfigMode.addCommandSequence(
   'TunnelIntf.config', after=[ 'Arnet.intf' ] )

#-------------------------------------------------------------------------------
# Saves the state of an Interface::EthIntfConfig object.
#-------------------------------------------------------------------------------

@CliSave.saver( 'Interface::TunnelIntfConfig', 'interface/config/tunnel/intf',
                requireMounts = ( 'interface/status/all',
                                  'interface/config/all' ) )
def saveTunnelIntfConfig( entity, root, sysdbRoot, options, requireMounts ):

   # Handle base class (Arnet::IntfConfig) first
   IntfCliSave.saveIntfConfig( entity, root, sysdbRoot, options,
                               requireMounts )

   cliMode = root[ IntfConfigMode ].getOrCreateModeInstance( entity.intfId )
   cmds = cliMode[ 'TunnelIntf.config' ]

   if entity.mode == 'tunnelIntfModeGre':
      cmds.addCommand( 'tunnel mode gre' )
   elif entity.mode == 'tunnelIntfModeIpip':
      cmds.addCommand( 'tunnel mode ipip' )
   elif entity.mode == 'tunnelIntfModeIpsec':
      cmds.addCommand( 'tunnel mode ipsec' )
   elif options.saveAll:
      cmds.addCommand( 'no tunnel mode' )

   if not entity.srcAddr.isUnspecified:
      cmds.addCommand( 'tunnel source %s' % entity.srcAddr )
   elif entity.srcIntf:
      cmds.addCommand( 'tunnel source interface %s' % entity.srcIntf )
   elif options.saveAll:
      cmds.addCommand( 'no tunnel source' )

   if not entity.dstAddr.isUnspecified:
      cmds.addCommand( 'tunnel destination %s' % entity.dstAddr )
   elif options.saveAll:
      cmds.addCommand( 'no tunnel destination' )

   if entity.pathMtuDiscovery:
      cmds.addCommand( 'tunnel path-mtu-discovery' )
   elif options.saveAll:
      cmds.addCommand( 'no tunnel path-mtu-discovery' )

   if entity.ttl != TunnelIntfUtil.tunnelDefaultTtl:
      cmds.addCommand( 'tunnel ttl %u' % entity.ttl )
   elif options.saveAll:
      cmds.addCommand( 'no tunnel ttl' )

   if entity.tos != TunnelIntfUtil.tunnelDefaultTos:
      cmds.addCommand( 'tunnel tos %u' % entity.tos )
   elif options.saveAll:
      cmds.addCommand( 'no tunnel tos' )

   if entity.key != TunnelIntfUtil.tunnelDefaultKey:
      cmds.addCommand( 'tunnel key %u' % entity.key )
   elif options.saveAll:
      cmds.addCommand( 'no tunnel key' )

   if entity.maxMss != TunnelIntfUtil.tunnelDefaultMaxMss:
      cmds.addCommand( 'tunnel mss ceiling %u' % entity.maxMss )
   elif options.saveAll:
      cmds.addCommand( 'no tunnel mss ceiling' )

   if entity.ipsec == True:
      cmds.addCommand( "tunnel ipsec profile %s" 
                       % ( entity.ipsecProfile ) )
   elif options.saveAll:
      cmds.addCommand( "no tunnel ipsec profile" ) 

   if entity.underlayVrfName != DEFAULT_VRF:
      cmds.addCommand( 'tunnel underlay vrf %s' % entity.underlayVrfName )
   elif options.saveAll:
      cmds.addCommand( 'no tunnel underlay vrf' )

   if toggleTapAggGreTunnelTerminationEnabled():
      # For tapAgg GRE tunnel termination
      tapGroups = entity.tapGroup.keys()
      if tapGroups:
         cmds.addCommand( 'tap default group %s'
                          % ' group '.join( sorted( tapGroups ) ) )
      elif options.saveAll:
         cmds.addCommand( 'no tap default group' )

      tapRawIntfs = entity.tapRawIntf.keys()
      phyIntfList = [ i for i in tapRawIntfs if not i.startswith( 'Po' ) ]
      lagIntfList = [ i for i in tapRawIntfs if i.startswith( 'Po' ) ]
      if tapRawIntfs:
         if phyIntfList:
            # Due to our parser, we can not specify range for physical interfaces
            # in the startup config. For lag interfaces, it works fine
            for intf in Arnet.sortIntf( phyIntfList ):
               cmds.addCommand( 'tap default interface %s' % intf )
         if lagIntfList:
            printLagList = Intf.IntfRange.intfListToCanonical( lagIntfList )
            cmds.addCommand( 'tap default interface %s'
                             % printLagList[ 0 ] )
      elif options.saveAll:
         cmds.addCommand( 'no tap default interface' )

      if entity.tapEncapGrePreserve:
         cmds.addCommand( 'tap encapsulation gre preserve' )
      elif options.saveAll:
         cmds.addCommand( 'no tap encapsulation gre preserve' )

