# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave, Tac
from RoutingIntfUtils import allRoutingProtocolIntfNames
from CliMode.Te import GlobalTeMode
from IntfCliSave import IntfConfigMode
import MultiRangeRule

def bitMaskToStr( bitMask ):
   '''
   Convert an integer representing a bitmask to a string representing each bit
   set to 1 in this bitmask.
   ie:
   0000 -> ''
   0001-> '0'
   0011 -> '0-1'
   0001 0011 -> '0-1,4'
   '''
   if not bitMask:
      return ''
   maskList = []
   i = 0
   while bitMask:
      if bitMask % 2:
         maskList.append( i )
      i += 1
      bitMask >>= 1

   return MultiRangeRule.multiRangeToCanonicalString( maskList )

#-------------------------------------------------------------------------------
# Object used for saving commands in router-traffic-engineering mode
#-------------------------------------------------------------------------------
class RouterGlobalTeMode( GlobalTeMode, CliSave.Mode ):
   def __init__( self, param ):
      GlobalTeMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( RouterGlobalTeMode,
                                      after=[ IntfConfigMode ] )
RouterGlobalTeMode.addCommandSequence( 'TrafficEngineering.global' )

IntfConfigMode.addCommandSequence( 'TrafficEngineering.intf' )
def saveIntfConfig( teConfig, entity, root, options ):
   mode = root[ IntfConfigMode ].getOrCreateModeInstance( entity.name )
   cmds = mode[ 'TrafficEngineering.intf' ]

   if entity.enabled != entity.enabledDefault:
      cmds.addCommand( 'traffic-engineering' )
   elif options.saveAll:
      cmds.addCommand( 'no traffic-engineering' )

   bwUnit = Tac.Type( "TrafficEngineering::BwUnitType" )
   if entity.maxReservableBw != entity.maxReservableBwDefault:
      if entity.maxReservableBwUnit == bwUnit.percent:
         unitStr = "percent"
      elif entity.maxReservableBwUnit == bwUnit.mbps:
         unitStr = "mbps"
      elif entity.maxReservableBwUnit == bwUnit.gbps:
         unitStr = "gbps"
      cmds.addCommand( 'traffic-engineering bandwidth %d %s' %
                        ( entity.maxReservableBw, unitStr ) )
   elif options.saveAll:
      cmds.addCommand( 'no traffic-engineering bandwidth' )

   if entity.adminGroup != entity.adminGroupDefault or options.saveAll:
      if entity.adminGroupListRepr:
         adminGroup = bitMaskToStr( entity.adminGroup )
         cmd = 'traffic-engineering administrative-group {}'.format( adminGroup )
      else:
         cmd = ( 'traffic-engineering administrative-group 0x%x' %
                 ( entity.adminGroup ) )
      cmds.addCommand( cmd )

   if entity.srlgIdList:
      for srlg in sorted( entity.srlgIdList ):
         cmds.addCommand( 'traffic-engineering srlg %d' % srlg )
   if entity.srlgNameList:
      for srlg in sorted( entity.srlgNameList ):
         cmds.addCommand( 'traffic-engineering srlg %s' % srlg )

   if entity.metric != entity.metricDefault:
      cmds.addCommand( 'traffic-engineering metric %d' % ( entity.metric ) )
   elif options.saveAll:
      cmds.addCommand( 'no traffic-engineering metric' )

CliSave.GlobalConfigMode.addCommandSequence( 'TrafficEngineering.config',
                                             after=[ RouterGlobalTeMode ] )

def saveGlobalConfig( teConfig, root ):
   if teConfig.enabled != teConfig.enabledDefault:
      mode = root [ RouterGlobalTeMode ].getOrCreateModeInstance( "teInstance" )
      cmds = mode [ 'TrafficEngineering.global' ]

   if teConfig.routerId != teConfig.routerIdDefault:
      cmds.addCommand( 'router-id ipv4 %s' % teConfig.routerId )

   if teConfig.routerIdV6 != teConfig.routerIdV6Default:
      cmds.addCommand( 'router-id ipv6 %s' % teConfig.routerIdV6 )

   for srlgNameToIdMap in sorted( teConfig.srlgMap.values() ):
      cmds.addCommand( 'srlg %s %d' % ( srlgNameToIdMap.srlgName,
                                        srlgNameToIdMap.srlgId ) )

@CliSave.saver( 'TrafficEngineering::Config', 'te/config',
                requireMounts = ( 'interface/config/all',
                                  'interface/status/all' ) )
def saveConfig( teConfig, root, sysdbRoot, options, requireMounts ):

   # Save traffic-engieering global config
   saveGlobalConfig( teConfig, root )

   # Check if any traffic-engieerning interface config is made
   teIntfConfigEnabled = len( teConfig.intfConfig ) > 0

   if not teIntfConfigEnabled and not options.saveAllDetail:
      return

   if options.saveAllDetail:
      rpIntfs = allRoutingProtocolIntfNames( sysdbRoot, includeEligible=True,
                                             requireMounts=requireMounts )
      intfs = set( rpIntfs + teConfig.intfConfig.keys() )
   elif options.saveAll:
      # Display TE configuration on routing protocol interfaces as well as
      # switchports present in intfConfig collection.
      rpIntfs = allRoutingProtocolIntfNames( sysdbRoot,
                                             requireMounts=requireMounts )
      intfs = set( rpIntfs + teConfig.intfConfig.keys() )
   else:
      intfs = teConfig.intfConfig

   # Save config for each interface
   for intfName in intfs:
      intfConfig = teConfig.intfConfig.get( intfName )
      if not intfConfig:
         if options.saveAll:
            intfConfig = Tac.newInstance( 'TrafficEngineering::IntfConfig',
                                          intfName )
         else:
            continue
      saveIntfConfig( teConfig, intfConfig, root, options )
