#!/usr/bin/env python
# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave
import Tac
import EthIntfUtil, Arnet
from IntfCliSave import IntfConfigMode
from CliMode.TapAgg import TapAggMode
import Intf.IntfRange

TapAggModeType = Tac.Type( "TapAgg::TapAggMode" )
isRecirc = Tac.Type( "Arnet::PortChannelIntfId" ).isRecirc
isUnconnected = Tac.Type( "Arnet::EthIntfId" ).isUnconnected
FcsModeType = Tac.Type( 'TapAgg::TapAggFcsMode' )
TimestampHeaderFormatType = Tac.Type( 'TapAgg::TapAggTimestampHeaderFormat' )
ToolIdentityMode = Tac.Type( 'Bridging::Input::ToolIdentityMode' )
VlanIdTuple = Tac.Type( 'Bridging::VlanIdTuple' )

class TapAggConfigMode( TapAggMode, CliSave.Mode ):

   def __init__( self, param ):
      self.modeKey = 'tap-agg'
      TapAggMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( TapAggConfigMode, before=[ IntfConfigMode ] )
TapAggConfigMode.addCommandSequence( 'Tapagg.tapaggMode' )

IntfConfigMode.addCommandSequence( 'Tapagg.switchport' )

@CliSave.saver( 'TapAgg::CliConfig', 'tapagg/cliconfig',
                requireMounts=( 'tapagg/hwstatus', 'interface/hwstatus' ) )
def saveTapAggConfig( entity, root, sysdbRoot, options, requireMounts ):
   if not entity.enabled and not entity.noerrdisableIntf and \
          not entity.trafficSteeringMacAclMatchIp and \
          not entity.sliceProfileConfig and not options.saveAll and \
          entity.globalTruncationSize == 0:
      return
   tapAggHwStatus = requireMounts[ 'tapagg/hwstatus' ]
   intfHwStatus = requireMounts[ 'interface/hwstatus' ]
   mode = root[ TapAggConfigMode ].getOrCreateModeInstance( 'TapAggConfig' )
   cmd = mode[ 'Tapagg.tapaggMode' ]
   profile = ''
   if entity.enabled:
      if entity.tcamProfile:
         profile = ' profile ' + entity.tcamProfile
      if entity.mode == TapAggModeType.tapAggModeExclusive:
         cmd.addCommand( 'mode exclusive' + profile )
      elif entity.mode == TapAggModeType.tapAggModeMixed:
         profileToCardMap = {}
         for ( card, profile ) in entity.sliceProfileConfig.items():
            if profile not in profileToCardMap:
               profileToCardMap[ profile ] = []
            profileToCardMap[ profile ].append( filter( str.isdigit, card ) )
         for ( profile, cardList ) in sorted( profileToCardMap.iteritems() ):
            command = 'mode mixed module linecard ' + ",".join( cardList )
            if profile:
               command += ' profile ' + profile
            cmd.addCommand( command )
      else:
         assert False, "unsupported mode"
   elif options.saveAll:
      cmd.addCommand( 'no mode' )

   if tapAggHwStatus.brVnTagStripSupported:
      if entity.vnTagStrip:
         cmd.addCommand( 'encapsulation vn-tag strip' )
      elif options.saveAll:
         cmd.addCommand( 'no encapsulation vn-tag strip' )

      if entity.brTagStrip:
         cmd.addCommand( 'encapsulation dot1br strip' )
      elif options.saveAll:
         cmd.addCommand( 'no encapsulation dot1br strip' )

   if tapAggHwStatus.lldpReceiveConfigurable:
      if entity.lldpReceive:
         cmd.addCommand( 'protocol lldp trap' )
      elif options.saveAll:
         cmd.addCommand( 'no protocol lldp trap' )

   for intf in Arnet.sortIntf( entity.noerrdisableIntf ):
      cmd.addCommand( 'mode exclusive no-errdisable %s' % intf )

   if tapAggHwStatus.macAclMatchIpConfigurable:
      if entity.trafficSteeringMacAclMatchIp:
         cmd.addCommand( 'service-policy type tapagg mac access-list match ip' )
      elif options.saveAll:
         cmd.addCommand( 'no service-policy type tapagg mac access-list match ip' )

   if tapAggHwStatus.globalIngressTruncationSizeSupported or \
      tapAggHwStatus.globalEgressTruncationSizeSupported:
      if entity.globalTruncationSize != 0:
         cmd.addCommand( 'truncation size {}'.format( entity.globalTruncationSize ) )
      elif options.saveAll:
         cmd.addCommand( 'default truncation size' )

   if intfHwStatus.timestampHeaderSupported:
      if entity.timestampReplaceSmac:
         cmd.addCommand( 'mac timestamp replace source-mac' )
      elif options.saveAll:
         cmd.addCommand( 'no mac timestamp replace source-mac' )

      if entity.timestampHeaderFormat == \
         TimestampHeaderFormatType.tapAggTimestamp48bit:
         cmd.addCommand( 'mac timestamp header format 48-bit' )
      elif entity.timestampHeaderFormat == \
           TimestampHeaderFormatType.tapAggTimestamp64bit:
         cmd.addCommand( 'mac timestamp header format 64-bit' )
      elif options.saveAll:
         cmd.addCommand( 'no mac timestamp header format' )

   if tapAggHwStatus.fcsModeSupported:
      if entity.fcsMode == FcsModeType.tapAggFcsAppend:
         cmd.addCommand( 'mac fcs append' )
      elif options.saveAll:
         cmd.addCommand( 'default mac fcs' )

   if tapAggHwStatus.fcsErrorModeSupported:
      if entity.fcsErrorMode == FcsModeType.tapAggFcsErrorCorrect:
         cmd.addCommand( 'mac fcs-error correct' )
      elif entity.fcsErrorMode == FcsModeType.tapAggFcsErrorDiscard:
         cmd.addCommand( 'mac fcs-error discard' )
      elif entity.fcsErrorMode == FcsModeType.tapAggFcsErrorPassThrough:
         cmd.addCommand( 'mac fcs-error pass-through' )
      elif options.saveAll:
         cmd.addCommand( 'default mac fcs-error' )

@CliSave.saver( 'Bridging::Input::CliConfig', 'bridging/input/config/cli',
                requireMounts = ( 'bridging/config', 'tapagg/cliconfig',
                                  'tapagg/hwstatus' ) )
def saveTapAggBridgingConfig( entity, root, sysdbRoot, options,
                              requireMounts ):
   # Display interface configs
   if options.saveAll:
      # Get all eligible switchports, since '[no] switchport' is always displayed.
      bridgingConfig = requireMounts[ 'bridging/config' ]
      cfgIntfNames = EthIntfUtil.allSwitchportNames( bridgingConfig,
                                                     includeEligible=True )
   else:
      cfgIntfNames = entity.switchIntfConfig

   tapaggHwStatus = requireMounts[ 'tapagg/hwstatus' ]
   truncationSizePerIngressPortSupported = \
         tapaggHwStatus.truncationSizePerIngressPortSupported
   truncationSizePerEgressPortSupported = \
         tapaggHwStatus.truncationSizePerEgressPortSupported
   tapAggConfig = requireMounts[ 'tapagg/cliconfig' ] 
   toolGroupNewFormat = tapAggConfig.toolGroupNewFormat
   for intfName in cfgIntfNames:
      # Tap agg not supported on these interfaces
      if isRecirc( intfName ) or isUnconnected( intfName ):
         continue

      intfConfig = entity.switchIntfConfig.get( intfName )
      if not intfConfig:
         if options.saveAll:
            intfConfig = Tac.newInstance( 'Bridging::Input::SwitchIntfConfig', 
                                          intfName, 'access' )
         else:
            continue  
      saveTapAggSwitchIntfConfig( intfConfig, root, sysdbRoot, options,
                                  toolGroupNewFormat,
                                  truncationSizePerIngressPortSupported,
                                  truncationSizePerEgressPortSupported )

#-------------------------------------------------------------------------------
# Saves the tap-agg specific state of an Bridging::SwitchIntfConfig object.
#-------------------------------------------------------------------------------
def saveTapAggSwitchIntfConfig( entity, root, sysdbRoot, options,
                                toolGroupNewFormat=False,
                                truncationSizePerIngressPortSupported=True,
                                truncationSizePerEgressPortSupported=True ):
   mode = root[ IntfConfigMode ].getOrCreateModeInstance( entity.intfId )
   cmds = mode[ 'Tapagg.switchport' ]
   sicType = Tac.Type( "Bridging::Input::SwitchIntfConfig" )
   saveAll = options.saveAll
   saveAllDetail = options.saveAllDetail

   # Display defaults on all ports for saveAllDetail and on L2 ports for saveAll. 
   saveDefault = saveAllDetail or ( saveAll and entity.enabled )
   
   if entity.switchportMode in ( 'tap', 'tool', 'tap-tool' ):
      cmds.addCommand( 'switchport mode %s' % entity.switchportMode )

   if entity.tapNativeVlan != 1 or saveDefault:
      cmds.addCommand( 'switchport tap native vlan %s'
                       % entity.tapNativeVlan )
   
   vlanTuple = entity.tapIdentificationVlan
   if vlanTuple.outerVid != 0:
      if vlanTuple.innerVid != 0:
         cmds.addCommand( 'switchport tap identity %s inner %s'
                          % ( vlanTuple.outerVid, vlanTuple.innerVid ) )
      else:
         cmds.addCommand( 'switchport tap identity %s'
                          % vlanTuple.outerVid )
   elif saveDefault:
      cmds.addCommand( 'no switchport tap identity' )

   if entity.tapVxlanStrip:
      cmds.addCommand( 'switchport tap encapsulation vxlan strip' )
   elif saveDefault:
      cmds.addCommand( 'no switchport tap encapsulation vxlan strip' )

   if entity.tapMplsPop:
      cmds.addCommand( 'switchport tap mpls pop all' )
   elif saveDefault:
      cmds.addCommand( 'no switchport tap mpls pop all' )

   if entity.toolMplsPop:
      cmds.addCommand( 'switchport tool mpls pop all' )
   elif saveDefault:
      cmds.addCommand( 'no switchport tool mpls pop all' )

   if entity.tapAllowedVlans == '':
      cmds.addCommand( 'switchport tap allowed vlan none' )
   elif entity.tapAllowedVlans != '1-4095' or saveDefault:
      cmds.addCommand( 'switchport tap allowed vlan %s' % 
                       entity.tapAllowedVlans )

   if entity.toolAllowedVlans == '':
      cmds.addCommand( 'switchport tool allowed vlan none' )
   elif entity.toolAllowedVlans != '1-4095' or saveDefault:
      cmds.addCommand( 'switchport tool allowed vlan %s' % 
                       entity.toolAllowedVlans )

   if entity.toolIdentityTagging != 'none':
      cmds.addCommand( 'switchport tool identity %s' % entity.toolIdentityTagging )
   elif saveDefault:
      cmds.addCommand( 'no switchport tool identity' )
      
   if entity.tapTruncationSize != 0:
      # Check if configured value is the special "tapTruncationUseGlobalSize" value
      # (1) and not print the size, this is to support downgrade to earlier releases
      # that didn't support configuring the size.
      if ( truncationSizePerIngressPortSupported and
           entity.tapTruncationSize != sicType.tapTruncationUseGlobalSize ):
         cmds.addCommand( 'switchport tap truncation %d' % entity.tapTruncationSize )
      else:
         cmds.addCommand( 'switchport tap truncation' )
   elif saveDefault:
      cmds.addCommand( 'no switchport tap truncation' )

   if entity.toolTruncationSize != 0:
      if truncationSizePerEgressPortSupported:
         cmds.addCommand( 'switchport tool truncation %d' %
                          entity.toolTruncationSize )
      else:
         cmds.addCommand( 'switchport tool truncation' )
   elif saveDefault:
      cmds.addCommand( 'no switchport tool truncation' )

   tapGroups = entity.tapGroup.keys()
   if tapGroups:
      cmds.addCommand( 'switchport tap default group %s'
                  % ' group '.join( sorted( tapGroups ) ) )
   elif saveDefault:
      cmds.addCommand( 'no switchport tap default group' )

   tapRawIntfs = entity.tapRawIntf.keys()
   phyIntfList = [ i for i in tapRawIntfs if not i.startswith('Po') ]
   lagIntfList = [ i for i in tapRawIntfs if i.startswith('Po') ]
   if tapRawIntfs:
      if phyIntfList:
         # Due to our parser, we can not specify range for physical interfaces
         # in the startup config. For lag interfaces, it works fine
         for intf in Arnet.sortIntf( phyIntfList ):
            cmds.addCommand( 'switchport tap default interface %s' % intf )
      if lagIntfList:
         printLagList = Intf.IntfRange.intfListToCanonical( lagIntfList )
         cmds.addCommand( 'switchport tap default interface %s'
               % printLagList[ 0 ] )
   elif saveDefault:
      cmds.addCommand( 'no switchport tap default interface' )

   # For toolGroups, both comma and blank saparated list is supported
   # but only old format is generated by default. To generate new format,
   # user can run 'config convert new-syntax' command.
   toolGroups = entity.toolGroup.keys()
   if toolGroups:
      if toolGroupNewFormat:
         cmds.addCommand( 'switchport tool group %s'
                  % ' group '.join( sorted( toolGroups ) ) )
      else:
         cmds.addCommand( 'switchport tool group set %s'
                       % ' '.join( sorted( toolGroups ) ) )
   elif saveDefault:
      cmds.addCommand( 'no switchport tool group' )

   if entity.toolDot1qRemoveVlans != '':
      cmds.addCommand( 'switchport tool dot1q remove outer %s'
                       % entity.toolDot1qRemoveVlans )
   elif saveDefault:
      cmds.addCommand( 'no switchport tool dot1q remove outer' )
