#!/usr/bin/env python
# Copyright (c) 2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable-msg=E1101

import Tac
import CliSave
from IntfCliSave import IntfConfigMode
import os, EthIntfUtil

IntfConfigMode.addCommandSequence( 'StormControl.intf' )
CliSave.GlobalConfigMode.addCommandSequence( 'Config.stormControlAggregate' )
CliSave.GlobalConfigMode.addCommandSequence( 'Config.stormControlBurst' )

@CliSave.saver( 'Bridging::StormControl::Config', 'bridging/stormcontrol/config',
                requireMounts = ( 'bridging/stormcontrol/status',
                                  'bridging/config' ) )
def saveConfig( entity, root, sysdbRoot, options,
                requireMounts ):
   statusDir = requireMounts[ 'bridging/stormcontrol/status' ]
   status = statusDir[ 'all' ]
   sliceDir = statusDir[ 'slice' ]
   saveAll = options.saveAll
   saveAllDetail = options.saveAllDetail
   # Display the default config only if stormcontrol is supported by the platform.

   stormControlFlag = False
   for sliceStatus in sliceDir.itervalues():
      if sliceStatus.stormControlSupported:
         stormControlFlag = True

   if not status.stormControlSupported and \
      not stormControlFlag and \
      not os.environ.get( 'STORMCONTROL_CLITEST' ):
      saveAll = False
      saveAllDetail = False

   # Get the list of interfaces based on command type
   bridgingConfig = requireMounts[ 'bridging/config' ]
   if saveAllDetail:
      cfgIntfNames = EthIntfUtil.allSwitchportNames( bridgingConfig,
                                                     includeEligible=True )
   elif saveAll:
      # We allow L2 configuration on routed ports. List of cfgIntfNames are all
      # switchports AND routed ports for which non-default config exists.
      cfgIntfNames = set( EthIntfUtil.allSwitchportNames( bridgingConfig ) + \
                          entity.intfConfig.keys() )
   else:
      cfgIntfNames = entity.intfConfig
   
   defaultThreshold = Tac.Value( 'Bridging::StormControl::Threshold' )
   
   # Aggregate storm-control command for traffic-class
   # [ no ] storm-control bum aggregate traffic-class <tc> level pps <rate>
   if status.aggregateStormControlSupported: 
      aggregateStormControlCmd = 'storm-control bum aggregate traffic-class'
      cmds = root[ 'Config.stormControlAggregate' ]
      if saveAll:
         for tc in xrange( 8 ):
            aggregateTcConfig = entity.aggregateTcConfig.get( tc )
            cmd = aggregateStormControlCmd + ' %d' % tc
            if not aggregateTcConfig:
               cmd = 'no ' + cmd
            else:
               assert aggregateTcConfig.level != defaultThreshold
               cmd += ' %s' % ( aggregateTcConfig.level.formatForCli() )
            cmds.addCommand( cmd )
      else:
         for tc in entity.aggregateTcConfig:
            aggregateTcConfig = entity.aggregateTcConfig[ tc ]
            if aggregateTcConfig and aggregateTcConfig.level != defaultThreshold:
               cmd = aggregateStormControlCmd +' %d' % tc
               cmd += ' %s' % ( aggregateTcConfig.level.formatForCli() )
               cmds.addCommand( cmd )

   for intfName in cfgIntfNames:
      intfConfig = entity.intfConfig.get( intfName )
      if not intfConfig:
         if saveAll:
            intfConfig = Tac.newInstance( "Bridging::StormControl::IntfConfig",
                                          intfName )
         else:
            continue

      # display intfConfig
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intfConfig.name )
      cmds = mode[ 'StormControl.intf' ]
      if intfConfig.broadcastLevel != defaultThreshold:
         cmds.addCommand(
            'storm-control broadcast %s'
            % intfConfig.broadcastLevel.formatForCli() )
      elif saveAll:
         cmds.addCommand( 'no storm-control broadcast' )
      
      if intfConfig.multicastLevel != defaultThreshold:
         cmds.addCommand(
            'storm-control multicast %s'
            % intfConfig.multicastLevel.formatForCli() )
      elif saveAll:
         cmds.addCommand( 'no storm-control multicast' )

      if intfConfig.uucastLevel != defaultThreshold:
         cmds.addCommand( 
            'storm-control unknown-unicast %s'
            % intfConfig.uucastLevel.formatForCli() )
      elif saveAll:
         cmds.addCommand( 'no storm-control unknown-unicast' )

      if intfConfig.allLevel != defaultThreshold:
         cmds.addCommand(
            'storm-control all %s'
            % intfConfig.allLevel.formatForCli() )
      elif saveAll:
         cmds.addCommand( 'no storm-control all' )

      if intfConfig.stormControlDropLoggingMode == 'on':
         cmds.addCommand( 'logging event storm-control discards' )
      elif intfConfig.stormControlDropLoggingMode == 'off':
         cmds.addCommand( 'no logging event storm-control discards' )
      elif saveAll:
         cmds.addCommand( 'logging event storm-control discards use-global' )

   # Global burst time
   # [ no ] storm-control burst time TIME ( milliseconds | microseconds )
   defaultBurst = Tac.Value( 'Bridging::StormControl::BurstInfo' )

   stormControlBurstFlag = False
   if status.stormControlBurstSupported:
      stormControlBurstFlag = True
   else :
      for sliceStatus in sliceDir.itervalues():
         if sliceStatus.stormControlBurstSupported:
            stormControlBurstFlag = True

   if stormControlBurstFlag:
      burstStormControlCmd = 'storm-control burst time'
      cmds = root[ 'Config.stormControlBurst' ]
      name = "burstConfig"
      burstConfig = entity.burstConfig.get( name )
      if saveAll:
         if not burstConfig:
            cmd = 'no ' + burstStormControlCmd
         else:
            cmd = burstStormControlCmd + ' %d %s' % ( burstConfig.burst.burstTime,
                                                   burstConfig.burst.type )
         cmds.addCommand( cmd )
      else:
         if burstConfig and burstConfig.burst != defaultBurst:
            cmd = burstStormControlCmd + ' %d %s' % ( burstConfig.burst.burstTime,
                                                      burstConfig.burst.type )
            cmds.addCommand( cmd )

CliSave.GlobalConfigMode.addCommandSequence( 'Config.stormControlDropsLog' )

@CliSave.saver( 'Bridging::StormControl::Config', 'bridging/stormcontrol/config' )
def globalStormControlDropsLogcmds( entity, root, sysdbRoot, options ):
   cmds = root[ 'Config.stormControlDropsLog' ]
   if entity.stormControlDropLoggingMode == 'on':
      cmds.addCommand( 'logging event storm-control discards global' )
   elif options.saveAll:
      cmds.addCommand( 'no logging event storm-control discards global' )

   if entity.stormControlDropLogInterval:
      cmds.addCommand( 'logging event storm-control discards interval %d' % \
            entity.stormControlDropLogInterval )
   elif options.saveAll:
      cmds.addCommand( 'no logging event storm-control discards interval' )
