# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave
import Tac
from CliSavePlugin.Security import SecurityConfigMode
from CliMode.Ssl import SslProfileMode
import DefaultSslProfile

Constants = Tac.Type( "Mgmt::Security::Ssl::Constants" )

class SslProfileConfigMode( SslProfileMode, CliSave.Mode ):
   def __init__( self, param ):
      SslProfileMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

SecurityConfigMode.addChildMode( SslProfileConfigMode, 
                                 after=[ 'Mgmt.security' ] )
SslProfileConfigMode.addCommandSequence( 'Mgmt.security.ssl.profile' )
SslProfileConfigMode.addCommandSequence( 'Mgmt.security.ssl' )

@CliSave.saver( 'Mgmt::Security::Ssl::Config', 'mgmt/security/ssl/config' )
def saveSslProfileConfig( entity, root, sysdbRoot, options ):
   for profileName, profileConfig in entity.profileConfig.iteritems():
      if not options.saveAll and profileName == DefaultSslProfile.ARISTA_PROFILE:
         continue
      if profileConfig.profileType != "profileTypeSsl":
         continue
      secMode = root[ SecurityConfigMode ].getOrCreateModeInstance( 'security' )
      mode = secMode[ SslProfileConfigMode ].getOrCreateModeInstance( profileName )
      cmds = mode[ 'Mgmt.security.ssl.profile' ]

      tlsVersionMap = [ ( Constants.tlsv1, '1.0' ),
                        ( Constants.tlsv1_1, '1.1' ),
                        ( Constants.tlsv1_2, '1.2' ) ]
      tlsVersionsEnabled = []
      for mask, version in tlsVersionMap:
         if profileConfig.tlsVersion & mask:
            tlsVersionsEnabled.append( version )
      if ( options.saveAll or not
           profileConfig.tlsVersion == Constants.allTlsVersion ):
         cmds.addCommand( 'tls versions %s' % ' '.join( tlsVersionsEnabled ) )

      if profileConfig.fipsMode:
         cmds.addCommand( 'fips restrictions' )
      elif options.saveAll:
         cmds.addCommand( 'no fips restrictions' )

      if profileConfig.cipherSuite != Constants.defaultCipherSuite():
         cmds.addCommand( 'cipher-list %s' % profileConfig.cipherSuite )
      elif options.saveAll:
         cmds.addCommand( 'no cipher-list' )

      if profileConfig.certKeyPair.certFile != "":
         cmds.addCommand( 'certificate %s key %s' %
                          ( profileConfig.certKeyPair.certFile, 
                            profileConfig.certKeyPair.keyFile ) )

      for certName in profileConfig.trustedCert:
         cmds.addCommand( 'trust certificate %s' % ( certName ) )

      for certName in profileConfig.chainedCert:
         cmds.addCommand( 'chain certificate %s' % ( certName ) )

      for crlName in profileConfig.crl:
         cmds.addCommand( 'crl %s' % ( crlName ) )

      if profileConfig.verifyExtendedParameters:
         cmds.addCommand( 'certificate requirement extended-key-usage' )
      elif options.saveAll:
         cmds.addCommand( 'no certificate requirement extended-key-usage' )

      if profileConfig.verifyHostnameMatch:
         cmds.addCommand( 'certificate requirement hostname match' )
      elif options.saveAll:
         cmds.addCommand( 'no certificate requirement hostname match' )

      if profileConfig.verifyBasicConstraintTrust:
         cmds.addCommand( 'trust certificate requirement '
                          'basic-constraint ca true' )
      elif options.saveAll:
         cmds.addCommand( 'no trust certificate requirement '
                          'basic-constraint ca true' )

      if profileConfig.verifyBasicConstraintChain:
         cmds.addCommand( 'chain certificate requirement '
                          'basic-constraint ca true' )
      elif options.saveAll:
         cmds.addCommand( 'no chain certificate requirement '
                          'basic-constraint ca true' )

      if not profileConfig.verifyExpiryDateEndCert:
         cmds.addCommand( 'certificate policy expiry-date ignore' )
      elif options.saveAll:
         cmds.addCommand( 'no certificate policy expiry-date ignore' )

      if profileConfig.verifyChainHasRootCA:
         cmds.addCommand( "chain certificate requirement include root-ca" )
      elif options.saveAll:
         cmds.addCommand( "no chain certificate requirement include root-ca" )

      for exp in profileConfig.commonNameRegex:
         cmds.addCommand( 'certificate common-name username regexp %s' % ( exp ) )
