# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave
import Tac
from SrTePolicyLib import (
   MplsLabel,
   Weight,
)
from SrTePolicy import Index
from CliMode.SrTePolicy import SrTeModeBase
from CliMode.SrTePolicy import SrTePolicyModeBase
from CliMode.SrTePolicy import SrTePolicyPathGroupModeBase
from CliSavePlugin.TeCliSave import RouterGlobalTeMode
from SrTePolicyCommonLib import tacEnlp, tacEnlpEnum
from Toggles.SrTePolicyToggleLib import toggleSrTePolicyMetricEnabled

class SrTeMode( SrTeModeBase, CliSave.Mode ):
   def __init__( self, param ):
      SrTeModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class SrTePolicyMode( SrTePolicyModeBase, CliSave.Mode ):
   def __init__( self, param ):
      SrTePolicyModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class SrTePolicyPathGroupMode( SrTePolicyPathGroupModeBase, CliSave.Mode ):
   def __init__( self, param ):
      SrTePolicyPathGroupModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

RouterGlobalTeMode.addChildMode( SrTeMode )
SrTeMode.addCommandSequence( 'SrTePolicy.config' )

SrTeMode.addChildMode( SrTePolicyMode )
SrTePolicyMode.addCommandSequence( 'SrTePolicy.policy.config' )

SrTePolicyMode.addChildMode( SrTePolicyPathGroupMode )
SrTePolicyPathGroupMode.addCommandSequence( 'SrTePolicy.policy.path.config' )

def getOrCreateSrTeModeInst( root ):
   teMode = root[ RouterGlobalTeMode ].getOrCreateModeInstance( "teInstance" )
   return teMode[ SrTeMode ].getOrCreateModeInstance( "srInstance" )

def sfbdMultiDef():
   return Tac.Type( 'Bfd::BfdMultiplier' ).defval

def sbfdConfigDef():
   return Tac.Value( 'SrTePolicy::Sbfd::PolicySbfdConfig', 0, 0,
                                                        sfbdMultiDef() )

def _saveSbfdConfig( policy, policyCmds, saveAll ):
   if policy.sbfdConfig.sbfdRemoteDiscriminator != 0:
      bfdStr = "sbfd remote-discriminator "
      discriminator = policy.sbfdConfig.sbfdRemoteDiscriminator
      if policy.sbfdConfig.sbfdDiscIsU32:
         bfdStr += str( discriminator )
      else:
         ipDisc = Tac.Value( 'Arnet::IpAddr', discriminator ).stringValue
         bfdStr += ipDisc

      if policy.sbfdConfig.sbfdInterval != 0:
         bfdStr += " interval %u multiplier %u" % (
                                   policy.sbfdConfig.sbfdInterval,
                                   policy.sbfdConfig.sfbdMulti )
      policyCmds.addCommand( bfdStr )
   elif saveAll:
      policyCmds.addCommand( 'no sbfd remote-discriminator' )

def saveInstanceConfig( srConfig, root, saveAll ):
   if srConfig.enabled:
      srteMode = getOrCreateSrTeModeInst( root )
   else:
      # Since we set srConfig.enabled=True the moment we enter the sr config submode,
      # we cannot have policy config unless srConfig.enabled is True
      assert len( srConfig.policy ) == 0, "SR is disabled, but policy "
      "configurations are present: {}".format( srConfig.policy.keys() )

   if srConfig.enabled:
      srTeConfigCmds = srteMode[ 'SrTePolicy.config' ]
      if srConfig.populateColoredTunnelRib:
         srTeConfigCmds.addCommand( 'rib system-colored-tunnel-rib' )
      elif saveAll:
         srTeConfigCmds.addCommand( 'no rib system-colored-tunnel-rib' )
      if srConfig.igpPrefCost != srConfig.defaultIgpPrefCost or saveAll:
         srTeConfigCmds.addCommand( 'igp-cost preference %u' % srConfig.igpPrefCost )
   for policyKey in sorted( srConfig.policy ):
      policy = srConfig.policy[ policyKey ]
      policyMode = srteMode[ SrTePolicyMode ].getOrCreateModeInstance( (
                                                           policy.key.endpoint,
                                                           policy.key.color ) )
      policyCmds = policyMode[ 'SrTePolicy.policy.config' ]
      if policy.bindingSid != MplsLabel.null or saveAll == True:
         policyCmds.addCommand( "binding-sid %s" % policy.bindingSid )
      if policy.policyName != '':
         policyCmds.addCommand( "name %s" % policy.policyName )
      if policy.policyDescription != '':
         policyCmds.addCommand( "description %s" % policy.policyDescription )

      if toggleSrTePolicyMetricEnabled():
         if ( policy.costMetric.metric and policy.costMetric.costSet ) or saveAll:
            # In saveAll, we do not save the setting for preference if it is not
            # configured but the metric is. There is no way to specify to use
            # parent/default preference with a specific metric (other
            # than configuring just the metric). In this case the preference
            # is only saved at the global
            # (traffic-engineering/segment-routing) level.
            if policy.costMetric.costSet:
               policyCmds.addCommand( "igp-cost preference %d metric %d" %
                       ( policy.costMetric.cost, policy.costMetric.metric ) )
            elif policy.costMetric.metric == policy.costMetric.defaultIgpMetric:
               policyCmds.addCommand( "no igp-cost" )
            else:
               policyCmds.addCommand( "igp-cost metric %d" %
                       policy.costMetric.metric )
         elif policy.costMetric.metric:
            policyCmds.addCommand( "igp-cost metric %d" %
                    policy.costMetric.metric )
         elif policy.costMetric.costSet:
            policyCmds.addCommand( "igp-cost preference %d" %
                    policy.costMetric.cost )

      for pathGroupPref in sorted( policy.candidatePath ):
         pathGroup = policy.candidatePath[ pathGroupPref ]
         pathMode = policyMode[ SrTePolicyPathGroupMode ].\
                                  getOrCreateModeInstance( ( policy.key.endpoint,
                                                             policy.key.color,
                                                             pathGroup.preference
                                                           ) )
         pathCmds = pathMode[ 'SrTePolicy.policy.path.config' ]
         if pathGroup.enlp != tacEnlp().valueDefault:
            enlpCommand = 'explicit-null '
            if pathGroup.enlp == tacEnlpEnum.ipv4AndIpv6:
               enlpCommand += 'ipv4 ipv6'
            else:
               enlpCommand += pathGroup.enlp
            pathCmds.addCommand( enlpCommand )
         for segmentList in sorted( pathGroup.staticSegmentList ):
            weight = pathGroup.staticSegmentList[ segmentList ].weight
            index = pathGroup.staticSegmentList[ segmentList ].index
            labelStackStr = "label-stack"
            for i in range( segmentList.stackSize ):
               labelStackStr += " "
               labelStackStr += str( segmentList.labelStack( i ) )
            weightStr = ""
            if weight != Weight().valueDefault or saveAll == True:
               weightStr += "weight " + str( weight )
            indexStr = ""
            if index != Index().valueDefault:
               indexStr += "index " + str( index )
            pathCmds.addCommand( "segment-list %s %s %s" % (
               labelStackStr, weightStr, indexStr ) )

      _saveSbfdConfig( policy, policyCmds, saveAll )

@CliSave.saver( 'SrTePolicy::Config',
                'te/segmentrouting/srtepolicy/staticconfig' )
def saveConfig( srConfig, root, sysdbRoot, options, requireMounts ):
   saveAll = options.saveAll

   saveInstanceConfig( srConfig, root, saveAll )
