#!/usr/bin/env python
# Copyright (c) 2008-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave
import Tracing
import Url
import Tac
from IpLibConsts import DEFAULT_VRF
from SnmpCliUtil import trapToToken
from CliSavePlugin.NetworkCliSave import networkConfigCmdSeq
import Toggles.SnmpToggleLib

__defaultTraceHandle__ = Tracing.Handle( 'SnmpCliSave' )

engineIdGenerator = None

# I want the Snmp commands to go after Network.config because some of the Snmp
# commands take hostname-or-ip-addrs and attempt to resolve hostnames, so it's
# better if the nameserver config is done first.
CliSave.GlobalConfigMode.addCommandSequence( 'Snmp.global',
   after=[ networkConfigCmdSeq ] )

_authTypes = { 'authMd5': 'md5', 'authSha': 'sha', 'authSha224': 'sha224',
      'authSha256': 'sha256', 'authSha384': 'sha384', 'authSha512': 'sha512' }
_authLevels = { 'levelNoAuth': 'noauth', 'levelAuth': 'auth',
                'levelAuthAndPriv': 'priv' }
_privTypes = { 'privacyAes': 'aes', 'privacyDes': 'des',
      'privacy3Des': '3des', 'privacyAes192': 'aes192',
      'privacyAes256': 'aes256' }
_defaultRemotePort = 162

# When someone types "show running-config", the Cli code walks the sysdb
# object tree, calling all the functions which registered with CliSave.saver
# to be called for a particular object. Below, I register
# saveSnmpConfig to be called for 'snmp/config'. When I'm called, I walk our
# entire Snmp config object tree, generating all non-default config.

@CliSave.saver( 'Snmp::Config', 'snmp/config',
                requireMounts=( 'hardware/entmib', ) )
def saveSnmpConfig( entity, root, sysdbRoot, options, requireMounts ):
   cmds = root[ 'Snmp.global' ]
   saveAll = options.saveAll

   if entity.engineId != "":
      cmds.addCommand( 'snmp-server engineID local %s' % entity.engineId )
   else:
      global engineIdGenerator
      if engineIdGenerator is None:
         Tracing.trace0( "Instantiating EngineIdGenerator" )
         entityMibStatus = requireMounts[ 'hardware/entmib' ]
         engineIdGenerator = Tac.newInstance( "Snmp::EngineIdGenerator",
                                              entityMibStatus, entity )

      # If there are any SNMPv3 users we emit the default engineID in the
      # running-config just to be safe, since the SNMPv3 users have localized
      # keys based on the engineID.
      engineIdDisplayed = False
      for u in entity.user.itervalues():
         if u.protocolVersion == 'v3':
            cmds.addCommand( 'snmp-server engineID local %s' %
                             engineIdGenerator.engineId )
            engineIdDisplayed = True
            break
      if not engineIdDisplayed and saveAll:
         cmds.addCommand( 'no snmp-server engineID local' )

   def _addCmdIfNotEmpty( attr, cliName ):
      val = getattr( entity, attr )
      if val != "":
         cmds.addCommand( 'snmp-server %s %s' % ( cliName, val ) )
      elif saveAll:
         cmds.addCommand( 'no snmp-server %s' % cliName )

   stringAttrInfo = [
      ( 'chassisId', 'chassis-id' ),
      ( 'contact', 'contact' ),
      ( 'location', 'location' ) ]

   for attr in stringAttrInfo:
      _addCmdIfNotEmpty( attr[ 0 ], attr[ 1 ] )

   if entity.tcpTransport == False:
      cmds.addCommand( "no snmp-server transport tcp" )
   elif saveAll:
      cmds.addCommand( "default snmp-server transport tcp" )

   srcIntfs = entity.notificationSourceIntf
   for vrf in srcIntfs:
      n = srcIntfs[ vrf ]
      if vrf == DEFAULT_VRF and not saveAll:
         cmds.addCommand( "snmp-server local-interface %s" % n )
      else:
         cmds.addCommand( "snmp-server vrf %s local-interface %s" % ( vrf, n ) )
   if not srcIntfs and saveAll:
      cmds.addCommand( "no snmp-server vrf default local-interface" )

   views = entity.view
   for key in sorted( views.keys() ):
      v = views[ key ]
      subtrees = v.subtree
      for skey in sorted( subtrees.keys() ):
         st = subtrees[ skey ]
         cmds.addCommand( "snmp-server view %s %s %s" % ( v.viewName, st.root,
                                                         st.viewType ) )

   communities = entity.community
   for key in sorted( communities.keys() ):
      c = communities[ key ]
      cmd = "snmp-server community " + CliSave.sanitizedOutput( options,
            c.communityString )
      if c.view != "":
         cmd += " view " + c.view
      cmd += " " + c.access
      if c.acl6 != "":
         cmd += " ipv6 " + c.acl6
      if c.acl != "":
         cmd += " " + c.acl
      cmds.addCommand( cmd )

   groups = entity.group
   for key in sorted( groups.keys() ):
      g = groups[ key ]
      cmd = "snmp-server group %s %s" % ( g.groupName, g.protocolVersion )
      if g.protocolVersion == 'v3':
         cmd += " " + _authLevels[ g.authLevel ]
      if g.context != "":
         cmd += " context " + g.context
      if g.readView != "":
         cmd += " read " + g.readView
      if g.writeView != "":
         cmd += " write " + g.writeView
      if g.notifyView != "":
         cmd += " notify " + g.notifyView
      cmds.addCommand( cmd )

   users = entity.user
   for key in sorted( users.keys() ):
      u = users[ key ]
      cmd = "snmp-server user %s %s %s" % ( u.userName, u.group,
         u.protocolVersion )
      if u.protocolVersion == 'v3':
         if u.authType != 'authNone' and u.authLocalizedKey != "":
            cmd += " localized %s auth %s %s" % ( u.engineId,
                     _authTypes[ u.authType ],
                     CliSave.sanitizedOutput( options, u.authLocalizedKey ) )
         if u.privacyType != 'privacyNone' and u.privacyLocalizedKey != "":
            cmd += " priv %s %s" % ( _privTypes[ u.privacyType ],
                     CliSave.sanitizedOutput( options, u.privacyLocalizedKey ) )
      cmds.addCommand( cmd )

   remoteEngineIds = entity.remoteEngineId
   for key in sorted( remoteEngineIds.keys() ):
      e = remoteEngineIds[ key ]
      cmd = "snmp-server engineID remote %s" % ( e.spec.hostname )
      if e.spec.port != _defaultRemotePort or saveAll:
         cmd += " udp-port %d" % ( e.spec.port )
      cmd += " " + e.engineId
      cmds.addCommand( cmd )

   remoteUsers = entity.remoteUser
   for key in sorted( remoteUsers.keys() ):
      u = remoteUsers[ key ]
      cmd = "snmp-server user %s %s remote %s" % ( u.userName, u.userName,
               u.spec.hostname )
      if u.spec.port != _defaultRemotePort or saveAll:
         cmd += " udp-port %d" % ( u.spec.port )
      cmd += " v3"
      if u.authType != 'authNone' and u.authLocalizedKey != "":
         cmd += " localized %s auth %s %s" % ( u.engineId,
                  _authTypes[ u.authType ], u.authLocalizedKey )
      if u.privacyType != 'privacyNone' and u.privacyLocalizedKey != "":
         cmd += " priv %s %s" % ( _privTypes[ u.privacyType ],
                  u.privacyLocalizedKey )
      cmds.addCommand( cmd )

   hosts = entity.notificationSink
   for key in sorted( hosts.keys() ):
      h = hosts[ key ]
      cmd = "snmp-server host %s" % ( h.hostname )
      if h.vrf:
         cmd += " vrf " + h.vrf
      elif saveAll:
         cmd += " vrf " + DEFAULT_VRF
      if h.notificationType != 'trap' or saveAll:
         # turn "inform" into "informs" and "trap" into "traps"
         cmd += " " + h.notificationType + "s"
      cmd += " version " + h.protocolVersion[ 1 : ] # drop the leading 'v'
      if h.protocolVersion == 'v3':
         cmd += " " + _authLevels[ h.authLevel ]
      cmd += " " + CliSave.sanitizedOutput( options, h.securityName )
      if h.port != _defaultRemotePort or saveAll:
         cmd += " udp-port " + str( h.port )
      cmds.addCommand( cmd )

   if entity.notificationsGlobalEnabled == 'notifEnabled':
      cmds.addCommand( "snmp-server enable traps" )
   elif entity.notificationsGlobalEnabled == 'notifDisabled':
      cmds.addCommand( "no snmp-server enable traps" )
   elif saveAll:
      cmds.addCommand( "default snmp-server enable traps" )

   def addAllNotifCmds( notificationTypes, parentPath ):
      for name, n in sorted( notificationTypes.items() ):
         path = ' '.join( parentPath + [ name ] )
         if n.enabled == 'notifEnabled':
            cmds.addCommand( "snmp-server enable traps %s" % path )
         elif n.enabled == 'notifDisabled':
            cmds.addCommand( "no snmp-server enable traps %s" % path )
         elif saveAll:
            cmds.addCommand( "default snmp-server enable traps %s" % path )

         for trap, t in sorted( n.notification.items() ):
            if t.enabled == 'notifEnabled':
               cmds.addCommand( "snmp-server enable traps %s %s" %
                     ( path, trapToToken( trap, n.strip ) ) )
            elif t.enabled == 'notifDisabled':
               cmds.addCommand( "no snmp-server enable traps %s %s" %
                     ( path, trapToToken( trap, n.strip ) ) )
            elif saveAll:
               cmds.addCommand( "default snmp-server enable traps %s %s" %
                     ( path, trapToToken( trap, n.strip ) ) )
         addAllNotifCmds( n.subtype, parentPath + [ name ] )

   addAllNotifCmds( entity.notificationConfig, [] )

   disabledObjects = entity.disableObjectsConfig
   for objectName in sorted( disabledObjects.keys() ):
      disabledConfig = disabledObjects[ objectName ]
      if not disabledConfig.enabled:
         cmds.addCommand( "snmp-server objects %s disable" % ( objectName, ) )
      elif saveAll:
         cmds.addCommand( "no snmp-server objects %s disable" % ( objectName, ) )

   for oid, e in sorted( entity.extension.items() ):
      # pylint: disable-msg=E1101
      cmds.addCommand( "snmp-server extension %s %s%s" % ( oid,
         Url.filenameToUrl( e.handler, simpleFile=False ),
         " one-shot" if e.oneShot else "" ) )

   # See BUG1720.
   if not DEFAULT_VRF in entity.vrf:
      cmds.addCommand( "no snmp-server vrf %s" % DEFAULT_VRF )
   for vrf in sorted( entity.vrf ):
      if vrf != DEFAULT_VRF or saveAll:
         cmds.addCommand( "snmp-server vrf %s" % vrf )

   notifLogEntryLimit = entity.notificationLogEntryLimit
   if notifLogEntryLimit != entity.notificationLogEntryLimitDefault or saveAll:
      cmds.addCommand( 'snmp-server notification log entry limit %s' %
                       notifLogEntryLimit )

   # snmp-server qos dscp <dscpValue>
   if entity.dscpValue != entity.dscpValueDefault:
      cmds.addCommand( 'snmp-server qos dscp %s' % entity.dscpValue )
   elif saveAll:
      cmds.addCommand( 'snmp-server qos dscp %s' % entity.dscpValueDefault )

   if entity.transmitMsgSize != entity.transmitMsgSizeDefault or saveAll:
      cmds.addCommand( 'snmp-server transmit max-size %u' %
                       entity.transmitMsgSize )

@CliSave.saver( 'Snmp::NetSnmpConfig', 'snmp/netSnmpConfig' )
def saveSnmpAgentConfig( entity, root, sysdbRoot, options ):
   cmds = root[ 'Snmp.global' ]
   if entity.clientRecvBuf > 0:
      cmds.addCommand( "snmp-server net-snmp buffer client receive %s" %
                       entity.clientRecvBuf )
   if entity.clientSendBuf > 0:
      cmds.addCommand( "snmp-server net-snmp buffer client send %s" %
                       entity.clientSendBuf )
   if entity.serverRecvBuf > 0:
      cmds.addCommand( "snmp-server net-snmp buffer server receive %s" %
                       entity.serverRecvBuf )
   if entity.serverSendBuf > 0:
      cmds.addCommand( "snmp-server net-snmp buffer server send %s" %
                       entity.serverSendBuf )

@CliSave.saver( 'Acl::Input::CpConfig', 'acl/cpconfig/cli',
                requireMounts=( 'acl/config/cli', ) )
def saveSnmpAclConfig( aclCpConfig, root, sysdbRoot, options, requireMounts ):
   srcAddrACLsEnabled = Toggles.SnmpToggleLib.toggleSnmpSrcAddressACLsEnabled()
   if not srcAddrACLsEnabled:
      return
   cmds = root[ 'Snmp.global' ]
   # save for snmp-server service acl
   for aclType in ( 'ip', 'ipv6' ):
      serviceAcl = aclCpConfig.cpConfig[ aclType ].serviceAcl
      for vrfName, serviceAclVrfConfig in sorted( serviceAcl.iteritems() ):
         # since the configuration for snmp-udp and snmp-tcp is always
         # the same and is not known to the user, only snmp-udp is used here
         serviceAclConfig = serviceAclVrfConfig.service.get( 'snmp-udp' )
         if serviceAclConfig:
            cmd = 'snmp-server '
            cmd += 'ipv4' if aclType == 'ip' else aclType
            cmd += ' access-list ' + serviceAclConfig.aclName
            if vrfName != DEFAULT_VRF:
               cmd += " vrf " + vrfName
            cmds.addCommand( cmd )
