# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave
from RoutingIntfUtils import allRoutingProtocolIntfNames
from IntfCliSave import IntfConfigMode
import Tac
from CliSavePlugin.RouterGeneralCliSave import RouterGeneralCliSaveMode
from CliMode.SegmentRoutingMode import SrModeBase

srSidInvalid = Tac.Type( "Routing::SegmentRoutingCli::Constants" ).srSidInvalid

class SrMode( SrModeBase, CliSave.Mode ):
   def __init__( self, param ):
      SrModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

RouterGeneralCliSaveMode.addChildMode( SrMode )
SrMode.addCommandSequence( 'SegmentRouting.config' )

def getOrCreateSrModeInst( root ):
   generalMode = root[ RouterGeneralCliSaveMode ].getOrCreateModeInstance(
                                                "generalconfig" )
   return generalMode[ SrMode ].getOrCreateModeInstance( "segmentRoutingInstance" )

# Node-segment configuration is shared by OSPF-SR and ISIS-SR, originally the
# configuration was stored in Routing::Isis::Config and was handled by the
# respective Isis plugins. Now with both OSPF and ISIS sharing the config,
# SegmentRoutingCli implements it. To maintain the order of the configuration
# statements as before we insert the node-segment config statements in between
# the isis intf config statements (so that customers who upgrade do not see any
# difference in the running config).
IntfConfigMode.addCommandSequence( 'Sr.intf', after=[ 'Ira.ipIntf' ] )

@CliSave.saver( 'Routing::SegmentRoutingCli::Config', 'routing/sr/config',
                requireMounts=( 'interface/config/all',
                                'interface/status/all' ) )
def saveConfig( srConfig, root, sysdbRoot, options, requireMounts ):

   if options.saveAllDetail:
      rpIntfs = allRoutingProtocolIntfNames( sysdbRoot, includeEligible=True,
                                             requireMounts=requireMounts )
      intfs = set( rpIntfs + srConfig.intfConfig.keys() )
   elif options.saveAll:
      # Display Isis configuration on routing protocol interfaces as well as
      # switchports present in intfConfig collection.
      rpIntfs = allRoutingProtocolIntfNames( sysdbRoot,
                                             requireMounts=requireMounts )
      intfs = set( rpIntfs + srConfig.intfConfig.keys() )
   else:
      intfs = srConfig.intfConfig

   for intfName in intfs:
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intfName )
      cmds = mode[ 'Sr.intf' ]

      srIntf = srConfig.intfConfig.get( intfName, None )
      if srIntf and srIntf.srNodeSegmentIndex != srSidInvalid:
         cmds.addCommand( 'node-segment ipv4 index %d' % srIntf.srNodeSegmentIndex )
      elif options.saveAll:
         cmds.addCommand( 'no node-segment ipv4 index' )

      if srIntf and srIntf.srV6NodeSegmentIndex != srSidInvalid:
         cmds.addCommand( 'node-segment ipv6 index %d' %
                              srIntf.srV6NodeSegmentIndex )
      elif options.saveAll:
         cmds.addCommand( 'no node-segment ipv6 index' )

   if srConfig.enabled:
      _ = getOrCreateSrModeInst( root )

@CliSave.saver( 'Mpls::Config', 'routing/mpls/config' )
def saveTunnelIgpFecSharing( entity, root, sysdbRoot, options, requireMounts ):
   if not entity.tunnelIgpFecSharing:
      srMode = getOrCreateSrModeInst( root )
      cmds = srMode[ 'SegmentRouting.config' ]
      cmds.addCommand( 'fec sharing igp tunnel disabled' )
