# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

from CliMode.RsvpLer import (
      RsvpTeMode,
      RsvpPathSpecMode,
      RsvpTunnelSpecMode,
      )
import CliSave
from CliSavePlugin.TeCliSave import RouterGlobalTeMode
import Tac
from TypeFuture import TacLazyType
import Toggles.RsvpToggleLib

RsvpLerCliConfig = TacLazyType( 'Rsvp::RsvpLerCliConfig' )
RsvpLerConstants = TacLazyType( 'Rsvp::RsvpLerConstants' )
RsvpLerPathSpecType = TacLazyType( 'Rsvp::RsvpLerPathSpecType' )

pathSpecDynamicType = RsvpLerPathSpecType.pathSpecDynamicType
pathSpecExplicitType = RsvpLerPathSpecType.pathSpecExplicitType
tunnelSourceCli = TacLazyType( 'Rsvp::RsvpLerTunnelSource' ).tunnelSourceCli

class RsvpTeConfigMode( RsvpTeMode, CliSave.Mode ):
   def __init__( self, param ):
      RsvpTeMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

RouterGlobalTeMode.addChildMode( RsvpTeConfigMode )
RsvpTeConfigMode.addCommandSequence( 'RsvpLer.config' )

# pylint: disable-msg=W0223, abstract-method
class RsvpPathSpecConfigMode( RsvpPathSpecMode, CliSave.Mode ):
   def __init__( self, param ):
      RsvpPathSpecMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

RsvpTeConfigMode.addChildMode( RsvpPathSpecConfigMode )
RsvpPathSpecConfigMode.addCommandSequence( 'RsvpLer.pathSpecConfig' )

class RsvpTunnelSpecConfigMode( RsvpTunnelSpecMode, CliSave.Mode ):
   def __init__( self, param ):
      RsvpTunnelSpecMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

RsvpTeConfigMode.addChildMode( RsvpTunnelSpecConfigMode )
RsvpTunnelSpecConfigMode.addCommandSequence( 'RsvpLer.tunnelSpecConfig' )

_cliSavers = []

def rsvpLerCliSaver( func ):
   _cliSavers.append( func )

@rsvpLerCliSaver
def saveLocalInterface( config, cmds, saveAll, **kwargs ):
   if config.localIntf != '':
      cmds.addCommand( 'local-interface ' + config.localIntf )
   elif saveAll:
      cmds.addCommand( 'no local-interface' )

@rsvpLerCliSaver
def saveOptimizationGlobal( config, cmds, saveAll, **kwargs ):
   if not Toggles.RsvpToggleLib.toggleRsvpLerReoptimizationEnabled():
      return
   if config.optimizationInterval != RsvpLerConstants.optimizationIntervalDisabled:
      cmds.addCommand(
         'optimization interval ' + str( config.optimizationInterval ) + ' seconds' )
   elif saveAll:
      cmds.addCommand( 'no optimization' )

def savePathSpecConfig( mode, pathSpec, cmds, saveAll ):
   cliCmds = mode.pathSpecToCliCmds( pathSpec, saveAll )
   for cmd in cliCmds:
      cmds.addCommand( cmd )

def saveRsvpLerPathSpec( rsvpPathSpec, root, sysdbRoot, options ):
   pathSpecName = rsvpPathSpec.pathSpecName
   pathSpecType = rsvpPathSpec.pathSpecType
   teMode = root[ RouterGlobalTeMode ].getOrCreateModeInstance( "teInstance" )
   rsvpMode = teMode[ RsvpTeConfigMode ].getOrCreateModeInstance( "rsvpInstance" )
   param = ( pathSpecName, pathSpecType )
   mode = rsvpMode[ RsvpPathSpecConfigMode ].getOrCreateModeInstance( param )
   cmds = mode[ 'RsvpLer.pathSpecConfig' ]
   savePathSpecConfig( mode, rsvpPathSpec, cmds, options.saveAll )

def saveTunnelSpecConfig( mode, tunnelSpec, cmds, saveAll ):
   cliCmds = mode.tunnelSpecToCliCmds( tunnelSpec, saveAll )
   for cmd in cliCmds:
      cmds.addCommand( cmd )

def saveRsvpLerTunnelSpec( rsvpTunnelSpec, root, sysdbRoot, options ):
   teMode = root[ RouterGlobalTeMode ].getOrCreateModeInstance( "teInstance" )
   rsvpMode = teMode[ RsvpTeConfigMode ].getOrCreateModeInstance( "rsvpInstance" )
   param = rsvpTunnelSpec.tunnelName
   mode = rsvpMode[ RsvpTunnelSpecConfigMode ].getOrCreateModeInstance( param )
   cmds = mode[ 'RsvpLer.tunnelSpecConfig' ]
   saveTunnelSpecConfig( mode, rsvpTunnelSpec, cmds, options.saveAll )

@CliSave.saver( 'Rsvp::RsvpLerCliConfig', RsvpLerCliConfig.mountPath,
                requireMounts=( 'routing/hardware/mpls/capability',
                                'routing/hardware/route/status' ) )
def saveRsvpLerCliConfig( rsvpLerCliConfig, root, sysdbRoot, options,
                          requireMounts ):
   saveAll = options.saveAll
   hwCapability = requireMounts[ 'routing/hardware/mpls/capability' ]
   routingHwRouteStatus = requireMounts[ 'routing/hardware/route/status' ]

   if ( hwCapability.mplsSupported is False or
        routingHwRouteStatus.hierarchicalFecsEnabled is False ):
      return

   if not rsvpLerCliConfig.isDefault() or saveAll:
      teMode = root[ RouterGlobalTeMode ].getOrCreateModeInstance( "teInstance" )
      mode = teMode[ RsvpTeConfigMode ].getOrCreateModeInstance( "rsvpInstance" )
      cmds = mode[ 'RsvpLer.config' ]

      for saver in _cliSavers:
         saver( rsvpLerCliConfig, cmds, saveAll, requireMounts=requireMounts,
                options=options )
      # Save pathSpecs
      cliPathSpecColl = \
               rsvpLerCliConfig.pathSpecCollTable.pathSpecColl.get( tunnelSourceCli )
      if cliPathSpecColl:
         for pathSpec in cliPathSpecColl.pathSpec.values():
            saveRsvpLerPathSpec( pathSpec, root, sysdbRoot, options )
      # Save tunnelSpecs
      cliTunnelSpecColl = rsvpLerCliConfig.tunnelSpecCollTable.tunnelSpecColl.get(
                             tunnelSourceCli )
      if cliTunnelSpecColl:
         for tunnelSpec in cliTunnelSpecColl.tunnelSpec.values():
            saveRsvpLerTunnelSpec( tunnelSpec, root, sysdbRoot, options )
