# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
from CliMode.Rsvp import RsvpMode
import CliSave
from RsvpLib import bandwidthHumanReadable
import IpLibConsts
import Toggles.RsvpToggleLib

EthType = Tac.Type( "Arnet::EthType" )
IpGenAddr = Tac.Type( "Arnet::IpGenAddr" )

RsvpCliConfig = Tac.Type( 'Rsvp::RsvpCliConfig' )
RsvpLspRequestId = Tac.Type( 'Rsvp::RsvpLspRequestId' )
FrrMode = Tac.Type( 'Rsvp::Cli::FrrMode' )
SrlgMode = Tac.Type( 'Rsvp::Cli::SrlgMode' )
LabelLocalTerminationMode = Tac.Type( 'Rsvp::Cli::LabelLocalTerminationMode' )

class RsvpConfigMode( RsvpMode, CliSave.Mode ):
   def __init__( self, param ):
      RsvpMode.__init__( self )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( RsvpConfigMode, after=[ 'Mpls.config' ] )
RsvpConfigMode.addCommandSequence( 'Rsvp.config' )

_cliSavers = []

def rsvpCliSaver( func ):
   _cliSavers.append( func )

@rsvpCliSaver
def saveRefreshInterval( config, cmds, saveAll, **kwargs ):
   if config.refreshInterval != config.refreshIntervalDefault or saveAll:
      cmds.addCommand( 'refresh interval %d' % config.refreshInterval )

@rsvpCliSaver
def saveRefreshMethod( config, cmds, saveAll, **kwargs ):
   if config.refreshMethod == 'refreshExplicit':
      cmds.addCommand( 'refresh method explicit' )
   elif saveAll:
      cmds.addCommand( 'refresh method bundled' )

@rsvpCliSaver
def saveHelloIntervalMultiplier( config, cmds, saveAll, **kwargs ):
   if config.helloInterval == config.helloIntervalDisabled:
      cmds.addCommand( 'no hello interval' )
   elif ( config.helloInterval != config.helloIntervalDefault or
        config.helloMultiplier != config.helloMultiplierDefault or
        saveAll ):
      cmds.addCommand( 'hello interval %d multiplier %d' % (
                           config.helloInterval, config.helloMultiplier ) )

@rsvpCliSaver
def saveAuthenticationType( config, cmds, saveAll, **kwargs ):
   if config.authType == 'md5':
      cmds.addCommand( 'authentication type md5' )
   elif saveAll:
      cmds.addCommand( 'authentication type none' )

@rsvpCliSaver
def saveAuthenticationWindow( config, cmds, saveAll, **kwargs ):
   if config.authSeqWindowSize != config.authSeqWindowSizeDefault or saveAll:
      cmds.addCommand( 'authentication sequence-number window %d' %
                       config.authSeqWindowSize )

@rsvpCliSaver
def saveAuthenticationPassword( config, cmds, saveAll, **kwargs ):
   options = kwargs[ 'options' ]
   for index, passwd in config.encodedPassword.iteritems():
      passwd = CliSave.sanitizedOutput( options, passwd )
      cmds.addCommand( 'authentication index %d password 7 %s' % ( index, passwd ) )

   if config.activePasswordIndex != 0:
      cmds.addCommand( 'authentication index %d active' %
                       config.activePasswordIndex )

@rsvpCliSaver
def saveServiceAclConfig( config, cmds, saveAll, requireMounts, **kwargs ):
   aclCpConfig = requireMounts[ 'acl/cpconfig/cli' ]
   for aclType in [ 'ip', 'ipv6' ]:
      serviceAclVrfConfig = aclCpConfig.cpConfig[
                              aclType ].serviceAcl.get( IpLibConsts.DEFAULT_VRF )
      if ( serviceAclVrfConfig and
           serviceAclVrfConfig.service.get( 'rsvp' ) and
           serviceAclVrfConfig.service.get( 'rsvp' ).aclName ):
         cmds.addCommand( '%s access-group %s' % ( aclType,
                           serviceAclVrfConfig.service[ 'rsvp' ].aclName ) )
      elif saveAll:
         cmds.addCommand( 'no %s access-group' % aclType )

@rsvpCliSaver
def saveFastRerouteMode( config, cmds, saveAll, **kwargs ):
   if config.frrMode == FrrMode.frrLinkProtection:
      cmds.addCommand( 'fast-reroute mode link-protection' )
   elif config.frrMode == FrrMode.frrNodeProtection:
      cmds.addCommand( 'fast-reroute mode node-protection' )
   elif saveAll:
      cmds.addCommand( 'fast-reroute mode none' )

@rsvpCliSaver
def saveSrlgMode( config, cmds, saveAll, **kwargs ):
   if config.srlgMode == SrlgMode.srlgStrict:
      cmds.addCommand( 'srlg strict' )
   elif config.srlgMode == SrlgMode.srlgLoose:
      cmds.addCommand( 'srlg' )
   elif saveAll:
      cmds.addCommand( 'no srlg' )

@rsvpCliSaver
def saveLabelLocalTermination( config, cmds, saveAll, **kwargs ):
   if not Toggles.RsvpToggleLib.toggleRsvpExplicitNullEnabled():
      return
   if config.labelLocalTerminationMode == LabelLocalTerminationMode.explicitNull:
      cmds.addCommand( 'label local-termination explicit-null' )
   elif saveAll:
      cmds.addCommand( 'label local-termination implicit-null' )

@rsvpCliSaver
def saveFastRerouteReversionMode( config, cmds, saveAll, **kwargs ):
   if config.reversionMode == 'reversionLocal':
      cmds.addCommand( 'fast-reroute reversion local' )
   elif saveAll:
      cmds.addCommand( 'fast-reroute reversion global' )

@rsvpCliSaver
def savePreemptionTimer( config, cmds, saveAll, **kwargs ):
   if config.preemptionTimer == config.preemptionTimerDisabled:
      cmds.addCommand( 'preemption method hard' )
   elif config.preemptionTimer != config.preemptionTimerDefault or saveAll:
      cmds.addCommand( 'preemption method soft timer %d' %
                       config.preemptionTimer )

@rsvpCliSaver
def saveMtuSignalingEnabled( config, cmds, saveAll, **kwargs ):
   if config.mtuSignalingEnabled:
      cmds.addCommand( 'mtu signaling' )
   elif saveAll:
      cmds.addCommand( 'no mtu signaling' )

@rsvpCliSaver
def saveDebugTunnels( config, cmds, saveAll, **kwargs ):
   for req in config.debugLspRequestColl.debugLspReq.values():
      lspReqId = req.lspReqId
      cmdParts = [ 'debug', 'tunnel' ]
      cmdParts.append( 'id %d' % lspReqId.reqId )
      cmdParts.append( 'sub-id %d' % lspReqId.reqSubId )
      cmdParts.append( 'src %s' % str( req.srcIp ) )
      cmdParts.append( 'dst %s' % str( req.dstIp ) )
      cliEthTypeToStr = { EthType.ethTypeIp : 'ip',
                          EthType.ethTypeIp6 : 'ip6',
                          EthType.ethTypeMpls : 'mpls' }
      cmdParts.append( 'ether-type %s' % cliEthTypeToStr.get( req.ethType, 'ip' ) )
      lpdStr = 'off'
      if req.nodeProtectionDesired:
         lpdStr = 'node'
      elif req.localProtectionDesired:
         lpdStr = 'link'
      cmdParts.append( 'lpd %s' % lpdStr )
      cmdParts.append( 'session-name "%s"' % req.sessionName )
      if req.bandwidth > 0.0:
         bw, unit = bandwidthHumanReadable( req.bandwidth * 8 )
         cmdParts.append( 'bandwidth %.2f %s' % ( bw, unit ) )
      if req.softPreemptionDesired:
         preemptStr = 'soft'
      else:
         preemptStr = 'hard'
      cmdParts.append( 'preemption %s' % preemptStr )
      if req.setupPriority < 7:
         cmdParts.append( 'setup-prio %d' % ( req.setupPriority ) )
      if req.holdPriority > 0:
         cmdParts.append( 'hold-prio %d' % ( req.holdPriority ) )
      cmdParts.append( 'hops' )
      for hop in req.hop.values():
         cmdParts.append( str( hop ) )
      cmds.addCommand( ' '.join( cmdParts ) )

# Always save this at end
@rsvpCliSaver
def saveShutdown( config, cmds, saveAll, **kwargs ):
   if config.enabled:
      cmds.addCommand( 'no shutdown' )
   elif saveAll:
      cmds.addCommand( 'shutdown' )

@CliSave.saver( 'Rsvp::RsvpCliConfig', RsvpCliConfig.mountPath,
                requireMounts=( 'routing/hardware/mpls/capability',
                                'acl/cpconfig/cli' ) )
def saveRsvpCliConfig( rsvpCliConfig, root, sysdbRoot, options, requireMounts ):
   saveAll = options.saveAll
   hwCapability = requireMounts[ 'routing/hardware/mpls/capability' ]

   if hwCapability.mplsSupported is False:
      return

   mode = root[ RsvpConfigMode ].getOrCreateModeInstance( 'mpls-rsvp' )
   cmds = mode[ 'Rsvp.config' ]

   for saver in _cliSavers:
      saver( rsvpCliConfig, cmds, saveAll, requireMounts=requireMounts,
             options=options )
