#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import Tac
from CliMode.Rpki import ( RpkiCacheMode,
                           RpkiTransportTcpMode,
                           RpkiTransportTlsMode,
                           RpkiOriginValidationBaseMode )
import CliSave
from CliSavePlugin.RoutingBgpCliSave import ( RouterBgpBaseConfigMode,
                                              neighborRpkiHook )
from IpLibConsts import DEFAULT_VRF
from RouteMapLib import isAsdotConfigured

import Toggles.RpkiToggleLib

ovMethodEnum = Tac.Type( "Routing::Bgp::Rpki::OriginValidationMethod" )

class RpkiCacheConfigCliSaveMode( RpkiCacheMode, CliSave.Mode ):
   def __init__( self, param ):
      RpkiCacheMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

RouterBgpBaseConfigMode.addChildMode( RpkiCacheConfigCliSaveMode )
RpkiCacheConfigCliSaveMode.addCommandSequence( 'Bgp.rpki.config' )

class RpkiTransportTcpCliSaveMode( RpkiTransportTcpMode,
                                        CliSave.Mode ):
   def __init__( self, param ):
      RpkiTransportTcpMode.__init__( self )
      CliSave.Mode.__init__( self, param )

RpkiCacheConfigCliSaveMode.addChildMode( RpkiTransportTcpCliSaveMode )
RpkiTransportTcpCliSaveMode.addCommandSequence(
      'Bgp.rpki.transport-tcp.config' )

class RpkiTransportTlsCliSaveMode( RpkiTransportTlsMode, CliSave.Mode ):
   def __init__( self, param ):
      RpkiTransportTlsMode.__init__( self )
      CliSave.Mode.__init__( self, param )

RpkiCacheConfigCliSaveMode.addChildMode( RpkiTransportTlsCliSaveMode )
RpkiTransportTlsCliSaveMode.addCommandSequence(
      'Bgp.rpki.transport-tls.config' )

#------------------------------------------------------------------------------------
# bgp rpki origin-validation mode
#------------------------------------------------------------------------------------
class RpkiOriginValidationCliSaveMode( RpkiOriginValidationBaseMode, CliSave.Mode ):
   def __init__( self, param ):
      RpkiOriginValidationBaseMode.__init__( self )
      CliSave.Mode.__init__( self, param )

RouterBgpBaseConfigMode.addChildMode( RpkiOriginValidationCliSaveMode )

RpkiOriginValidationCliSaveMode.addCommandSequence( 'Bgp.rpki.origin-validation' )

def saveRpkiOriginValidationConfig( bgpConfig, saveAll=False ):
   cmds = []

   if bgpConfig.rpkiEbgpOvMethod.isSet:
      method = bgpConfig.rpkiEbgpOvMethod.value
      if method == ovMethodEnum.ovLocal:
         cmds.append( 'ebgp local' )
      elif method == ovMethodEnum.ovCommunity:
         cmds.append( 'ebgp community' )
      elif method == ovMethodEnum.ovPreferCommunity:
         cmds.append( 'ebgp prefer-community' )
      else:
         assert False, "Unrecognized method %s" % method
   elif saveAll:
      cmds.append( 'no ebgp' )

   if bgpConfig.rpkiEbgpOvSend == 'isTrue':
      cmds.append( 'ebgp send' )
   elif saveAll:
      cmds.append( 'no ebgp send' )

   if bgpConfig.rpkiIbgpOvMethod.isSet:
      method = bgpConfig.rpkiIbgpOvMethod.value
      if method == ovMethodEnum.ovLocal:
         cmds.append( 'ibgp local' )
      elif method == ovMethodEnum.ovCommunity:
         cmds.append( 'ibgp community' )
      elif method == ovMethodEnum.ovPreferCommunity:
         cmds.append( 'ibgp prefer-community' )
      else:
         assert False, "Unrecognized method %s" % method
   elif saveAll:
      cmds.append( 'no ibgp' )

   if bgpConfig.rpkiIbgpOvSend == 'isTrue':
      cmds.append( 'ibgp send' )
   elif saveAll:
      cmds.append( 'no ibgp send' )

   if bgpConfig.rpkiOvRouteMap:
      cmds.append( 'validation route-map %s' % bgpConfig.rpkiOvRouteMap )
   elif saveAll:
      cmds.append( 'no validation route-map' )

   if bgpConfig.rpkiRedistOvMethod.isSet:
      method = bgpConfig.rpkiRedistOvMethod.value
      if method == ovMethodEnum.ovLocal:
         cmds.append( 'redistributed local' )
      else:
         assert False, "Unrecognized method %s" % method
   elif saveAll:
      cmds.append( 'no redistributed' )

   if bgpConfig.rpkiRedistOvRouteMap:
      cmds.append( 'redistributed validation route-map %s' %
                   bgpConfig.rpkiRedistOvRouteMap )
   elif saveAll:
      cmds.append( 'no redistributed validation route-map' )

   return cmds

def saveCacheConfig( cacheId, cacheConfig, saveAll=False ):
   cmds = []
   if cacheConfig.host:
      cmd = "host {}".format( cacheConfig.host )
      if cacheConfig.vrf != DEFAULT_VRF or saveAll:
         cmd += " vrf {}".format( cacheConfig.vrf )
      if ( cacheConfig.transportConfig.transportAuthenticationType == 'tcp' and
           cacheConfig.port != cacheConfig.tcpPortDefault ) or saveAll:
         cmd += " port {}".format( cacheConfig.port )
      cmds.append( cmd )
   elif saveAll:
      cmds.append( 'no host' )

   if cacheConfig.preference != cacheConfig.preferenceDefault or saveAll:
      cmds.append( 'preference {}'.format( cacheConfig.preference ) )

   if cacheConfig.refreshInterval != 0:
      cmds.append( 'refresh-interval {}'.format( cacheConfig.refreshInterval ) )
   elif saveAll:
      cmds.append( 'no refresh-interval' )

   if cacheConfig.retryInterval != 0:
      cmds.append( 'retry-interval {}'.format( cacheConfig.retryInterval ) )
   elif saveAll:
      cmds.append( 'no retry-interval' )

   if cacheConfig.expireInterval != 0:
      cmds.append( 'expire-interval {}'.format( cacheConfig.expireInterval ) )
   elif saveAll:
      cmds.append( 'no expire-interval' )

   if cacheConfig.sourceIntf != cacheConfig.sourceIntfDefault:
      cmds.append( 'local-interface {}'.format( cacheConfig.sourceIntf ) )
   elif saveAll:
      cmds.append( 'no local-interface' )
   return cmds

def saveTransportTcp( cacheConfig, saveAll=False ):
   cmds = []
   if ( cacheConfig.transportConfig.tcpKeepaliveOptions !=
        cacheConfig.transportConfig.tcpKeepaliveDefault ):
      tcpKeepaliveOptions = cacheConfig.transportConfig.tcpKeepaliveOptions
      cmd = 'tcp keepalive {} {} {}'.format(
            tcpKeepaliveOptions.idleTime,
            tcpKeepaliveOptions.probeInterval,
            tcpKeepaliveOptions.probeCount )
      cmds.append( cmd )
   elif saveAll:
      cmds.append( 'no tcp keepalive' )
   return cmds

def saveTransportTls( cacheConfig, saveAll=False ):
   cmds = []
   sslProfileName = cacheConfig.transportConfig.sslProfileName
   if sslProfileName != cacheConfig.transportConfig.sslProfileNameDefault:
      cmds.append( "ssl profile {}".format( sslProfileName ) )
   elif saveAll:
      cmds.append( 'no ssl profile' )
   return cmds

@CliSave.saver( 'Rpki::CacheConfigDir', 'routing/rpki/cache/config',
                requireMounts=( 'routing/bgp/config', 'routing/bgp/asn/config' ) )
def saveRpkiCacheConfig( cacheConfigDir, root, sysdbRoot, options, requireMounts ):
   bgpConfig = requireMounts[ 'routing/bgp/config' ]
   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]
   # Adding this to not add "routing bgp 0" to running config when bgp is not
   # configured.
   if bgpConfig.asNumber == 0:
      return

   bgpMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
      bgpConfig.asNumber, isAsdotConfigured( asnConfig ) ) )

   for cacheId in sorted( cacheConfigDir.cacheConfig ):
      cacheConfig = cacheConfigDir.cacheConfig[ cacheId ]
      cacheMode = bgpMode[ RpkiCacheConfigCliSaveMode ].getOrCreateModeInstance(
         cacheId.stringValue )
      cacheModeCmds = cacheMode[ 'Bgp.rpki.config' ]
      cmds = saveCacheConfig( cacheId, cacheConfig, options.saveAll )
      for cmd in cmds:
         cacheModeCmds.addCommand( cmd )
      if Toggles.RpkiToggleLib.toggleRpkiCacheTcpKeepaliveEnabled():
         if cacheConfig.transportConfig.transportAuthenticationType == 'tcp':
            transportTcpMode = cacheMode[
                  RpkiTransportTcpCliSaveMode ].getSingletonInstance()
            transportModeCmds = transportTcpMode[ 'Bgp.rpki.transport-tcp.config' ]
            cmds = saveTransportTcp( cacheConfig, options.saveAll )
            for cmd in cmds:
               transportModeCmds.addCommand( cmd )
      if Toggles.RpkiToggleLib.toggleRpkiSecureTransportToggleEnabled():
         if cacheConfig.transportConfig.transportAuthenticationType == 'tls':
            transportTlsMode = cacheMode[
               RpkiTransportTlsCliSaveMode ].getSingletonInstance()
            tlsModeCmds = transportTlsMode[ 'Bgp.rpki.transport-tls.config' ]
            cmds = saveTransportTls( cacheConfig, options.saveAll )
            for cmd in cmds:
               tlsModeCmds.addCommand( cmd )

@CliSave.saver( 'Routing::Bgp::Config', 'routing/bgp/config',
                requireMounts=( 'routing/bgp/asn/config', ) )
def saveRpkiConfig( bgpConfig, root, sysdbRoot, options, requireMounts ):
   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]
   # If a BGP instance is not configured, do not generate any config
   if bgpConfig.asNumber == 0:
      return

   bgpMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
      bgpConfig.asNumber, isAsdotConfigured( asnConfig ), ) )
   cmds = saveRpkiOriginValidationConfig( bgpConfig, options.saveAll )
   if cmds:
      rpkiMode = bgpMode[ RpkiOriginValidationCliSaveMode ].getSingletonInstance()
      rpkiModeCmds = rpkiMode[ 'Bgp.rpki.origin-validation' ]
      for cmd in cmds:
         rpkiModeCmds.addCommand( cmd )

# CliSave for neighbor rpki origin validation settings
def saveNeighborRpkiOvMethod( peer, peerConfig, saveAll ):
   ovMethodCmd = 'neighbor %s rpki origin-validation' % peer
   if peerConfig.rpkiOvMethodPresent:
      if peerConfig.rpkiOvMethod == ovMethodEnum.ovLocal:
         ovMethodCmd += ' local'
      elif peerConfig.rpkiOvMethod == ovMethodEnum.ovCommunity:
         ovMethodCmd += ' community'
      elif peerConfig.rpkiOvMethod == ovMethodEnum.ovPreferCommunity:
         ovMethodCmd += ' prefer-community'
      elif peerConfig.rpkiOvMethod == ovMethodEnum.ovDisabled:
         ovMethodCmd += ' disabled'
      else:
         assert False, "Unrecognized method %s" % peerConfig.rpkiOvMethod
      return ovMethodCmd
   elif saveAll and not peerConfig.isPeerGroupPeer:
      return 'default ' + ovMethodCmd
   return None

def saveNeighborRpkiOvSend( peer, peerConfig, saveAll ):
   ovSendCmd = 'neighbor %s rpki origin-validation send' % peer
   if peerConfig.rpkiOvSendPresent:
      if not peerConfig.rpkiOvSend:
         ovSendCmd += ' disabled'
      return ovSendCmd
   elif saveAll and not peerConfig.isPeerGroupPeer:
      return 'default ' + ovSendCmd
   return None

neighborRpkiHook.addExtension( saveNeighborRpkiOvMethod )
neighborRpkiHook.addExtension( saveNeighborRpkiOvSend )
