# Copyright (c) 2006-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements the CliSave code for routing rip commands
#-------------------------------------------------------------------------------
import CliSave, Tac
from RoutingIntfUtils import allRoutingProtocolIntfNames
from IntfCliSave import IntfConfigMode
from RibCliLib import protoDict
from CliMode.Rip import RipMode
from ReversibleSecretCli import encodeKey
import Toggles.RipToggleLib

#-------------------------------------------------------------------------------
# Object used for saving commands in router-rip mode.
#-------------------------------------------------------------------------------
class RouterRipConfigMode( RipMode, CliSave.Mode ):

   def __init__( self, param ):
      RipMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( RouterRipConfigMode,
                                       after=[ IntfConfigMode ] )
RouterRipConfigMode.addCommandSequence( 'Rip.instance' )

IntfConfigMode.addCommandSequence( 'Rip.intf', after=[ 'Ira.ipIntf' ] )

def saveIntfConfig( ripConfig, entity, root, sysdbRoot, options, requireMounts ):
   mode = root[ IntfConfigMode ].getOrCreateModeInstance( entity.name )
   cmds = mode[ 'Rip.intf' ]

   if entity.version != entity.versionDefault:
      cmds.addCommand( 'rip v2 multicast disable' )
   elif options.saveAll:
      cmds.addCommand( 'no rip v2 multicast disable' )

   if Toggles.RipToggleLib.toggleRipAuthEnabled():
      sanitize = lambda word: CliSave.sanitizedOutput( options, word )
      if entity.authMode != entity.authModeDefault:
         cmd = 'rip authentication algorithm %s' % entity.authMode
         cmd += ' tx key-id %d' % entity.authTxKeyid
         if entity.authRxDisabled:
            cmd += ' rx-disabled'
         cmds.addCommand( cmd )
      elif options.saveAll:
         cmds.addCommand( 'no rip authentication algorithm md5 tx key-id' )

      securityConfig = requireMounts[ 'mgmt/security/config' ]
      for keyid in sorted( entity.authKeys ):
         authKey = entity.authKeys[ keyid ]
         useKey = None if securityConfig.commonKeyEnabled else \
                  '%s_md5_%d' % ( entity.intf, authKey.id )
         authKeySanitized = sanitize( encodeKey( authKey.authKey, key=useKey,
                                                 algorithm='MD5' ) )
         cmds.addCommand( 'rip authentication algorithm md5 key-id %d key 7 %s' %
                          ( authKey.id, authKeySanitized ) )

def _saveInstanceConfig( entity, root, ripConfig, sysdbRoot, saveAll,
                         saveAllDetail, requireMounts ):

   mode = root[ RouterRipConfigMode ].getOrCreateModeInstance( entity.vrfName )
   cmds = mode[ 'Rip.instance' ]

   if entity.metric != entity.metricDefault or saveAll:
      cmds.addCommand( 'metric default %d' % entity.metric )

   if entity.preference != entity.preferenceDefault or saveAll:
      cmds.addCommand( 'distance %d' % entity.preference)

   if ( saveAll or
        entity.updateTime != entity.updateTimeDefault or
        entity.expireTime != entity.expireTimeDefault or
        entity.garbageTime != entity.garbageTimeDefault ):
      cmds.addCommand( 'timers %d %d %d' % (
         entity.updateTime, entity.expireTime, entity.garbageTime ) )

   for network in sorted( entity.network ):
      # BUG25925
      cmds.addCommand( 'network %s' % network.stringValue )

   for proto in entity.redistributeConfig:
      command = ''
      if ( proto =='protoOspf' ) or ( proto == 'protoOspfAse' ):
         command += 'redistribute ospf'
         if entity.redistributeConfig[ proto ].routeType != '': 
            command += ' match %s' % entity.redistributeConfig[ proto ].routeType
      else:
         command += 'redistribute %s' % protoDict[ proto ]
            
      if entity.redistributeConfig[ proto ].rmc != "":
         command += ' route-map %s' % entity.redistributeConfig[ proto ].rmc
      cmds.addCommand( command )

   for intf in entity.distListIn:
      command = 'distribute-list in route-map %s' % \
          entity.distListIn[ intf ].rmc
      if intf != '':
         command += ' %s' % intf
      cmds.addCommand( command )
   
   for intf in entity.distListOut:
      command = 'distribute-list out route-map %s' % \
          entity.distListOut[ intf ].rmc
      if intf != '':
         command += ' %s' % intf
      cmds.addCommand( command )

   # router rip always shows shutdown command
   if entity.shutdown:
      cmds.addCommand( 'shutdown' )
   else:
      cmds.addCommand( 'no shutdown' )

   if Toggles.RipToggleLib.toggleRipEcmpEnabled():
      if entity.ecmp != entity.ecmpDefault:
         cmds.addCommand( 'ecmp' )
      elif saveAll:
         cmds.addCommand( 'no ecmp' )

   aclCpConfig = requireMounts[ 'acl/cpconfig/cli' ]
   serviceAclVrfConfig = aclCpConfig.cpConfig['ip'].serviceAcl.get( entity.vrfName )
   if ( serviceAclVrfConfig and
        serviceAclVrfConfig.service.get( 'rip' ) and
        serviceAclVrfConfig.service[ 'rip' ].aclName ):
      cmds.addCommand( 'ip access-group %s' % 
                       serviceAclVrfConfig.service[ 'rip' ].aclName )
   elif saveAll:
      cmds.addCommand( 'no ip access-group' )

#-------------------------------------------------------------------------------
# save config under global config mode
#-------------------------------------------------------------------------------

CliSave.GlobalConfigMode.addCommandSequence( 'Ip.Rip',
                                             after=[ RouterRipConfigMode ] )

@CliSave.saver( 'Routing::Rip::Config', 'routing/rip/config',
                requireMounts=( 'interface/config/all',
                                'interface/status/all',
                                'acl/cpconfig/cli',
                                'mgmt/security/config' ) )
def saveConfig( ripConfig, root, sysdbRoot, options,
                requireMounts ):

   saveAll = options.saveAll
   saveAllDetail = options.saveAllDetail

   for vrfName in sorted( ripConfig.instanceConfig, key=str ):
                                 
      ripDisabled = ripConfig.instanceConfig[ vrfName ].shutdown
      if ripDisabled and not saveAllDetail:
         # Change saveAll to be set only if Rip is enabled or
         # saveAllDetail is set.
         saveAll = False

      _saveInstanceConfig( ripConfig.instanceConfig[ vrfName ],
            root, ripConfig, sysdbRoot, saveAll, saveAllDetail, 
                           requireMounts )

   if saveAllDetail:
      intfs = allRoutingProtocolIntfNames( sysdbRoot, includeEligible=True,
                                           requireMounts=requireMounts )
   elif saveAll:
      # Display Rip configuration on routing protocol interfaces as well as 
      # switchports present in intfConfig collection.
      intfs = set(
            allRoutingProtocolIntfNames( sysdbRoot, requireMounts=requireMounts ) +
            ripConfig.intfConfig.keys() )
   else:
      intfs = ripConfig.intfConfig
 
   # save config for each interface
   for intfName in intfs:
      intfConfig = ripConfig.intfConfig.get( intfName )
      if not intfConfig:
         if saveAll:
            intfConfig = Tac.newInstance( 'Routing::Rip::IntfConfig', intfName )
         else:
            continue
      saveIntfConfig( ripConfig, intfConfig, root, sysdbRoot, options,
                      requireMounts )
