# Copyright (c) 2006-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements the CliSave code routing ospf commands
#-------------------------------------------------------------------------------
import CliSave, Arnet, Tac
from IntfCliSave import IntfConfigMode
import os
from ReversibleSecretCli import encodeKey
from RibCliLib import protoDict
import IpUtils
from CliMode.Ospf import (
      RoutingOspfMode,
      RoutingOspfTeMode,
      RoutingOspfSrMplsMode,
      RoutingOspfGeneralMode,
      )
from RoutingIntfUtils import (
      isLoopback,
      isManagement,
      allIpIntfNames,
      allRoutingProtocolIntfNames,
      )
from IpLibConsts import DEFAULT_VRF
import OspfConsts
import Toggles.OspfToggleLib

ISIS_LEVEL_MAP = { 'level1' : 'level-1',
                   'level2' : 'level-2',
                   'level1_2' : 'level-1-2' }

#-------------------------------------------------------------------------------
# Object used for saving commands in router-ospf mode.
#-------------------------------------------------------------------------------
class RouterOspfConfigMode( RoutingOspfMode, CliSave.Mode ):
   def __init__( self, param ):
      RoutingOspfMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class RouterOspfGeneralConfigMode( RoutingOspfGeneralMode, CliSave.Mode ):
   def __init__( self, param ):
      RoutingOspfGeneralMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class RouterOspfTeConfigMode( RoutingOspfTeMode, CliSave.Mode ):
   def __init__( self, param ):
      RoutingOspfTeMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class RouterOspfSrMplsConfigMode( RoutingOspfSrMplsMode, CliSave.Mode ):
   def __init__( self, param ):
      RoutingOspfSrMplsMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( RouterOspfGeneralConfigMode,
                                       after=[ IntfConfigMode ] )
RouterOspfGeneralConfigMode.addCommandSequence( 'Ospf.general' )

CliSave.GlobalConfigMode.addChildMode( RouterOspfConfigMode,
                                       after=[ RouterOspfGeneralConfigMode ] )
RouterOspfConfigMode.addCommandSequence( 'Ospf.instance' )

IntfConfigMode.addCommandSequence( 'Ospf.intf', after=[ 'Ira.ipIntf' ] )

RouterOspfConfigMode.addChildMode( RouterOspfTeConfigMode )
RouterOspfTeConfigMode.addCommandSequence( 'Ospf.te' )

RouterOspfConfigMode.addChildMode( RouterOspfSrMplsConfigMode )
RouterOspfSrMplsConfigMode.addCommandSequence( 'Ospf.srMpls' )

def saveIntfConfig( entity, root, sysdbRoot, options, requireMounts ):
   if isLoopback( entity.name ):
      saveLoopbackIntfConfig( entity, root, sysdbRoot, options )
   else:
      saveNonLoopbackIntfConfig( entity, root, sysdbRoot, options, 
                                 requireMounts )

def saveLoopbackIntfConfig( entity, root, sysdbRoot, options ):
   mode = root[ IntfConfigMode ].getOrCreateModeInstance( entity.name )
   cmds = mode[ 'Ospf.intf' ]
   saveAll = options.saveAll

   if entity.cost != entity.costDefault:
      cmds.addCommand( 'ip ospf cost %d' % entity.cost )
   elif saveAll:
      cmds.addCommand( 'no ip ospf cost' )

   if entity.areaIdPresent != entity.areaIdPresentDefault:
      cmds.addCommand( 'ip ospf area %s' % entity.areaId )

def saveNonLoopbackIntfConfig( entity, root, sysdbRoot, options, requireMounts ):
   mode = root[ IntfConfigMode ].getOrCreateModeInstance( entity.name )
   cmds = mode[ 'Ospf.intf' ]
   saveAll = options.saveAll

   if entity.bfdIntfState == 'ospfIntfBfdEnabled':
      cmds.addCommand( 'ip ospf neighbor bfd' )
   elif entity.bfdIntfState == 'ospfIntfBfdDisabled':
      cmds.addCommand( 'no ip ospf neighbor bfd' )
   elif saveAll:
      cmds.addCommand( 'default ip ospf neighbor bfd' )

   if entity.cost != entity.costDefault:
      cmds.addCommand( 'ip ospf cost %d' % entity.cost )
   elif saveAll:
      cmds.addCommand( 'no ip ospf cost' )
      
   if entity.routerDeadInterval != entity.routerDeadIntervalDefault or saveAll:
      cmds.addCommand( 'ip ospf dead-interval %d' % entity.routerDeadInterval )
      
   if entity.helloInterval != entity.helloIntervalDefault or saveAll:
      cmds.addCommand( 'ip ospf hello-interval %d' % entity.helloInterval )
      
   if entity.priority != entity.priorityDefault or saveAll:
      cmds.addCommand( 'ip ospf priority %d' % entity.priority )
      
   if entity.rxInt != entity.rxIntDefault or saveAll:
      cmds.addCommand( 'ip ospf retransmit-interval %d' % entity.rxInt )
      
   if not entity.enable:
      cmds.addCommand( 'ip ospf disabled' )
   elif saveAll:
      cmds.addCommand( 'no ip ospf disabled' )
      
   if entity.transDelay != entity.transDelayDefault or saveAll:
      cmds.addCommand( 'ip ospf transmit-delay %d' % entity.transDelay )

   if entity.intfType != entity.intfTypeDefault:
      cmds.addCommand( 'ip ospf network %s' % {
         'intfTypePointToPoint': 'point-to-point',
         'intfTypePointToMultipoint': 'point-to-multipoint',
         'intfTypeNonBroadcast': 'nonbroadcast' }[ entity.intfType ] )

   authType = entity.authType
   if authType == 'ospfAuthTypeSimple':
      cmds.addCommand( 'ip ospf authentication' )
   elif authType == 'ospfAuthTypeMd':
      cmds.addCommand( 'ip ospf authentication message-digest' )
   elif saveAll:
      cmds.addCommand( 'no ip ospf authentication' )

   sanitize = lambda word: CliSave.sanitizedOutput( options, word )
   securityConfig = requireMounts[ 'mgmt/security/config' ]
   if entity.simpleAuth:
      myKey = None if securityConfig.commonKeyEnabled else \
            entity.name + '_passwd'

      cmds.addCommand( 'ip ospf authentication-key 7 %s' %
                       sanitize( encodeKey( entity.simpleAuth,
                                            key=myKey, algorithm='MD5' ) ) )

   if entity.mtuIgnore:
      cmds.addCommand( 'ip ospf mtu-ignore' )
   elif saveAll:
      cmds.addCommand( 'no ip ospf mtu-ignore' )

   if entity.areaIdPresent != entity.areaIdPresentDefault:
      cmds.addCommand( 'ip ospf area %s' % entity.areaId )
   
   # process message digest / cryptographic auth keys
   for keyId in sorted( entity.mdAuth ):
      authAlgorithm = entity.mdAuth[ keyId ].authAlgo 
      myKey = None if securityConfig.commonKeyEnabled else \
         entity.name + '_%sKey_%d' % ( authAlgorithm, keyId )
      encryptedKey = encodeKey( entity.mdAuth[ keyId ].authKey, 
                             key=myKey, algorithm = "MD5" )
      cmds.addCommand( 'ip ospf message-digest-key %d %s 7 %s' %
                       ( keyId, authAlgorithm, sanitize( encryptedKey ) ) )
   
def _saveInstanceConfig( entity, root, ospfConfig, sysdbRoot, saveAll,
                         requireMounts ):

   mode = root[ RouterOspfConfigMode ].getOrCreateModeInstance( ( entity.instance,
                                                                  entity.vrfName ) )
   cmds = mode[ 'Ospf.instance' ]

   if entity.routerId != entity.routerIdDefault:
      cmds.addCommand( 'router-id %s' % entity.routerId )
   elif saveAll:
      cmds.addCommand( 'no router-id' )
       
   if not entity.enable:
      cmds.addCommand( 'shutdown' )
   elif saveAll:
      cmds.addCommand( 'no shutdown' )

   if entity.refBw != entity.refBwDefault:
      cmds.addCommand( 'auto-cost reference-bandwidth %s' % entity.refBw )
   elif saveAll:
      cmds.addCommand( 'no auto-cost reference-bandwidth' )

   if entity.bfdEnabledState:
      cmds.addCommand( 'bfd default' )
   elif saveAll:
      cmds.addCommand( 'no bfd default' )
      
   if Toggles.OspfToggleLib.toggleBfdForAllStatesEnabled():
      if entity.bfdAnyState:
         cmds.addCommand( 'bfd adjacency state any' )
      elif saveAll:
         cmds.addCommand( 'no bfd adjacency state any' )

   if Toggles.OspfToggleLib.toggleOspfUnnumberedHelloMaskTxEnabled():
      if entity.unnumberedHelloMaskTx:
         cmds.addCommand( 'interface unnumbered hello mask tx 0.0.0.0' )
      elif saveAll:
         cmds.addCommand( 'no interface unnumbered hello mask tx 0.0.0.0' )
         
   # BUG7063: we are not shipping LLS
   #if not entity.lls:
   #   cmds.addCommand( 'no capability lls' )
   
   # Display the default only if the command is enabled.
   if entity.defaultMetric != entity.defaultMetricDefault or \
      ( saveAll and os.environ.get( 'ENABLE_CMDS_NOT_IN_FIRST_REL' ) ):
      cmds.addCommand( 'default-metric %d' % entity.defaultMetric )
       
   if entity.internalPref != entity.internalPrefDefault or saveAll:
      cmds.addCommand( 'distance ospf intra-area %d' % entity.internalPref )

   if entity.asePref != entity.asePrefDefault or saveAll:
      cmds.addCommand( 'distance ospf external %d' % entity.asePref )

   if entity.interPref != entity.interPrefDefault or saveAll:
      cmds.addCommand( 'distance ospf inter-area %d' % entity.interPref )

   if entity.passiveByDefault:
      cmds.addCommand( 'passive-interface default' )
   
   instanceId = entity.instance
   # The passive-interface config is actually in IntfConfig.
   for intfName in Arnet.sortIntf( ospfConfig.intfConfig ):
      intfConfig = ospfConfig.intfConfig[ intfName ]
      if instanceId not in intfConfig.passiveInstances:
         continue
      if intfConfig.passiveInstances[instanceId] == 'ospfIntfPassive':
         cmds.addCommand( 'passive-interface %s' % intfName )
      elif intfConfig.passiveInstances[instanceId] == 'ospfIntfActive':
         cmds.addCommand( 'no passive-interface %s' % intfName )

   for protocol in sorted( entity.redistributeConfig.values(), 
                        key=lambda p: p.proto ):
      redistOspf = protocol.proto in ( 'protoOspf', 'protoOspfAse', 'protoOspfNssa' )
      command = 'redistribute %s' % \
                  ( protoDict[ protocol.proto ] if not redistOspf else 'ospf' )
      # isisLevel will be set to either level1/level2/level1_2 for Isis
      # It will be set to levelNone for all other protocols
      if redistOspf:
         if protocol.allowOspfInstance:
            command += ' instance'
            command += ' include leaked' if protocol.includeLeaked else ''
         else:
            assert protocol.includeLeaked
            command += ' leaked'
         if protocol.matchRouteType != '':
            command += ' match %s' % protocol.matchRouteType
      elif protocol.includeLeaked:
         command += ' include leaked'
      if protocol.isisLevel != 'levelNone':
         command += ' %s' % ISIS_LEVEL_MAP[ protocol.isisLevel ]
      if protocol.routeMap:
         command += ' route-map %s' % protocol.routeMap       
      cmds.addCommand( command )

   if entity.distListIn.distType == "distRouteMap":
      command = "distribute-list route-map %s in" % (
         entity.distListIn.distName )
      cmds.addCommand( command )
   elif entity.distListIn.distType == "distPrefixList":
      command = "distribute-list prefix-list %s in" % (
         entity.distListIn.distName )
      cmds.addCommand( command )

   # XXXXX TODO handle auth attributes

   for area in sorted( entity.areaConfig, cmp=IpUtils.compareIpAddress ):
      areaConfig = entity.areaConfig[ area ]

      if areaConfig.defaultMetric != areaConfig.defaultMetricDefault or saveAll:
         cmds.addCommand( 'area %s default-cost %d' %
                          ( area, areaConfig.defaultMetric ) )

      if areaConfig.areaType == 'areaTypeNssa' or \
         areaConfig.areaType == 'areaTypeNssaNoSummary':
         cmd = 'area %s nssa' % area
         if not areaConfig.nssaTranslateAlways and not areaConfig.nssaOnly and \
            areaConfig.nssaDefInfoOrigin != 'ospfDefInfoOriginEnable' and \
            areaConfig.areaType != 'areaTypeNssaNoSummary':
            cmds.addCommand( cmd )
         if areaConfig.areaType == 'areaTypeNssaNoSummary':
            cmdNoSummary = cmd + ' no-summary'
            cmds.addCommand( cmdNoSummary )
         if areaConfig.nssaTranslateAlways:
            cmd1 = "area %s not-so-stubby lsa type-7 convert type-5" % area
            cmds.addCommand( cmd1 )
         if areaConfig.nssaOnly:
            cmd2 = cmd + ' nssa-only'
            cmds.addCommand( cmd2 )
         if areaConfig.nssaDefInfoOrigin == 'ospfDefInfoOriginEnable':
            cmd3 = cmd + ' default-information-originate'
            if areaConfig.nssaDefInfoMetric != areaConfig.nssaDefInfoMetricDefault \
               or saveAll:
               cmd3 += ' metric %d' % areaConfig.nssaDefInfoMetric
            if areaConfig.nssaDefInfoMetricType != \
               areaConfig.nssaDefInfoMetricTypeDefault or saveAll:
               cmd3 += ' metric-type %d' % \
                     int( areaConfig.nssaDefInfoMetricType.lstrip( "extMetricType" ) 
                        )
            if areaConfig.nssaDefInfoNssaOnly != \
               areaConfig.nssaDefInfoNssaOnlyDefault: 
               cmd3 += ' nssa-only'
            cmds.addCommand( cmd3 )
         elif areaConfig.nssaDefInfoOrigin == 'ospfDefInfoOriginSuppress':
            cmds.addCommand( 'no ' + cmd + ' default-information-originate' )
      elif areaConfig.areaType == 'areaTypeStub':
         cmds.addCommand( 'area %s stub' % area )
      elif areaConfig.areaType == 'areaTypeStubNoSummary':
         cmds.addCommand( 'area %s stub no-summary' % area )
      for prefix in sorted( areaConfig.summaryFilter,
                            cmp=IpUtils.compareIpPrefix ):
         # BUG25925
         cmds.addCommand( 'area %s filter %s' % ( area, prefix.stringValue ) ) 

      if areaConfig.summaryPrefixList:
         cmds.addCommand( 'area %s filter prefix-list %s'
                          % ( area, areaConfig.summaryPrefixList ) )

      for prefix in sorted( areaConfig.networkList,
                            cmp=IpUtils.compareIpPrefix ):
         # BUG25925
         cmd = 'area %s range %s' % ( area, prefix.stringValue )
         if areaConfig.networkList[ prefix ].restricted:
            cmd += ' not-advertise'
         elif saveAll:
            cmd += ' advertise'
         if areaConfig.networkList[ prefix ].cost != 0:
            cmd += ' cost %d' % areaConfig.networkList[ prefix ].cost

         cmds.addCommand( cmd )

      for routerId in areaConfig.virtualLink:
         virtualLink = areaConfig.virtualLink[ routerId ]
         cmd = 'area %s virtual-link %s' % ( area, routerId )

         # XXXXX TODO handle auth attributes

         cmd1 = None
         if virtualLink.helloInterval != virtualLink.helloIntervalDefault or saveAll:
            cmd1 = cmd + ' hello-interval %d' % virtualLink.helloInterval
            cmds.addCommand( cmd1 ) 
         if virtualLink.transDelay != virtualLink.transDelayDefault or saveAll:
            cmd1 = cmd + ' transmit-delay %d' % virtualLink.transDelay
            cmds.addCommand( cmd1 ) 
         if virtualLink.rxInt != virtualLink.rxIntDefault or saveAll:
            cmd1 = cmd + ' retransmit-interval %d' % virtualLink.rxInt
            cmds.addCommand( cmd1 ) 
         if virtualLink.routerDeadInterval != \
                virtualLink.routerDeadIntervalDefault or saveAll:
            cmd1 = cmd + ' dead-interval %d' % virtualLink.routerDeadInterval
            cmds.addCommand( cmd1 ) 

         if cmd1 is None:
            cmds.addCommand( cmd ) 

   if entity.rfc1583Compat:
      cmds.addCommand( 'compatible rfc1583' )

   prefixes = sorted( entity.networkArea, cmp=IpUtils.compareIpPrefix )
   for prefix in prefixes:
      areaConfig = entity.networkArea.get( prefix )
      if ( areaConfig is not None and \
            areaConfig == entity.areaConfig.get( areaConfig.area ) ):
         # TODO .stringValue should not be necessary, BUG25925
         cmds.addCommand( 'network %s area %s' \
                             % ( prefix.stringValue, areaConfig.area ) )

   # Always add the max-lsa command, to give its default value extra
   # visibility.
   cmd = 'max-lsa %d' % entity.maxLsa
   if entity.maxLsaThreshold != entity.maxLsaThresholdDefault or saveAll:
      cmd += ' %d' % entity.maxLsaThreshold
   if entity.maxLsaWarningOnly != entity.maxLsaWarningOnlyDefault:
      cmd += ' warning-only'
   else:
      if entity.maxLsaIgnoreTime != entity.maxLsaIgnoreTimeDefault or saveAll:
         cmd += ' ignore-time %d' % entity.maxLsaIgnoreTime
      if entity.maxLsaIgnoreCount != entity.maxLsaIgnoreCountDefault or saveAll:
         cmd += ' ignore-count %d' % entity.maxLsaIgnoreCount
      if entity.maxLsaResetTime != entity.maxLsaResetTimeDefault or saveAll:
         cmd += ' reset-time %d' % entity.maxLsaResetTime
   
   cmds.addCommand( cmd )

   if entity.simultaneousBringUp != entity.simultaneousBringUpDefault:
      cmds.addCommand( 'adjacency exchange-start threshold %d'
                       % entity.simultaneousBringUp )
   elif saveAll:
      cmds.addCommand( 'adjacency exchange-start threshold %d'
                       % entity.simultaneousBringUpDefault )
   if entity.lsaRetransmissionThreshold != entity.lsaRetransmissionThresholdDefault \
      or saveAll:
      cmds.addCommand( 'retransmission-threshold lsa %d'
                       % entity.lsaRetransmissionThreshold )
   if entity.adjacencyLogging != entity.adjacencyLoggingDefault or saveAll:
      cmd = 'log-adjacency-changes'
      if entity.adjacencyLogging == 'adjacencyLoggingTypeDetail':
         cmd += ' detail'
      elif entity.adjacencyLogging == 'adjacencyLoggingTypeNone':
         cmd = 'no ' + cmd
      cmds.addCommand( cmd )
 
   # print out command in the timers spf format if timerSpfCommandActive
   # else we print the command in timers [spf delay initial|throttle spf] format
   timerConfig = entity.spfThrottleTimerConfig
   if timerConfig.spfStartInt != entity.spfStartIntDefault or \
         timerConfig.spfHoldInt != entity.spfHoldIntDefault or \
         timerConfig.spfMaxWaitInt != entity.spfMaxWaitIntDefault:
      if entity.timerSpfCommandActive:
         holdIntSec = timerConfig.spfHoldInt/1000
         cmd = 'timers spf %d' % holdIntSec
      else:
         cmd = 'timers spf delay initial %d %d %d' % ( timerConfig.spfStartInt,
                                                       timerConfig.spfHoldInt,
                                                       timerConfig.spfMaxWaitInt )
      cmds.addCommand( cmd )
   elif saveAll:
      if entity.timerSpfCommandActive:
         holdIntSec = timerConfig.spfHoldInt/1000
         cmd = 'timers spf %d' % holdIntSec
      else:
         cmd = 'timers spf delay initial %d %d %d' % ( timerConfig.spfStartInt,
                                                       timerConfig.spfHoldInt,
                                                       timerConfig.spfMaxWaitInt )
      cmds.addCommand( cmd )

   # timers lsa rx min interval
   if entity.lsaArrivalInt != entity.lsaArrivalIntDefault:
      cmd = 'timers lsa rx min interval %d' % ( entity.lsaArrivalInt )
      cmds.addCommand( cmd )
   elif saveAll:
      cmd = 'timers lsa rx min interval %d' % ( entity.lsaArrivalInt )
      cmds.addCommand( cmd )

   # timers lsa tx delay initial 
   timerConfig = entity.lsaThrottleTimerConfig
   if timerConfig.lsaStartInt != entity.lsaStartIntDefault or \
         timerConfig.lsaHoldInt != entity.lsaHoldIntDefault or \
         timerConfig.lsaMaxWaitInt != entity.lsaMaxWaitIntDefault:
      cmdStr = 'timers lsa tx delay initial %d %d %d'
      cmd = cmdStr % ( timerConfig.lsaStartInt, timerConfig.lsaHoldInt,
                       timerConfig.lsaMaxWaitInt )
      cmds.addCommand( cmd )
   elif saveAll:
      cmd = 'timers lsa tx delay initial %d %d %d' % ( timerConfig.lsaStartInt,
                                 timerConfig.lsaHoldInt, timerConfig.lsaMaxWaitInt )
      cmds.addCommand( cmd )

   # timers out-delay <lsa out-delay>
   if entity.lsaOutDelayTimerConfig != entity.lsaOutDelayTimerDefault:
      cmd = "timers out-delay %d" % entity.lsaOutDelayTimerConfig
      cmds.addCommand( cmd )
   elif saveAll:
      cmd = "no timers out-delay"
      cmds.addCommand( cmd )
 
   if entity.maxEcmp > 0:
      cmd = 'maximum-paths %d' % entity.maxEcmp
      cmds.addCommand( cmd )
   elif saveAll:
      routingHwStatus = requireMounts[ 'routing/hardware/status' ]
      if routingHwStatus and routingHwStatus.maxEcmp > 0:
         cmd = 'maximum-paths %d' % routingHwStatus.maxEcmp
         cmds.addCommand( cmd )

   if entity.floodPacing != entity.floodPacingDefault:
      cmds.addCommand( 'timers pacing flood %d' % entity.floodPacing )
   elif saveAll:
      cmd = "no timers pacing flood"
      cmds.addCommand( cmd )

   maxMetric = entity.maxMetricConfig
   cmd = ''
   if maxMetric.maxMetricFlag == entity.maxMetricFlagDefault and saveAll:
      cmd = 'no max-metric router-lsa'
   else:
      if maxMetric.maxMetricFlag != entity.maxMetricFlagDefault:
         cmd = 'max-metric router-lsa'
      if maxMetric.maxMetricExtMetric != entity.maxMetricExtMetricDefault:
         cmd += ' external-lsa'
         if maxMetric.maxMetricExtMetric != entity.maxMetricLsaMetricDefault or \
               saveAll:
            cmd += ' %d' % maxMetric.maxMetricExtMetric
      if maxMetric.maxMetricStubFlag != entity.maxMetricStubFlagDefault:
         cmd += ' include-stub'
      if maxMetric.maxMetricOnStartDuration != \
            entity.maxMetricOnStartDurationDefault:
         cmd += ' on-startup %d' % maxMetric.maxMetricOnStartDuration
      if maxMetric.maxMetricWaitBgpFlag != entity.maxMetricWaitBgpFlagDefault:
         cmd += ' on-startup wait-for-bgp'
      if maxMetric.maxMetricSumMetric != entity.maxMetricSumMetricDefault:
         cmd += ' summary-lsa'
         if maxMetric.maxMetricSumMetric != entity.maxMetricLsaMetricDefault or \
               saveAll:
            cmd += ' %d' % maxMetric.maxMetricSumMetric
   if cmd:
      cmds.addCommand( cmd )
   
   if entity.pointToPointRoutes != True:
      cmds.addCommand( 'no point-to-point routes' )
   elif saveAll:
      cmds.addCommand( 'point-to-point routes' )

   if entity.tunnelRoutes:
      cmds.addCommand( 'tunnel routes' )
   elif saveAll:
      cmds.addCommand( 'no tunnel routes' )

   if entity.defInfoOrigin == 'ospfDefInfoOriginEnable':
      cmd = 'default-information originate'
      if entity.defInfoOriginAlways != entity.defInfoOriginAlwaysDefault:
         cmd += ' always'
      if entity.defInfoMetric != entity.defInfoMetricDefault or saveAll:
         cmd += ' metric %d' % entity.defInfoMetric
      if entity.defInfoMetricType != entity.defInfoMetricTypeDefault or saveAll:
         cmd += ' metric-type %d' % \
               int( entity.defInfoMetricType.lstrip( "extMetricType" ) )
      if entity.defInfoRouteMap != entity.defInfoRouteMapDefault:
         cmd += ' route-map %s' % entity.defInfoRouteMap
      cmds.addCommand( cmd )
   elif entity.defInfoOrigin == 'ospfDefInfoOriginSuppress':
      cmds.addCommand( 'no default-information originate' )

   # summary-address
   for summAddr in sorted( entity.summaryAddrConfig, IpUtils.compareIpPrefix ):
      summCfg = entity.summaryAddrConfig.get( summAddr )
      if summCfg is not None:
         cmd = 'summary-address %s' % summCfg.prefix
         if summCfg.notAdvertise:
            cmd = cmd + ' not-advertise'
         elif  summCfg.tag:
            cmd = cmd + ' tag %d' % summCfg.tag
         elif summCfg.attrMap:
            cmd = cmd + ' attribute-map %s' % summCfg.attrMap
         cmds.addCommand( cmd )

   if entity.gracefulRestart != entity.gracefulRestartDefault:
      cmdDisplayed = False
      if entity.grGracePeriod != entity.grGracePeriodDefault:
         cmds.addCommand( 'graceful-restart grace-period %d' %
                          ( entity.grGracePeriod ) )
         cmdDisplayed = True
      elif saveAll:
         cmds.addCommand( 'graceful-restart grace-period %d' %
                          ( entity.grGracePeriodDefault ) )
         cmdDisplayed = True
      if entity.grPlannedOnly != entity.grPlannedOnlyDefault:
         cmds.addCommand( 'graceful-restart planned-only' )
         cmdDisplayed = True
      if not cmdDisplayed:
         cmds.addCommand( 'graceful-restart' )
   elif saveAll:
      cmds.addCommand( 'no graceful-restart' )

   if entity.mplsLdpSync != entity.mplsLdpSyncDefault:
      cmds.addCommand( 'mpls ldp sync default' )
   if entity.grHelper != entity.grHelperDefault:
      cmds.addCommand( 'no graceful-restart-helper' )
   elif saveAll:
      cmds.addCommand( 'no mpls ldp sync default' )
      cmds.addCommand( 'graceful-restart-helper' )

   aclVrfName = entity.vrfName
   aclCpConfig = requireMounts[ 'acl/cpconfig/cli' ]
   serviceAclVrfConfig = aclCpConfig.cpConfig[ 'ip' ].serviceAcl.get( aclVrfName )
   if ( serviceAclVrfConfig and 
        OspfConsts.serviceName in serviceAclVrfConfig.service and
        serviceAclVrfConfig.service[ OspfConsts.serviceName ].aclName != '' ):
      cmds.addCommand( 'ip access-group %s' % 
            serviceAclVrfConfig.service[ OspfConsts.serviceName ].aclName )
   elif saveAll:
      cmds.addCommand( 'no ip access-group' )

   if entity.ignoreDnBit != entity.ignoreDnBitDefault:
      if entity.backwardCompatibilityIgnoreDnBitType5Type7 != \
         entity.backwardCompatibilityIgnoreDnBitType5Type7Default:
         cmds.addCommand( 'dn-bit-ignore lsa type-5 type-7' )
      else:
         cmds.addCommand( 'dn-bit-ignore' )
   elif saveAll:
      cmds.addCommand( 'no dn-bit-ignore' )
   
   if entity.intraAreaRouteEcmp != entity.intraAreaRouteEcmpDefault:
      cmds.addCommand( 'ecmp intra-area route multi-area next-hops' )
   elif saveAll:
      cmds.addCommand( 'no ecmp intra-area route multi-area next-hops' )

   if entity.ospfTeMode != entity.ospfTeModeDefault:
      saveInstanceTeConfig( entity, mode, saveAll )
   elif saveAll and entity.vrfName == DEFAULT_VRF:
      # Don't do this in the non default vrf because we have a guard error
      cmds.addCommand( 'no traffic-engineering' )

   srDataPlaneEnum = Tac.Type( 'Routing::Ospf::SrDataPlane' )
   srConfig = entity.srConfig
   if srConfig and srConfig.dataPlane == srDataPlaneEnum.srDataPlaneMpls:
      saveInstanceSrMplsConfig( entity, mode, saveAll )
   elif saveAll and entity.vrfName == DEFAULT_VRF:
      # Don't do this in the non default vrf because we have a guard error
      cmds.addCommand( 'no segment-routing mpls' )

def saveGeneralConfig( ospfConfig, root, saveAll ):
   mode = root[ RouterOspfGeneralConfigMode ].\
                           getOrCreateModeInstance( 'ospfGeneralConfig' )
   cmds = mode[ 'Ospf.general' ]

   generalConfig = ospfConfig.generalConfig
   if not generalConfig:
      return

   if generalConfig.dscp != generalConfig.dscpDefault or saveAll:
      cmds.addCommand( 'qos dscp %d' % generalConfig.dscp )

def saveInstanceTeConfig( instanceConfig, root, saveAll ):
   mode = root[ RouterOspfTeConfigMode ].getOrCreateModeInstance(
      instanceConfig.instance )
   cmds = mode[ 'Ospf.te' ]

   if instanceConfig.ospfTeEnabled != instanceConfig.ospfTeEnabledDefault:
      cmds.addCommand( 'no shutdown' )
   elif saveAll:
      cmds.addCommand( 'shutdown' )

   if ( instanceConfig.ospfTeConfiguredAllAreas !=
        instanceConfig.ospfTeConfiguredAllAreasDefault ):
      cmds.addCommand( 'no area all' )
   elif saveAll:
      cmds.addCommand( 'area all' )

   areaIdList = []
   for area in sorted( instanceConfig.areaConfig, cmp=IpUtils.compareIpAddress ):
      areaConfig = instanceConfig.areaConfig[ area ]
      if areaConfig.teConfigured != areaConfig.teConfiguredDefault:
         areaIdList.append( area )

   if areaIdList:
      cmds.addCommand( 'area %s' % ' '.join( areaIdList ) )

def saveInstanceSrMplsConfig( instanceConfig, root, saveAll ):
   mode = root[ RouterOspfSrMplsConfigMode ].getOrCreateModeInstance(
      instanceConfig.instance )
   cmds = mode[ 'Ospf.srMpls' ]

   srConfig = instanceConfig.srConfig
   if srConfig.shutdown != srConfig.shutdownDefault:
      cmds.addCommand( 'no shutdown' )
   elif saveAll:
      cmds.addCommand( 'shutdown' )

   proxyNodeSegmentList = []
   prefixSegmentList = []
   for prefixSegment in instanceConfig.srConfig.prefixSegments.values():
      if prefixSegment.isProxyNode is True:
         proxyNodeSegmentList.append( [ prefixSegment.prefix,
                                        prefixSegment.index ] )
      else:
         prefixSegmentList.append( [ prefixSegment.prefix,
                                     prefixSegment.index ] )

   for entry in proxyNodeSegmentList:
      cmds.addCommand( 'proxy-node-segment %s index %d' % ( entry[ 0 ],
                                                            entry[ 1 ] ) )

   for entry in prefixSegmentList:
      cmds.addCommand( 'prefix-segment %s index %d' % ( entry[ 0 ],
                                                        entry[ 1 ] ) )
#-------------------------------------------------------------------------------
# save config under global config mode
#-------------------------------------------------------------------------------

CliSave.GlobalConfigMode.addCommandSequence( 'Ip.Ospf',
                                             after=[ RouterOspfConfigMode ] )

@CliSave.saver( 'Routing::Ospf::Config', 'routing/ospf/config',
                requireMounts = ( 'routing/hardware/status',
                                  'interface/config/all', 'interface/status/all',
                                  'acl/cpconfig/cli',
                                  'mgmt/security/config' ) )
def saveConfig( ospfConfig, root, sysdbRoot, options,
                requireMounts ):
   # Check if any Ospf configuration is made.
   ospfInstanceConfigured = len( ospfConfig.instanceConfig ) > 0
   ospfInterfacesConfigured = len( ospfConfig.intfConfig ) > 0
   ospfNameLookupConfigured = ospfConfig.nameLookup != ospfConfig.nameLookupDefault
   ospfGeneralConfigured = ospfConfig.generalConfig is not None

   ospfConfigEnabled = ( ospfInstanceConfigured or ospfInterfacesConfigured or \
                         ospfNameLookupConfigured or ospfGeneralConfigured )

   # No need to show ospf configs if nothing is configured
   if not ospfConfigEnabled and not options.saveAllDetail:
      return

   if ospfGeneralConfigured and ( options.saveAll or
                                  ospfConfig.generalConfig.dscp !=
                                    ospfConfig.generalConfig.dscpDefault ):
      saveGeneralConfig( ospfConfig, root, options.saveAll )

   if ospfConfig.nameLookup != ospfConfig.nameLookupDefault:
      root[ 'Ip.Ospf' ].addCommand( 'ip ospf router-id output-format hostnames' )
   elif options.saveAll:
      root[ 'Ip.Ospf' ].addCommand( 'no ip ospf router-id output-format hostnames' )
 
   for instanceId in sorted( ospfConfig.instanceConfig, key=int ):
      _saveInstanceConfig( ospfConfig.instanceConfig[ instanceId ],
            root, ospfConfig, sysdbRoot, options.saveAll, requireMounts )

   if options.saveAllDetail:
      intfs = allIpIntfNames( sysdbRoot, includeEligible=True,
                              requireMounts=requireMounts )
   elif options.saveAll:
      # Display Ospf configuration on routing protocol interfaces as well as 
      # switchports present in intfConfig collection.
      intfs = set(
            allRoutingProtocolIntfNames( sysdbRoot, requireMounts=requireMounts ) +
            ospfConfig.intfConfig.keys() ) 
   else:
      intfs = ospfConfig.intfConfig
   
   # save config for each interface
   for intfName in intfs:
      if isManagement(intfName): 
         continue

      intfConfig = ospfConfig.intfConfig.get( intfName )
      if not intfConfig:
         if options.saveAll:
            intfConfig = Tac.newInstance( 'Routing::Ospf::IntfConfig', intfName )
         else:
            continue
      saveIntfConfig( intfConfig, root, sysdbRoot, options, requireMounts )
