# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements the CliSave code for router ospfv3 Multi AF commands
#-------------------------------------------------------------------------------
import CliSave, Tac
from RoutingIntfUtils import allRoutingProtocolIntfNames, allIpIntfNames, \
                             isManagement
from RoutingOspf3CliSave import saveInstanceConfig, saveIntfConfig, \
                                oldStyleConfigPresent
from IpLibConsts import DEFAULT_VRF
import Ospf3Consts

addrFamilyInstanceIDEnum = Tac.Type( "Routing6::Ospf3::AfInstanceId" )
allAfInstanceID = addrFamilyInstanceIDEnum.allAf
afIPv4InstanceID = addrFamilyInstanceIDEnum.afIPv4
afIPv6InstanceID = addrFamilyInstanceIDEnum.afIPv6

#-------------------------------------------------------------------------------
# save config under global config mode
#-------------------------------------------------------------------------------
@CliSave.saver( Ospf3Consts.configType, Ospf3Consts.configPath,
                requireMounts = ( 'routing/hardware/status',
                                  'routing6/hardware/status',
                                  'interface/config/all',
                                  'interface/status/all',
                                  'acl/cpconfig/cli',
                                  'mgmt/security/config' ) )
def saveConfig( ospfConfig, root, sysdbRoot, options,
                requireMounts ):

   # Check if any Ospf configuration is made.
   ospfVrfConfigured = len( ospfConfig.vrfConfig ) > 0
   ospfInterfacesConfigured = len( ospfConfig.intfConfig ) > 0

   ospfConfigEnabled = ( ospfVrfConfigured or ospfInterfacesConfigured )

   # No need to show ospf configs if nothing is configured
   if not ospfConfigEnabled and not options.saveAllDetail:
      return

   # Check if we dont have any old style configuration, only then proceed.
   # If there is no old style config, the default save-all configs are handled
   # by this function.
   if oldStyleConfigPresent( ospfConfig ):
      return

   vrfConfigListSorted = sorted( ospfConfig.vrfConfig.itervalues(),
                          key=lambda vrfConfig: '' if vrfConfig.vrfName == \
                          DEFAULT_VRF else vrfConfig.vrfName )
   for vrfConfig in vrfConfigListSorted:
      # skip old-style CLI
      if vrfConfig.processId:
         continue
      for instId in ( allAfInstanceID, afIPv4InstanceID, afIPv6InstanceID ):
         instanceConfig = vrfConfig.instanceConfig.get( instId, None )
         if not instanceConfig:
            continue
         saveInstanceConfig( vrfConfig, instanceConfig, root, ospfConfig,
                           sysdbRoot, options, requireMounts )

   if options.saveAllDetail:
      intfs = allIpIntfNames( sysdbRoot, includeEligible=True,
                              requireMounts=requireMounts )
   elif options.saveAll:
      # Display Ospf configuration on routing protocol interfaces as well as 
      # switchports present in intfConfig collection.
      intfs = set(
            allRoutingProtocolIntfNames( sysdbRoot, requireMounts=requireMounts ) +
            ospfConfig.intfConfig.keys() ) 
   else:
      intfs = ospfConfig.intfConfig
   
   # save config for each interface
   for intfName in intfs:
      if isManagement( intfName ): 
         continue

      intfConfig = ospfConfig.intfConfig.get( intfName )
      if not intfConfig:
         if options.saveAll:
            intfConfig = Tac.newInstance( 'Routing6::Ospf3::IntfConfig', intfName )
            intfInstanceConfig = intfConfig.intfInstanceConfig.newMember( 
                                    allAfInstanceID, intfName )
            saveIntfConfig( intfConfig, intfInstanceConfig, ospfConfig, root,
                           sysdbRoot, options, requireMounts )
      else:
         if intfConfig.oldStyleCli:
            continue
         for instId in ( allAfInstanceID, afIPv4InstanceID, afIPv6InstanceID ):
            intfInstanceConfig = intfConfig.intfInstanceConfig.get( instId, None )
            if not intfInstanceConfig:
               continue
            saveIntfConfig( intfConfig, intfInstanceConfig, ospfConfig, root,
                            sysdbRoot, options, requireMounts )
