#-------------------------------------------------------------------------------
# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
# Copyright (c) 2009, 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
#-------------------------------------------------------------------------------
# This module implements the CliSave code for routing ospf3 commands
#-------------------------------------------------------------------------------
import CliSave, Arnet, Tac, IpUtils
from RoutingIntfUtils import allRoutingProtocolIntfNames, allIpIntfNames, \
                             isManagement
from IntfCliSave import IntfConfigMode
from RibCliLib import protoDict
from CliMode.Ospf3 import RoutingOspf3Mode
from CliMode.Ospf3MultiAf import RoutingOspfv3Mode, RoutingOspfv3GeneralMode
from CliMode.Ospf3MultiAf import RoutingOspfv3AfMode
import OspfConsts, Ospf3Consts
from ReversibleSecretCli import encodeKey
import Toggles.Ospf3ToggleLib
import Toggles.OspfToggleLib

addrFamilyInstanceIDEnum = Tac.Type( "Routing6::Ospf3::AfInstanceId" )
allAfInstanceID = addrFamilyInstanceIDEnum.allAf
afIPv4InstanceID = addrFamilyInstanceIDEnum.afIPv4
afIPv6InstanceID = addrFamilyInstanceIDEnum.afIPv6

ISIS_LEVEL_MAP = { 'level1' : 'level-1',
                   'level2' : 'level-2',
                   'level1_2' : 'level-1-2' }
#-------------------------------------------------------------------------------
# Object used for saving commands in router-ospfv3 general mode.
#-------------------------------------------------------------------------------
class RouterOspfv3GeneralConfigMode( RoutingOspfv3GeneralMode, CliSave.Mode ):
   def __init__( self, param ):
      RoutingOspfv3GeneralMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

#-------------------------------------------------------------------------------
# Object used for saving commands in router-ospf3 mode.
#-------------------------------------------------------------------------------
class RouterOspf3ConfigMode( RoutingOspf3Mode, CliSave.Mode ):
   def __init__( self, param ):
      RoutingOspf3Mode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( RouterOspfv3GeneralConfigMode,
                                       after=[ IntfConfigMode ] )
RouterOspfv3GeneralConfigMode.addCommandSequence( 'Ospf3.general' )

CliSave.GlobalConfigMode.addChildMode( RouterOspf3ConfigMode,
                                       after=[ RouterOspfv3GeneralConfigMode ] )
RouterOspf3ConfigMode.addCommandSequence( 'Ospf3.instance' )

IntfConfigMode.addCommandSequence( 'Ospf3.intf', after=[ 'Ira.intf6' ] )

def oldStyleConfigPresent( ospfConfig ):
   # Check for old style instance configs
   for vc in ospfConfig.vrfConfig.itervalues():
      if vc.processId:
         return True
   # Check for old style intf configs
   for ic in ospfConfig.intfConfig.itervalues():
      if ic.oldStyleCli:
         return True

   return False

#Get the encrypt algo config from Tac model values
def getEncryptAlgo ( algo ):
   if algo == "tripleDes":
      return "3des-cbc"
   algo = algo.lower()
   return algo[:3] + "-" + algo[3:6] + "-" + algo[6:9]

def saveGeneralConfig( ospfConfig, root, saveAll ):
   mode = root[ RouterOspfv3GeneralConfigMode ].\
                           getOrCreateModeInstance( 'ospfv3GeneralConfig' )
   cmds = mode[ 'Ospf3.general' ]

   generalConfig = ospfConfig.generalConfig
   if not generalConfig:
      return

   if generalConfig.dscp != generalConfig.dscpDefault or saveAll:
      cmds.addCommand( 'qos dscp %d' % generalConfig.dscp )

def saveIntfConfig( intfConfig, entity, ospfConfig, root, sysdbRoot, options,
                    requireMounts ):
   mode = root[ IntfConfigMode ].getOrCreateModeInstance( intfConfig.name )
   saveAll = options.saveAll

   afInstanceId = entity.instanceId
   if not intfConfig.oldStyleCli:
      cmds = mode[ 'Ospf3MultiAf.intf' ]
      if afInstanceId == afIPv4InstanceID:
         intfCliPrefix = 'ospfv3 ipv4'
      elif afInstanceId == afIPv6InstanceID:
         intfCliPrefix = 'ospfv3 ipv6'
      else:
         intfCliPrefix = 'ospfv3'
   else:
      assert afInstanceId == afIPv6InstanceID
      intfCliPrefix = 'ipv6 ospf'
      cmds = mode[ 'Ospf3.intf' ]

   val = getIntfConfigAttr( intfConfig, entity, 'bfdIntf', saveAll )
   if val == 'ospfIntfBfdEnabled':
      cmds.addCommand( '%s bfd' % intfCliPrefix )
   elif val == 'ospfIntfBfdDisabled':
      cmds.addCommand( 'no %s bfd' % intfCliPrefix )
   elif val == 'ospfIntfBfdNone':
      cmds.addCommand( 'default %s bfd' % intfCliPrefix )

   if not intfConfig.oldStyleCli:
      val = getIntfConfigAttr( intfConfig, entity, 'passive', saveAll )
      if val == 'ospfIntfPassiveNone':
         cmds.addCommand( 'default %s passive-interface' % intfCliPrefix )
      elif val == 'ospfIntfPassiveEnabled':
         cmds.addCommand( '%s passive-interface' % intfCliPrefix )
      elif val == 'ospfIntfPassiveDisabled':
         cmds.addCommand( 'no %s passive-interface' % intfCliPrefix )

   val = getIntfConfigAttr( intfConfig, entity, 'cost', saveAll )
   if val:
      cmds.addCommand( '%s cost %d' % ( intfCliPrefix, val ) )
   elif val is 0:
      cmds.addCommand( 'no %s cost' % ( intfCliPrefix ) )   

   val = getIntfConfigAttr( intfConfig, entity, 'helloInterval', saveAll )
   if val is not None:
      cmds.addCommand( '%s hello-interval %d' % ( intfCliPrefix, val ) )

   val = getIntfConfigAttr( intfConfig, entity, 'routerDeadInterval', saveAll )
   if val is not None:
      cmds.addCommand( '%s dead-interval %d' % ( intfCliPrefix, val ) )
  
   val = getIntfConfigAttr( intfConfig, entity, 'transDelay', saveAll )
   if val is not None:
      cmds.addCommand( '%s transmit-delay %d' % ( intfCliPrefix, val ) )

   val = getIntfConfigAttr( intfConfig, entity, 'rxInt', saveAll )
   if val is not None:
      if afInstanceId == afIPv6InstanceID:
         cmds.addCommand( 'ospfv3 ipv6 retransmit-interval %d' % val )
      elif afInstanceId == afIPv4InstanceID:
         cmds.addCommand( 'ospfv3 ipv4 retransmit-interval %d' % val )
      else:
         assert afInstanceId == allAfInstanceID
         cmds.addCommand( 'ospfv3 retransmit-interval %d' % val )

   val = getIntfConfigAttr( intfConfig, entity, 'priority', saveAll )
   if val is not None:
      cmds.addCommand( '%s priority %d' % ( intfCliPrefix, val ) )

   val = getIntfConfigAttr( intfConfig, entity, 'intfType', saveAll )
   if val and val != entity.intfTypeDefault:
      cmds.addCommand( '%s network %s' % ( intfCliPrefix, {
         'intfTypePointToPoint': 'point-to-point' }[ entity.intfType ] ) )
   
   val = getIntfConfigAttr( intfConfig, entity, 'securityEnabled', saveAll )
   if val:
      val = getIntfConfigAttr( intfConfig, entity, 'sa', saveAll )
      if val:
         securityConfig = requireMounts[ 'mgmt/security/config' ]
         useKey = None if securityConfig.commonKeyEnabled else \
                  entity.intfName + "_secretKey"
         cmd = intfCliPrefix + ' '
         encryptKey = ''
         passphraseDefined = val.passphraseKey != ""
         key = val.passphraseKey if passphraseDefined else \
               val.authKey
         if val.securityProto == "ah":
            cmd += "authentication ipsec spi %d " % val.spi
         else:
            if val.encryptAlgo == "null":
               cmd += "encryption ipsec spi %d esp null " % val.spi
            else:
               encryptAlgo = getEncryptAlgo( val.encryptAlgo )
               encryptKey = val.encryptKey
               cmd += "encryption ipsec spi %d esp %s " \
                     % ( val.spi, encryptAlgo )
         if passphraseDefined:
            cmd += val.hmacAlgo + " "
            cmd += "passphrase 7 "
         else:
            if val.encryptAlgo != "null":
               cmd += "7 "
               eKey = encodeKey( encryptKey, key=useKey, algorithm='MD5' )
               sanitize = lambda word: CliSave.sanitizedOutput( options, word )
               cmd += sanitize( eKey )
               cmd += ' '+ val.hmacAlgo + ' 7 '
            else:
               cmd += val.hmacAlgo + ' 7 '
         encryptedKey = encodeKey( key, key=useKey, algorithm='MD5' )
         sanitize = lambda word: CliSave.sanitizedOutput( options, word )
         cmd += sanitize( encryptedKey )
         cmds.addCommand( cmd )
   elif val is False:
      cmds.addCommand( "no %s authentication" % intfCliPrefix )

   if intfConfig.processId != intfConfig.processIdInvalid:
      assert intfConfig.oldStyleCli and entity.areaIdPresent
      cmds.addCommand( 'ipv6 ospf %u area %s' % ( intfConfig.processId,
                                                      entity.areaId ) )
   else:
      # Area ID can only be set with AF prefix
      if entity.areaIdPresent:
         cmds.addCommand( '%s area %s' % ( intfCliPrefix, entity.areaId ) )

   val = getIntfConfigAttr( intfConfig, entity, 'mtuIgnore', saveAll )
   if val:
      cmds.addCommand( '%s mtu-ignore' % intfCliPrefix )
   elif val is False:
      cmds.addCommand( 'no %s mtu-ignore' % intfCliPrefix )


def saveInstanceConfig( vrfConfig, entity, root, ospfConfig, sysdbRoot, options,
                         requireMounts ):
   processId = vrfConfig.processId
   if processId:
      mode = root[ RouterOspf3ConfigMode ].getOrCreateModeInstance( ( processId,
                                                              entity.vrfName ) )
      cmds = mode[ 'Ospf3.instance' ]
   else:
      parentMode = root[ RouterOspfv3ConfigMode ].getOrCreateModeInstance(
                                                            entity.vrfName )
      if entity.instanceId == allAfInstanceID:
         cmds = parentMode[ 'Ospf3MultiAf.instance' ]
      elif entity.instanceId == afIPv4InstanceID:
         mode = parentMode[ RouterOspfv3AfConfigMode ].getOrCreateModeInstance(
                  ( entity.vrfName, 'ipv4', entity.instanceId )  )
         cmds = mode[ 'Ospf3MultiAf.v4Af.config' ]
      elif entity.instanceId == afIPv6InstanceID:
         mode = parentMode[ RouterOspfv3AfConfigMode ].getOrCreateModeInstance(
                  ( entity.vrfName, 'ipv6', entity.instanceId )  )
         cmds = mode[ 'Ospf3MultiAf.v6Af.config' ]

   saveAll = options.saveAll
   
   val = getInstanceConfigAttr( vrfConfig, entity, 'routerId', saveAll )
   if val:
      cmds.addCommand( 'router-id %s' % val )

   val = getInstanceConfigAttr( vrfConfig, entity, 'enable', saveAll )
   if val is False:
      cmds.addCommand('shutdown')

   val = getInstanceConfigAttr( vrfConfig, entity, 'bfdEnabled', saveAll )
   if val:
      cmds.addCommand( 'bfd default' )
   elif val is False:
      cmds.addCommand( 'no bfd default' )

   if Toggles.OspfToggleLib.toggleBfdForAllStatesEnabled():
      val = getInstanceConfigAttr( vrfConfig, entity, 'bfdAnyState', saveAll )
      if val:
         cmds.addCommand( 'bfd adjacency state any' )
      elif val is False:
         cmds.addCommand( 'no bfd adjacency state any' )


   val = getInstanceConfigAttr( vrfConfig, entity, 'refBw', saveAll )
   if val:
      cmds.addCommand( 'auto-cost reference-bandwidth %s' % val )
   elif val is 0:
      cmds.addCommand( 'no auto-cost reference-bandwidth' )
      
   if entity.instanceId != allAfInstanceID:
      # default-metric and distance are AF submode only so we only check
      # entity value if instance is not allAfInstanceID (global mode)
      if entity.defaultMetric != entity.defaultMetricDefault or saveAll:
         cmds.addCommand( 'default-metric %d' % entity.defaultMetric )
      if entity.internalPref != entity.internalPrefDefault or saveAll:
         cmds.addCommand( 'distance ospf intra-area %d' % entity.internalPref )
      if entity.asePref != entity.asePrefDefault or saveAll:
         cmds.addCommand( 'distance ospf external %d' % entity.asePref )

   instancePassive = getInstanceConfigAttr( vrfConfig, entity,
                                    'instancePassive', saveAll )
   if instancePassive:
      cmds.addCommand( 'passive-interface default' )

   if processId:
      # The passive-interface config is actually in IntfConfig.
      # This works since, we support only one instance.
      for intfName in Arnet.sortIntf( ospfConfig.intfConfig ):
         intfConfig = ospfConfig.intfConfig[ intfName ]
         if intfConfig.processId != processId:
            continue
         intfInstanceConfig = intfConfig.intfInstanceConfig.get(
                                 entity.instanceId, None )
         if not intfInstanceConfig:
            continue
         passive = getIntfConfigAttr( intfConfig, intfInstanceConfig,
                                       'passive', saveAll )
         if passive == 'ospfIntfPassiveDisabled':
            cmds.addCommand( 'no passive-interface %s' % intfName )
         elif passive == 'ospfIntfPassiveEnabled':
            cmds.addCommand( 'passive-interface %s' % intfName )

   # FIPS mode command
   if ospfConfig.fipsEnabled:
      cmds.addCommand( 'fips restrictions' )
   # Redistribute cmd is AF submode only so we can just display entity config
   for protocol in sorted( entity.redistributeConfig.values(), 
                           key=lambda p: p.proto ):
      command = 'redistribute %s' % protoDict[ protocol.proto ]
      if protocol.includeLeaked:
         command += ' include leaked'
      if protocol.isisLevel != 'levelNone':
         command += ' %s' % ISIS_LEVEL_MAP[ protocol.isisLevel ]
      if protocol.routeMap:
         command += ' route-map %s' % protocol.routeMap       
      cmds.addCommand( command )

   for area in entity.areaConfig:
      areaConfig = getInstAreaConfig( vrfConfig, entity, area )
      if not areaConfig:
         continue

      if areaConfig.defaultMetric != areaConfig.defaultMetricDefault or saveAll:
         cmds.addCommand( 'area %s default-cost %d' %
                          ( area, areaConfig.defaultMetric ) )

      if areaConfig.areaType == 'areaTypeNssa' or \
             areaConfig.areaType == 'areaTypeNssaNoSummary':
         cmd = 'area %s nssa' % area
         if not areaConfig.nssaTranslateAlways and \
            not areaConfig.disablePBit and \
            areaConfig.nssaDefInfoOrigin != 'ospfDefInfoOriginEnable' and \
            areaConfig.areaType != 'areaTypeNssaNoSummary':
            cmds.addCommand( cmd )
         if areaConfig.areaType == 'areaTypeNssaNoSummary':
            cmdNoSummary = cmd + ' no-summary'
            cmds.addCommand( cmdNoSummary )
         if areaConfig.nssaTranslateAlways:
            cmd1 = 'area %s not-so-stubby lsa type-7 convert type-5' % area
            cmds.addCommand( cmd1 )
         if areaConfig.disablePBit:
            cmd2 = cmd + ' nssa-only'
            cmds.addCommand( cmd2 )
         if areaConfig.nssaDefInfoOrigin == 'ospfDefInfoOriginEnable':
            cmd3 = cmd + ' default-information-originate'
            if areaConfig.nssaDefInfoMetric != areaConfig.nssaDefInfoMetricDefault \
               or saveAll:
               cmd3 += ' metric %d' % areaConfig.nssaDefInfoMetric
            if areaConfig.nssaDefInfoMetricType != \
               areaConfig.nssaDefInfoMetricTypeDefault or saveAll:
               cmd3 += ' metric-type %d' % \
                     int( areaConfig.nssaDefInfoMetricType.lstrip( "extMetricType" )
                        )
            if areaConfig.nssaDefInfoNssaOnly != \
               areaConfig.nssaDefInfoNssaOnlyDefault:
               cmd3 += ' nssa-only'
            cmds.addCommand( cmd3 )
         elif areaConfig.nssaDefInfoOrigin == 'ospfDefInfoOriginSuppress':
            cmds.addCommand( 'no ' + cmd + ' default-information-originate' )
      elif areaConfig.areaType == 'areaTypeStub':
         cmds.addCommand( 'area %s stub' % area )
      elif areaConfig.areaType == 'areaTypeStubNoSummary':
         cmds.addCommand( 'area %s stub no-summary' % area )

      for prefix in sorted( areaConfig.networkList,
                               cmp=IpUtils.compareIp6Prefix ):
         cmd = 'area %s range %s' % ( area, prefix.stringValue )
         if areaConfig.networkList[ prefix ].notAdvertise:
            cmd += ' not-advertise'
         elif areaConfig.networkList[ prefix ].cost:
            cmd += ' cost %d' % areaConfig.networkList[ prefix ].cost
         cmds.addCommand( cmd )
      
      if areaConfig.securityEnabled != areaConfig.securityEnabledDefault:
         securityConfig = requireMounts[ 'mgmt/security/config' ]
         useKey = None if securityConfig.commonKeyEnabled else \
                  area + "_secretKey"
         if areaConfig.sa != areaConfig.saDefault:
            cmd = "area %s " % area
            key = ""
            sa = areaConfig.sa
            passphraseDefined = sa.passphraseKey != ""
            key = sa.passphraseKey if passphraseDefined else \
                  sa.authKey
            if sa.securityProto == "ah":
               cmd += "authentication ipsec spi %d " % sa.spi
            else:
               if sa.encryptAlgo == "null":
                  cmd += "encryption ipsec spi %d esp null " % sa.spi
               else:
                  encryptAlgo = getEncryptAlgo( sa.encryptAlgo )
                  encryptKey = sa.encryptKey
                  cmd += "encryption ipsec spi %d esp %s " \
                        % ( sa.spi, encryptAlgo )
         if passphraseDefined:
            cmd += sa.hmacAlgo + " "
            cmd += "passphrase 7 "
         else:
            if sa.encryptAlgo != "null":
               cmd += "7 "
               eKey = encodeKey( encryptKey, key=useKey, algorithm='MD5' )
               sanitize = lambda word: CliSave.sanitizedOutput( options, word )
               cmd += sanitize( eKey )
               cmd += ' '+ sa.hmacAlgo + ' 7 '
            else:
               cmd += sa.hmacAlgo + ' 7 '
         encryptedKey = encodeKey( key, key=useKey, algorithm='MD5' )
         sanitize = lambda word: CliSave.sanitizedOutput( options, word )
         cmd += sanitize( encryptedKey )
         cmds.addCommand( cmd )

   val = getInstanceConfigAttr( vrfConfig, entity, 'adjacencyLogging', saveAll )
   if val:
      cmd = 'log-adjacency-changes'
      if val == 'adjacencyLoggingTypeDetail':
         cmd += ' detail'
      elif val == 'adjacencyLoggingTypeNone':
         cmd = 'no ' + cmd
      cmds.addCommand( cmd )

   val = getInstanceConfigAttr( vrfConfig, entity, 'spfThrottleTimerConfig',
                                 saveAll )
   if val is not None:
      if entity.timerSpfCommandActive:
         # Backward compatibility, support old timer spf command in old-style CLI
         holdIntSec = val.spfHoldInt / 1000
         cmd = 'timers spf %d' % holdIntSec
      else:
         cmd = 'timers spf delay initial %d %d %d' % ( val.spfStartInt,
                                                       val.spfHoldInt,
                                                       val.spfMaxWaitInt )
      cmds.addCommand( cmd )

   # timers lsa rx min interval
   val = getInstanceConfigAttr( vrfConfig, entity, 'lsaArrivalInt', saveAll )
   if val is not None:
      cmd = 'timers lsa rx min interval %d' % val
      cmds.addCommand( cmd )

   # timers lsa tx delay initial
   val = getInstanceConfigAttr( vrfConfig, entity, 'lsaThrottleTimerConfig',
                                 saveAll )
   if val is not None:
      cmd = 'timers lsa tx delay initial %d %d %d' % ( val.lsaStartInt,
                                                       val.lsaHoldInt,
                                                       val.lsaMaxWaitInt )
      cmds.addCommand( cmd )

   # Maximum-paths is AF submode specific only so we can just print from entity
   if entity.maxEcmp > 0:
      cmd = 'maximum-paths %d' % entity.maxEcmp
      cmds.addCommand( cmd )
   elif saveAll:
      if entity.instanceId == afIPv4InstanceID:
         routingHwStatus = requireMounts[ 'routing/hardware/status' ]
         if routingHwStatus and routingHwStatus.maxEcmp > 0:
            cmd = 'maximum-paths %d' % routingHwStatus.maxEcmp
            cmds.addCommand( cmd )
      elif entity.instanceId == afIPv6InstanceID:
         routing6HwStatus = requireMounts[ 'routing6/hardware/status' ]
         if routing6HwStatus and routing6HwStatus.maxEcmp > 0:
            cmd = 'maximum-paths %d' % routing6HwStatus.maxEcmp
            cmds.addCommand( cmd )

   if entity.floodPacing != entity.floodPacingDefault:
      cmd = 'timers pacing flood %d' % entity.floodPacing
      cmds.addCommand( cmd )
   elif saveAll:
      cmd = 'no timers pacing flood'
      cmds.addCommand( cmd )

   # Default info originate is AF submode specific only so we just print from entity
   if entity.defInfoOrigin == 'ospfDefInfoOriginEnable':
      cmd = 'default-information originate'
      if entity.defInfoOriginAlways != entity.defInfoOriginAlwaysDefault:
         cmd += ' always'
      if entity.defInfoMetric != entity.defInfoMetricDefault or saveAll:
         cmd += ' metric %d' % entity.defInfoMetric
      if entity.defInfoMetricType != entity.defInfoMetricTypeDefault or saveAll:
         cmd += ' metric-type %d' % \
               int( entity.defInfoMetricType.lstrip( "extMetricType" ) )
      if entity.defInfoRouteMap != entity.defInfoRouteMapDefault:
         cmd += ' route-map %s' % entity.defInfoRouteMap
      cmds.addCommand( cmd )
   elif entity.defInfoOrigin == 'ospfDefInfoOriginSuppress':
      cmds.addCommand( 'no default-information originate' )

   maxMetric = entity.maxMetricConfig
   val = getInstanceConfigAttr( vrfConfig, entity, 'maxMetricConfig', saveAll )
   if val:
      cmd = ''
      if maxMetric.maxMetricOn == entity.maxMetricOnDefault and saveAll:
         cmd = 'no max-metric router-lsa'
      else:
         if maxMetric.maxMetricOn != entity.maxMetricOnDefault:
            cmd = 'max-metric router-lsa'
         if maxMetric.maxMetricExtVal != entity.maxMetricExtValDefault:
            cmd += ' external-lsa'
            if maxMetric.maxMetricExtVal != entity.maxMetricLsaValDefault or saveAll:
               cmd += ' %d' % maxMetric.maxMetricExtVal
         if maxMetric.maxMetricStub != entity.maxMetricStubDefault:
            cmd += ' include-stub'
         if maxMetric.maxMetricInterAreaVal != entity.maxMetricInterAreaValDefault:
            cmd += ' inter-area-lsa'
            if maxMetric.maxMetricInterAreaVal != entity.maxMetricLsaValDefault or \
                  saveAll:
               cmd += ' %d' % maxMetric.maxMetricInterAreaVal
         if maxMetric.maxMetricStartTimer != entity.maxMetricStartTimerDefault:
            cmd += ' on-startup %d' % maxMetric.maxMetricStartTimer
         if maxMetric.maxMetricWaitForBgp != entity.maxMetricWaitForBgpDefault:
            cmd += ' on-startup wait-for-bgp'
         if maxMetric.maxMetricPrefixLsa != entity.maxMetricPrefixLsaDefault:
            cmd += ' prefix-lsa'
         if maxMetric.maxMetricStubPrefixVal != entity.maxMetricStubPrefixValDefault:
            cmd += ' stub-prefix-lsa'
            if maxMetric.maxMetricStubPrefixVal != entity.maxMetricLsaValDefault or \
                  saveAll:
               cmd += ' %d' % maxMetric.maxMetricStubPrefixVal
         if maxMetric.maxMetricSumVal != entity.maxMetricSumValDefault:
            cmd += ' summary-lsa'
            if maxMetric.maxMetricSumVal != entity.maxMetricLsaValDefault or saveAll:
               cmd += ' %d' % maxMetric.maxMetricSumVal
      if cmd:
         cmds.addCommand( cmd )

   val = getInstanceConfigAttr( vrfConfig, entity, 'exchStartThreshold', saveAll )
   if val is not None:
      cmds.addCommand( 'adjacency exchange-start threshold %d' % val )

   # timers out-delay <lsa out-delay>
   val = getInstanceConfigAttr( vrfConfig, entity, 'lsaOutDelayTimerConfig',
                                                                  saveAll )
   if val:
      cmd = "timers out-delay %d" % val
      cmds.addCommand( cmd )
   elif val is not None:
      # out-delay 0 is valid value, disables out-delay
      cmd = "no timers out-delay"
      cmds.addCommand( cmd )

   if entity.gracefulRestart != entity.gracefulRestartDefault:
      cmdDisplayed = False
      if entity.grGracePeriod != entity.grGracePeriodDefault:
         cmds.addCommand( 'graceful-restart grace-period %d' %
                          ( entity.grGracePeriod ) )
         cmdDisplayed = True
      elif saveAll:
         cmds.addCommand( 'graceful-restart grace-period %d' %
                          ( entity.grGracePeriodDefault ) )
         cmdDisplayed = True
      if entity.grPlannedOnly != entity.grPlannedOnlyDefault:
         cmds.addCommand( 'graceful-restart planned-only' )
         cmdDisplayed = True
      if not cmdDisplayed:
         cmds.addCommand( 'graceful-restart' )
   elif saveAll:
      cmds.addCommand( 'no graceful-restart' )

   if entity.grHelper != entity.grHelperDefault:
      cmds.addCommand( 'no graceful-restart-helper' )
   elif saveAll:
      cmds.addCommand( 'graceful-restart-helper' )

   if Toggles.Ospf3ToggleLib.toggleOspf3DnBitIgnoreEnabled():
      if entity.ignoreDnBit != entity.ignoreDnBitDefault:
         cmds.addCommand( 'dn-bit-ignore' )
      elif saveAll:
         cmds.addCommand( 'no dn-bit-ignore' )

# db XXX Not required - config CLI disabled
#   if entity.grHelper == entity.grHelperDefault:
#      if entity.grHelperLooseLsaChk != entity.grHelperLooseLsaChkDefault:
#         cmds.addCommand( 'graceful-restart-helper loose-lsa-checking' )
#      elif saveAll:
#         cmds.addCommand( 'graceful-restart-helper' )
#   else:
#      cmds.addCommand( 'no graceful-restart-helper' )

   aclVrfName = entity.vrfName
   aclCpConfig = requireMounts[ 'acl/cpconfig/cli' ]
   serviceAclVrfConfig = aclCpConfig.cpConfig[ 'ipv6' ].serviceAcl.get( aclVrfName )
   if ( serviceAclVrfConfig and
        OspfConsts.serviceName in serviceAclVrfConfig.service and
        serviceAclVrfConfig.service[ OspfConsts.serviceName ].aclName != '' ):
      cmds.addCommand( 'ipv6 access-group %s' %
            serviceAclVrfConfig.service[ OspfConsts.serviceName ].aclName )
   elif saveAll:
      cmds.addCommand( 'no ipv6 access-group' )

#-------------------------------------------------------------------------------
# save config under global config mode
#-------------------------------------------------------------------------------
CliSave.GlobalConfigMode.addCommandSequence( 'Ip6.Ospf',
                                             after=[ RouterOspf3ConfigMode ] )

@CliSave.saver( Ospf3Consts.configType, Ospf3Consts.configPath,
                requireMounts = ( 'routing6/hardware/status',
                                  'interface/config/all', 'interface/status/all',
                                  'acl/cpconfig/cli',
                                  'mgmt/security/config' ) )
def saveConfig( ospfConfig, root, sysdbRoot, options,
                requireMounts ):

   # Check if any Ospf configuration is made.
   ospfVrfConfigured = len( ospfConfig.vrfConfig ) > 0
   ospfInterfacesConfigured = len( ospfConfig.intfConfig ) > 0
   ospfGeneralConfigured = ospfConfig.generalConfig is not None

   ospfConfigEnabled = ( ospfVrfConfigured or ospfInterfacesConfigured or
                         ospfGeneralConfigured )

   # No need to show ospf configs if nothing is configured
   if not ospfConfigEnabled and not options.saveAllDetail:
      return

   if ospfGeneralConfigured and ( options.saveAll or
                                  ospfConfig.generalConfig.dscp !=
                                    ospfConfig.generalConfig.dscpDefault ):
      saveGeneralConfig( ospfConfig, root, options.saveAll )

   # Check if we have any old style configuration, only then proceed.
   # While displaying config in detail-all mode, the new style
   # configuration is preferred hence nothing to do if there isn't
   # any existing old style configuration.
   # Mix-n-match of old and new styles isn't allowed so this is ok
   if not oldStyleConfigPresent( ospfConfig ):
      return

   for vrfConfig in sorted( ospfConfig.vrfConfig.itervalues(),
            key=lambda vrfConfig: vrfConfig.processId ):
      if not vrfConfig.processId:
         continue
      instanceConfig = vrfConfig.instanceConfig[ afIPv6InstanceID ]
      saveInstanceConfig( vrfConfig, instanceConfig, root, ospfConfig,
                           sysdbRoot, options, requireMounts )

   if options.saveAllDetail:
      intfs = allIpIntfNames( sysdbRoot, includeEligible=True,
                              requireMounts=requireMounts )
   elif options.saveAll:
      # Display Ospf configuration on routing protocol interfaces as well as 
      # switchports present in intfConfig collection.
      intfs = set(
            allRoutingProtocolIntfNames( sysdbRoot, requireMounts=requireMounts ) +
            ospfConfig.intfConfig.keys() ) 
   else:
      intfs = ospfConfig.intfConfig
   
   # save config for each interface
   for intfName in intfs:
      if isManagement( intfName ): 
         continue

      intfConfig = ospfConfig.intfConfig.get( intfName )
      if not intfConfig:
         if options.saveAll:
            intfConfig = Tac.newInstance( 'Routing6::Ospf3::IntfConfig', intfName )
            intfConfig.oldStyleCli = True
            intfInstanceConfig = intfConfig.intfInstanceConfig.newMember( 
                                    afIPv6InstanceID, intfName )
            saveIntfConfig( intfConfig, intfInstanceConfig, ospfConfig, root,
                              sysdbRoot, options, requireMounts )
      elif intfConfig.oldStyleCli:
         intfInstanceConfig = intfConfig.intfInstanceConfig[ afIPv6InstanceID ]
         saveIntfConfig( intfConfig, intfInstanceConfig, ospfConfig, root, sysdbRoot,
                         options, requireMounts )

#-------------------------------------------------------------------------------
# Objects used for saving commands in router-ospfv3 Multi AF mode.
#-------------------------------------------------------------------------------
class RouterOspfv3ConfigMode( RoutingOspfv3Mode, CliSave.Mode ):
   def __init__( self, param ):
      RoutingOspfv3Mode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class RouterOspfv3AfConfigMode( RoutingOspfv3AfMode, CliSave.Mode ):
   def __init__( self, param ):
      RoutingOspfv3AfMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( RouterOspfv3ConfigMode,
                                       after=[ IntfConfigMode ] )
RouterOspfv3ConfigMode.addCommandSequence( 'Ospf3MultiAf.instance' )

RouterOspfv3ConfigMode.addChildMode( RouterOspfv3AfConfigMode )
RouterOspfv3AfConfigMode.addCommandSequence( 'Ospf3MultiAf.v4Af.config' )
RouterOspfv3AfConfigMode.addCommandSequence( 'Ospf3MultiAf.v6Af.config' )

IntfConfigMode.addCommandSequence( 'Ospf3MultiAf.intf', after=[ 'Ira.intf6' ] )

CliSave.GlobalConfigMode.addCommandSequence( 'Ip6.Ospf3MultiAf',
                                             after=[ RouterOspfv3ConfigMode ] )

# Returns the interface OSPFv3 config attribute value to display in CliSave by
# checking the interface instance config collection. The attribute value
# returned can be zero for non-bool attributes. The caller must check return
# value is not None to determine if the attribute should be saved.
#
# For saving config in either of the V4/V6/global instance config collections we
# check if the attr is set in the AF instance config and save. We also
# save the attr value if saveAll flag is set.
#
def getIntfConfigAttr( intfConfig, intfInstanceConfig, attr, saveAll ):
   if saveAll or getattr( intfInstanceConfig, attr + "Present", False ):
      return getattr( intfInstanceConfig, attr )
   return None

# Returns the instance OSPFv3 config attribute value to display in CliSave by
# checking the instance config collection in vrfConfig. Similar to
# getIntfConfigAttr but this routine is for instance config. The only difference
# is we check for old-style CLI is by checking processId != 0 i.e.,
# for saveAll case we only want to display the attribute for global multi AF or
# the old style config.
# The attribute value returned can be zero for non-bool attributes. The caller
# must check return value is not None to determine if the attribute should be
# saved.
def getInstanceConfigAttr( vrfConfig, instanceConfig, attr, saveAll ):
   instanceId = instanceConfig.instanceId
   v4ConfigPresent = v6ConfigPresent = False

   if instanceId == allAfInstanceID:
      v4InstConfig = vrfConfig.instanceConfig.get( afIPv4InstanceID, None )
      if v4InstConfig and getattr( v4InstConfig, attr + "Present", False ):
         v4ConfigPresent = True
      v6InstConfig = vrfConfig.instanceConfig.get( afIPv6InstanceID, None )
      if v6InstConfig and getattr( v6InstConfig, attr + "Present", False ):
         v6ConfigPresent = True

      if ( saveAll and not v4ConfigPresent and not v6ConfigPresent ) \
               or getattr( instanceConfig, attr + "Present", False ):
         return getattr( instanceConfig, attr )
   else:
      if ( saveAll and vrfConfig.processId ) or \
            getattr( instanceConfig, attr + 'Present', False ):
         return getattr( instanceConfig, attr )
   return None

# Returns True if any of the area config attributes have been set
# Used to check if the area configuration in a particular AF instance
# should override the global area config.
#
def areaConfigOverridden( instanceConfig, areaId ):
   # check if all the attributes of the area are not changed from default
   areaConfig = instanceConfig.areaConfig.get( areaId, None )
   if not areaConfig:
      return False
   for attr in areaConfig.attributes:
      # For singleton attributes in areaConfig, check defaults
      try:
         if getattr( areaConfig, attr ) != \
                getattr( areaConfig, attr + 'Default' ):
            return True
      except AttributeError:
         # For collections, check the length of collection
         if hasattr( attr, "items" ):
            if getattr( areaConfig, attr ):
               return True
   return False

# When saving instance area configuration this routine is called to return
# the global area config in the all AF instance only if none of the AF
# instances have overridden area configuration with the same area ID.
# If area config has been overridden in AF submodes then return None and
# we will save only the AF specific area configuration.
#
def getInstAreaConfig( vrfConfig, instanceConfig, areaId ):
   instanceId = instanceConfig.instanceId
   if instanceId == allAfInstanceID:
      v4InstConfig = vrfConfig.instanceConfig.get( afIPv4InstanceID, None )
      if v4InstConfig and areaConfigOverridden( v4InstConfig, areaId ):
         return None 
      v6InstConfig = vrfConfig.instanceConfig.get( afIPv6InstanceID, None )
      if v6InstConfig and areaConfigOverridden( v6InstConfig, areaId ):
         return None
   return instanceConfig.areaConfig.get( areaId, None )

