# Copyright (c) 2009, 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements the CliSave code for routing bgp commands
#-------------------------------------------------------------------------------
from Toggles import (
   BgpCommonToggleLib,
   IpRibLibToggleLib,
   RcfLibToggleLib,
   RoutingLibToggleLib
)
import CliSave, Tac, Tracing
from IntfCliSave import IntfConfigMode
from BgpLib import ( isGracefulRestartGlobalOrAfConfigured, createKey, Ip6AddrType,
                     routeTargetToPrint )
import BgpLib
from CliMode.BgpCommon import RoutingBgpBaseMode, RoutingBgpVrfMode
from CliMode.BgpCommon import RoutingBgpBaseAfMode, RoutingBgpVrfAfMode
# dependency on the MgmtSecurity for "mgmt/security/config"
# pkgdeps: import CliSavePlugin.Security
import MultiRangeRule
from ArnetLib import bgpFormatAsn, formatRd
from RouteMapLib import isAsdotConfigured
import CliExtensions
from ReversibleSecretCli import encodeKey
from ProtoEnums import protoDict
import Intf.IntfRange as IntfRange
from BgpLib import (
      addressFamilies,
      vpnAfTypeCliKeywords,
      vpnAfTypeMap,
      vpnTypeAttrMap,
)
from IpLibConsts import DEFAULT_VRF

traceHandle = Tracing.Handle( 'Bgp' )
#pylint: disable-msg=E1101
#pylint: disable-msg=C0321
t0 = traceHandle.trace0
t1 = traceHandle.trace1
t2 = traceHandle.trace2

def vpnNlriTypeKeyword( vpnNlriType ):
   return { 'evpnType5Ipv4': 'evpn ipv4',
            'evpnType5Ipv6': 'evpn ipv6',
            'mplsVpnIpv4Unicast': 'vpn-ipv4',
            'mplsVpnIpv6Unicast': 'vpn-ipv6' }[ vpnNlriType.toStrep() ]

# The CLI hook for BGP is used to allow external features dependant on BGP mode
# to clear their configurations. This function is called as
#
# hook( bgpConfig, defaultVrfBgpConfig, saveAll )
#
# bgpConfig is the current VRF Bgp::Config
# defaultVrfBgpConfig is the default VRF Bgp::Config
# saveAll is an boolean that must print the config
#
# Expects to return a cli command as a string or None
# if the config doesn't exist
bgpMonitoringHook = CliExtensions.CliHook()

# The CLI hook for BGP is used to allow external features dependant on BGP mode
# to clear their configurations. This function is called as
#
# hook( peer, peerConfig, saveAll )
#
# peer is the BGP peer address or peer group name
# peerConfig is the configuration of the peer in the current VRF
# saveAll is an boolean that must print the config
#
# Expects to return a cli command as a string or None
# if the config doesn't exist
neighborMonitoringHook = CliExtensions.CliHook()
neighborRpkiHook = CliExtensions.CliHook()

reflectedRouteAttrStateEnum = Tac.Type( "Routing::Bgp::ReflectedRouteAttrState" )
redistInternalStateEnum = Tac.Type( "Routing::Bgp::RedistInternalState" )
SoftReconfigInboundStateEnum = Tac.Type( "Routing::Bgp::SoftReconfigInboundState" )
MetricOutStateEnum = Tac.Type( "Routing::Bgp::MetricOutState" )
triStateBoolEnum = Tac.Type( "Ip::TristateBool" )
SendCommunityLinkBandwidthStateEnum = \
    Tac.Type( "Routing::Bgp::SendCommunityLinkBandwidthState" )
LinkBandwidthGenerationStateEnum = \
    Tac.Type( "Routing::Bgp::LinkBandwidthGenerationState" )
ExtendedNextHopCapabilityEnum = Tac.Type( "Routing::Bgp::ExtendedNextHopCapability" )
policyActionEnum = Tac.Type( "Routing::Bgp::PolicyActionType" )
outDelayApplyEnum = Tac.Type( "Routing::Bgp::OutDelayApplyType" )
NlriType = Tac.Type( "Routing::Bgp::NlriType" )

class RouterBgpBaseConfigMode( RoutingBgpBaseMode, CliSave.Mode ):

   def __init__( self, param ):
      assert isinstance( param, tuple )
      asn, asdotConfigured = param
      asnParam = bgpFormatAsn( asn, asdotConfigured )
      RoutingBgpBaseMode.__init__( self, asnParam )
      CliSave.Mode.__init__( self, asnParam )

class RouterBgpVrfConfigMode( RoutingBgpVrfMode, CliSave.Mode ):

   def __init__( self, param ):
      RoutingBgpVrfMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

routeTypeDict = { 'protoOspf' : 'internal', 'protoOspfAse' : 'external',
      'protoOspfNssa' : 'nssa-external', 'protoOspf3' : 'internal', 
      'protoOspf3Ase' : 'external', 'protoOspf3Nssa' : 'nssa-external' }

isLevelDict = { 1 : 'level-1', 2 : 'level-2', 3 : 'level-1-2' }
def redistributeCommand( protocol, saveAll=False ):
   # internally we redistribute ospf-ase along with ospf depending on whether
   # it is match external or internal.
   if type( protocol ) is str:
      proto = protocol
      routeType = ''
      metricType = 0
      isLevel = 0
      leaked = False
   else:
      proto = protocol.proto
      routeType = protocol.routeType
      metricType = protocol.metricType
      isLevel = protocol.isisLevel
      leaked = protocol.includeLeaked

   command = ''
   if proto in ( 'protoOspf', 'protoOspfAse', 'protoOspfNssa',
         'protoOspf3', 'protoOspf3Ase', 'protoOspf3Nssa' ):
      if proto in ( 'protoOspf', 'protoOspfAse', 'protoOspfNssa' ):
         command += 'redistribute ospf'
      else:
         command += 'redistribute ospfv3'
      if routeType != '': 
         command += ' match %s' % routeType
      elif saveAll:
         command += ' match %s' % routeTypeDict[ proto ]
      if metricType != 0:
         command += ' %d' % metricType
   elif proto == 'protoIsis':
      if isLevel:
         command += 'redistribute isis %s' % isLevelDict[ isLevel ]
      elif saveAll:
         command += 'redistribute isis'
   else:
      command += 'redistribute %s' % protoDict[ proto ]
   if leaked:
      if proto == 'protoBgp':
         command += ' leaked'
      else:
         command += ' include leaked'
   return command

def addPathSendCommon( app, appConfig=None, saveAll=False ):
   command = 'additional-paths send '
   if app == 'appAny':
      command += 'any' 
   else:
      raise ValueError()
   return command

peerAfStateEnum = Tac.Type( "Routing::Bgp::PeerAFState" )
def hasAfConfig( bgpConfig ):
   for nconf in bgpConfig.neighborConfig.values():
      peerAfStateAttrs = BgpLib.peerConfigAttrsAfMap[ 'af' ]
      for attrName in peerAfStateAttrs.values():
         if getattr( nconf, attrName ) != peerAfStateEnum.afDefault:
            return True
   return False

# The 'service ...' command must appear before interface configuration, 
# see BUG163576
# Additionally, the 'service ...' command must appear before BGP configuration.
# This is ensured by rendering 'Bgp.service' before IntfConfigMode while
# RouterBgpBaseConfigMode comes after IntfConfigMode.
CliSave.GlobalConfigMode.addCommandSequence( 'Bgp.service',
                                             before=[ IntfConfigMode ] )
CliSave.GlobalConfigMode.addChildMode( RouterBgpBaseConfigMode,
                                       after=[ IntfConfigMode ] )
RouterBgpBaseConfigMode.addCommandSequence( 'Bgp.config' )

RouterBgpBaseConfigMode.addChildMode( RouterBgpVrfConfigMode )
RouterBgpVrfConfigMode.addCommandSequence( 'Bgp.vrf.config' )

# Returns a tuple: isSet, value
#  isSet identifies if the attribute 'attr' is set in the vrfBgpConfig.
#  If the value inside vrfBgpConfig is "invalid", we try inheriting from
#  the baseBgpConfig.
def evaluateValue( vrfBgpConfig, baseBgpConfig, attr, defVal=None, invalVal=None ):
   if defVal == None:
      defVal = getattr( vrfBgpConfig, attr + 'Default' )
   if invalVal == None:
      invalVal = getattr( vrfBgpConfig, attr + 'Invalid' )
   val = getattr( vrfBgpConfig, attr )
   if val == invalVal:
      val = getattr( baseBgpConfig, attr )
      if val == invalVal:
         return False, defVal
      else:
         return False, val
   return True, val

# Returns a tuple: isSet, bool-value
def evaluateTristate( vrfBgpConfig, baseBgpConfig, attr ):
   meaningOfInvalid = getattr( vrfBgpConfig, attr + 'Default' )
   val = getattr( vrfBgpConfig, attr ) 
   if val == 'isInvalid':
      isSet = False
      val = getattr( baseBgpConfig, attr ) 
      if val == 'isInvalid':
         return isSet, meaningOfInvalid
      else:
         return ( isSet, True ) if val == 'isTrue' else ( isSet, False )
   else:
      isSet = True
      return ( isSet, True ) if val == 'isTrue' else ( isSet, False )

# Returns a tuple: isSet, pref-value
def evaluateRoutePref( vrfBgpConfig, baseBgpConfig, attr ):
   rp = getattr( vrfBgpConfig, attr ) 
   rpDef = getattr( baseBgpConfig, attr ) 
   isSet = rp.isSet
   val = rp.pref if isSet else rpDef.pref
   return isSet, val

def isGrConfigured( vrfBgpConfig, baseBgpConfig ):
   def _isGrConfigured( bgpOrPeerConfig, configuredValue ):
      grAttrs = BgpLib.bgpConfigAttrsAfMap[ 'gracefulRestart' ]
      for attrName in grAttrs.values():
         if getattr( bgpOrPeerConfig, attrName ) == configuredValue:
            return True
      return False

   # Check if GR is in the BGP config
   if _isGrConfigured( vrfBgpConfig, triStateBoolEnum.isTrue ):
      return True
   # Or check if a static nbr, dynPeerGroup, static Peergroup is configured for GR
   for neighborConfig in vrfBgpConfig.neighborConfig.values():
      if not neighborConfig.isPeerGroup and _isGrConfigured( neighborConfig, True ):
         return True
      if neighborConfig.isPeerGroupPeer:
         peerGroupConfig = \
               baseBgpConfig.neighborConfig.get( neighborConfig.peerGroupKey )
         if peerGroupConfig and _isGrConfigured( peerGroupConfig, True ): 
            return True
   for pgName in vrfBgpConfig.listenRange:
      peerGroupConfig = baseBgpConfig.neighborConfig.get( createKey( pgName ) )
      if peerGroupConfig and _isGrConfigured( peerGroupConfig, True ):
         return True
   return False

def getExplicitNoCmd( cmd, nedMode, defaultValue=None ):
   if nedMode:
      if defaultValue is not None:
         # explicit-no default value for remote-port is 179, not 0
         if 'transport remote-port' in cmd:
            defaultValue = 179
         return cmd + ' ' + str( defaultValue )
      else:
         # If no defaultValue is provided, this command has a 'disabled' form
         return cmd + ' disabled'
   else:
      return 'no ' + cmd

def weightConfiguredInAf( config ): 
   attrs = BgpLib.peerConfigAttrsAfMap[ 'weight' ]
   weightAfAttrs = [ ( af, c ) for af, c in attrs.iteritems() if af != 'all' ]
   for _, c in weightAfAttrs:
      if getattr( config, c + 'Present' ):
         return True
   return False

def outDelayConfiguredInAf( config ):
   attrs = BgpLib.peerConfigAttrsAfMap[ 'outDelay' ]
   outDelayAfAttrs = [ ( af, c ) for af, c in attrs.iteritems() if af != 'all' ]
   for _, c in outDelayAfAttrs:
      if getattr( config, c + 'Present' ):
         return True
   return False

def multiRangeToBgpAsdotString( inputString ):
   result = []
   tmpList = inputString.split()
   for substr in tmpList:
      if '-' in substr: 
         low, high = substr.split( '-' )
         lowInAsdot = bgpFormatAsn( int(low), True ) 
         highInAsdot = bgpFormatAsn( int(high), True )
         result.append( lowInAsdot + '-' + highInAsdot )
      else:
         result.append( bgpFormatAsn( int(substr), True ) )
   return ' '.join( result )

def saveNetworkList( cmds, bgpConfig, addrFamily='all', filterNetworkAf=None ):
   attrName = BgpLib.bgpConfigAttrsAfMap[ 'networkList' ].get( addrFamily )
   networkList = getattr( bgpConfig, attrName )
   for network in networkList.values():
      if filterNetworkAf and network.network.af != filterNetworkAf:
         continue
      cmd =  'network %s%s' % ( network.network, '' )
      # Once we support 'restrict', then:
      #' restrict' if network.restricted else '' ) )
      routeMap = network.routeMap
      if routeMap is not None and routeMap != "" :
         cmd += " route-map %s" % routeMap
      cmds.append( cmd )

def saveBgpVpnConfig( vrfName, bgpConfig, defaultVrfBgpConfig, asdotConfigured,
                      saveAll ):
   """This method returns the VPN config for the given VRF"""
   cmds = []
   for importExport, attrName in [ ( 'import', 'routeTargetImport' ),
                                   ( 'export', 'routeTargetExport' ) ]:
      for rt, isRtSet in getattr( bgpConfig, attrName ).iteritems():
         assert isRtSet
         cmds.append( "route-target %s %s" % ( importExport,
            routeTargetToPrint( rt, asdotConfigured ) ) )
      # Display these sorted on the cli keyword for the vpnAf.
      for vpnType, vpnTypeStr in sorted(
            vpnTypeAttrMap.items(),
            key=( lambda ( vpnType, vpnTypeStr ): vpnTypeStr ) ):
         vpnTypeAttrName = attrName + vpnType
         for rt, isRtSet in getattr( bgpConfig, vpnTypeAttrName ).iteritems():
            assert isRtSet
            cmds.append( "route-target %s %s %s" % ( importExport,
                         vpnTypeStr, routeTargetToPrint( rt, asdotConfigured ) ) )

   for importExport, attrName in [ ( 'import', 'routeMapImport' ),
                                   ( 'export', 'routeMapExport' ) ]:
      # An explicit key is required for sort method here because vpnAf is of type
      # Routing::Bgp::AfiSafi but we need to sort them based on thier cli keyword
      # here.
      for vpnAf, rmName in sorted(
            getattr( bgpConfig, attrName ).items(),
                     key=lambda item: vpnAfTypeMap.get( item[ 0 ] ) ):
         if vpnAfTypeMap.get( vpnAf ):
            cmds.append( "route-target %s %s route-map %s" % ( importExport,
               vpnAfTypeMap[ vpnAf ], rmName ) )

   for importExport, attrName in [ ( 'export', 'allowImportedRouteToExport' ) ]:
      for vpnAf, isImpRoSet in sorted(
            getattr( bgpConfig, attrName ).items(),
                     key=lambda item: vpnAfTypeMap.get( item[ 0 ] ) ):
         assert isImpRoSet
         if vpnAfTypeMap.get( vpnAf ):
            cmds.append( "route-target %s %s %s" % ( importExport,
                         vpnAfTypeMap[ vpnAf ], 'imported-route' ) )

   for peerKey in bgpConfig.neighborConfig:
      addrOrPgName = peerKey.stringValue
      peer = bgpConfig.neighborConfig.get( peerKey )
      savePeerRouteTargetExportFilterDisabled( peer, cmds, addrOrPgName, 'all',
                                               saveAll )

   return cmds

def saveBgpConfig( bgpConfig, defaultVrfBgpConfig, 
                   routingHwStatus, vrfName, securityConfig, aclCpConfig,
                   asdotConfigured=False, options=None ):
   cmds = []
   saveAll = options.saveAll if options else False
   afCfgPresent = hasAfConfig( bgpConfig )

   # "command clear all disabled" is never written to for a non-default VRF
   if not vrfName:
      if ( defaultVrfBgpConfig.clearAllDisabled !=
           defaultVrfBgpConfig.clearAllDisabledDefault ):
         cmds.append( 'command clear all disabled' )
      elif saveAll:
         cmds.append( 'no command clear all disabled' )

   cmd = 'bgp labeled-unicast rib'
   if bgpConfig.bgpLuRib.rib == 'ipAndTunnel':
      cmd += ' ip'
      if bgpConfig.bgpLuRib.ipRmName != \
         bgpConfig.bgpLuRib.rmNameDefault:
         cmd += ' route-map %s' % bgpConfig.bgpLuRib.ipRmName
      cmd += ' tunnel'
      if bgpConfig.bgpLuRib.tunnelRmName != \
         bgpConfig.bgpLuRib.rmNameDefault:
         cmd += ' route-map %s' % bgpConfig.bgpLuRib.tunnelRmName
   elif bgpConfig.bgpLuRib.rib == 'tunnel':
      cmd += ' tunnel'
      if bgpConfig.bgpLuRib.tunnelRmName != \
         bgpConfig.bgpLuRib.rmNameDefault:
         cmd += ' route-map %s' % bgpConfig.bgpLuRib.tunnelRmName
   elif bgpConfig.bgpLuRib.rib == 'ip':
      cmd += ' ip'
      if bgpConfig.bgpLuRib.ipRmName != \
         bgpConfig.bgpLuRib.rmNameDefault:
         cmd += ' route-map %s' % bgpConfig.bgpLuRib.ipRmName
   else:
      assert False, "Unexpected BGP LU RIB"

   if cmd != 'bgp labeled-unicast rib tunnel' or saveAll:
      cmds.append( cmd )

   # The per VPN route-target and route-maps are printed in sorted order based on
   # their display vpnAf string. The display vpnAf string (cli keyword for the vpn
   # Af) is storeed in vpnAfTypeCliKeywords in a sorted manner.

   if vrfName:
      if bgpConfig.asNumber != 0:
         cmds.append( 'local-as %d' % bgpConfig.asNumber )
      elif saveAll:
         cmds.append( 'local-as %d' % defaultVrfBgpConfig.asNumber )
      cmds.extend( saveBgpVpnConfig( vrfName, bgpConfig, defaultVrfBgpConfig,
                                     asdotConfigured, saveAll ) )

   if bgpConfig.shutdown != bgpConfig.shutdownDefault:
      if bgpConfig.shutdownMsg != bgpConfig.shutdownMsgDefault:
         reasonStr = " reason %s" % bgpConfig.shutdownMsg
      else:
         reasonStr = ""
      cmds.append( 'shutdown%s' % reasonStr )
   elif saveAll:
      cmds.append( 'no shutdown' )
   
   if bgpConfig.cpFilterDefaultAllow:
      cmds.append( 'bgp control-plane-filter default-allow' )
   elif saveAll:
      cmds.append( 'no bgp control-plane-filter default-allow' )

   if bgpConfig.routerId != bgpConfig.routerIdDefault:
      cmds.append( 'router-id %s' % bgpConfig.routerId )
   elif saveAll:
      cmds.append( 'no router-id' )

   isStSet, stVal = evaluateValue( bgpConfig, defaultVrfBgpConfig,
                                   'convergenceTime' )
   if isStSet or saveAll:
      cmds.append( 'bgp convergence time %d' % stVal )

   isStSet, stVal = evaluateValue( bgpConfig, defaultVrfBgpConfig,
                                   'convergenceIdlePeerTime' )
   if isStSet or saveAll:
      cmds.append( 'bgp convergence slow-peer time %d' % stVal )

   if bgpConfig.confedId != 0:
      cmds.append( 'bgp confederation identifier %s' %
            bgpFormatAsn( bgpConfig.confedId, asdotConfigured ) )
   elif saveAll:
      cmds.append( 'no bgp confederation identifier' )

   for substr in MultiRangeRule.multiRangeToCanonicalStringGen(
            bgpConfig.confedPeer, 50, ' ' ):
      if asdotConfigured:
         substr = multiRangeToBgpAsdotString( substr )
      cmds.append( 'bgp confederation peers %s' % substr )

   isSet, val = evaluateTristate( bgpConfig, defaultVrfBgpConfig, 
                                  'convergenceNoSync' )
   if isSet or saveAll:
      cmds.append( ( 'no ' if val else '' ) +
                         'update wait-for-convergence' )

   isSet = bgpConfig.fibSync != bgpConfig.fibSyncDefault
   if isSet or saveAll:
      if bgpConfig.fibSync.batchSize:
         batchStr = ' batch-size %s' % (bgpConfig.fibSync.batchSize)
      else:
         batchStr = ''
      cmds.append( '%supdate wait-install%s' \
            %('' if isSet else 'no ', batchStr ) )

   isSet, val = evaluateTristate( bgpConfig, defaultVrfBgpConfig, 
                                  'logNeighborChanges' )
   if isSet or saveAll:
      cmds.append( ( '' if val else 'no ' ) + 'bgp log-neighbor-changes' )

   isSet, val = evaluateTristate( bgpConfig, defaultVrfBgpConfig, 'defaultV4Uni' )
   if isSet or saveAll:
      cmds.append( ( '' if val else 'no ' ) + 'bgp default ipv4-unicast' )

   isSet, val = evaluateTristate( bgpConfig, defaultVrfBgpConfig,
                                  'defaultV4UniOverV6' )
   if isSet or saveAll:
      cmds.append( ( '' if val else 'no ' ) +
                       'bgp default ipv4-unicast transport ipv6' )

   isSet, val = evaluateTristate( bgpConfig, defaultVrfBgpConfig, 'defaultV6Uni' )
   if isSet or saveAll:
      cmds.append( ( '' if val else 'no ' ) + 'bgp default ipv6-unicast' )

   policyActions = dict(
      policyActionUnspecified = 'unspecified',
      policyActionPermit = 'permit',
      policyActionDeny = 'deny',
      policyActionDenyBoth = 'deny-in-out' )
   for direction in [ "In", "Out" ]:
      tok = [ 'bgp', 'missing-policy', 'direction', direction.lower(), 'action' ]
      isSet, val = evaluateValue( bgpConfig, defaultVrfBgpConfig, 
                                  'missingPolicyAction' + direction )
      addrFamilyAllSet, addrFamilyAllVal = evaluateTristate(
                                                 bgpConfig, defaultVrfBgpConfig,
                                                 'missingPolicyAddressFamilyAll' +
                                                 direction )
      subRmSet, subRmVal = evaluateTristate( bgpConfig, defaultVrfBgpConfig,
                                             'missingPolicyIncludeSubRouteMap' +
                                             direction )
      pfxListSet, pfxListVal = evaluateTristate( bgpConfig, defaultVrfBgpConfig,
                                                 'missingPolicyIncludePrefixList' +
                                                 direction )
      if isSet:
         tok.append( policyActions[ val ] )
         insertPos = 2
         if addrFamilyAllSet and addrFamilyAllVal:
            tok[ insertPos : insertPos ] = [ 'address-family', 'all' ]
            insertPos += 2
         if ( subRmSet and subRmVal ) or ( pfxListSet and pfxListVal ):
            tok.insert( insertPos, 'include' )
            insertPos += 1
            if subRmSet and subRmVal:
               # insert these keywords into the command token list
               tok.insert( insertPos, 'sub-route-map' )
               insertPos += 1
            if pfxListSet and pfxListVal:
               # insert these keywords into the command token list
               tok.insert( insertPos, 'prefix-list' )
               insertPos += 1
         missingPolicyCommand = ' '.join( tok )
         cmds.append( missingPolicyCommand )
      elif saveAll:
         missingPolicyCommand = 'no ' + ' '.join( tok )
         cmds.append( missingPolicyCommand )

   isKtSet, ktVal = evaluateValue( bgpConfig, defaultVrfBgpConfig, 'keepaliveTime' )
   isHtSet, htVal = evaluateValue( bgpConfig, defaultVrfBgpConfig, 'holdTime' )
   if isKtSet or isHtSet or saveAll:
      cmds.append( 'timers bgp %d %d' % ( ktVal, htVal ) )

   idSet, idVal = evaluateRoutePref( bgpConfig, defaultVrfBgpConfig,
                                     'internalDistance' )
   edSet, edVal = evaluateRoutePref( bgpConfig, defaultVrfBgpConfig,
                                     'externalDistance' )
   ldSet, ldVal = evaluateRoutePref( bgpConfig, defaultVrfBgpConfig,
                                     'localDistance' )
   if edSet or idSet or ldSet or saveAll:
      cmds.append( 'distance bgp %d %d %d' % ( edVal, idVal, ldVal ) )

   isRtSet, rtVal = evaluateValue( bgpConfig, defaultVrfBgpConfig,
                                   'grRestartTime' )
   if isRtSet or saveAll or isGrConfigured( bgpConfig, defaultVrfBgpConfig ):
      cmds.append( 'graceful-restart restart-time %d' % rtVal )

   isStSet, stVal = evaluateValue( bgpConfig, defaultVrfBgpConfig,
                                   'grStalepathTime' )
   if isStSet or saveAll:
      cmds.append( 'graceful-restart stalepath-time %d' % stVal )

   aaVal = bgpConfig.allowAs
   if aaVal != bgpConfig.allowAsDefault:
      if aaVal == bgpConfig.allowAsEnabledDefault:
         cmds.append( 'bgp allowas-in' )
      else:
         cmds.append( 'bgp allowas-in %s' % ( aaVal ) )
   elif saveAll:
      if aaVal:
         cmds.append( 'bgp allowas-in %s' % ( aaVal ) )
      else:
         cmds.append( 'no bgp allowas-in' )

   if bgpConfig.bgpClusterId != bgpConfig.bgpClusterIdDefault:
      cmds.append( 'bgp cluster-id %s' % ( bgpConfig.bgpClusterId ) )
   elif saveAll:
      cmds.append( 'no bgp cluster-id' )

   if not vrfName and bgpConfig.autoRd:
      cmds.append( 'rd auto' )
   elif saveAll and not vrfName:
      cmds.append( 'no rd auto' )

   if not vrfName:
      for vpnNlriType in [ NlriType( n ) for n in
                           [ 'evpnType5Ipv4', 'evpnType5Ipv6',
                             'mplsVpnIpv4Unicast', 'mplsVpnIpv6Unicast' ] ]:
         present = vpnNlriType in bgpConfig.routeTargetExportFilterDisabled
         if present or saveAll:
            cmds.append( '%sroute-target export %s filter%s' %
                         ( '' if present else 'no ',
                           vpnNlriTypeKeyword( vpnNlriType ),
                           ' disabled' if present else '' ) )

   if bgpConfig.rrDisabled:
      cmds.append( 'no bgp client-to-client reflection' )
   elif saveAll:
      cmds.append( 'bgp client-to-client reflection' )

   isGrSet, grVal = evaluateTristate( bgpConfig, defaultVrfBgpConfig,
                                      'gracefulRestart' )
   if isGrSet or saveAll:
      cmds.append( ( '' if grVal else 'no ' ) + 'graceful-restart' )

   isGrHelperSet, grHelperVal = evaluateTristate( bgpConfig, defaultVrfBgpConfig,
                                      'grHelper' )
   isGrHelperRestartTimeSet, grHelperRestartTimeVal = evaluateValue( bgpConfig,
            defaultVrfBgpConfig, 'grHelperRestartTime' )
   isLlgrHelperSet, llgrHelperVal = evaluateTristate( bgpConfig, defaultVrfBgpConfig,
                                                      'llgrHelper' )

   grHelperCmd = 'graceful-restart-helper'
   if isGrHelperSet and not grHelperVal:
      cmds.append( 'no graceful-restart-helper' )
   elif isGrHelperRestartTimeSet or isLlgrHelperSet:
      grHelperCmdStr = grHelperCmd
      if isGrHelperRestartTimeSet and grHelperRestartTimeVal:
         grHelperCmdStr += ' restart-time %d' % grHelperRestartTimeVal
      if isLlgrHelperSet and llgrHelperVal:
         grHelperCmdStr += ' long-lived'
      cmds.append( grHelperCmdStr )
   elif saveAll:
      cmds.append( grHelperCmd )

   isPmrtSet, pmrtVal = evaluateValue( bgpConfig, defaultVrfBgpConfig,
                                       'peerMacResolutionTimeout' )
   if isPmrtSet or saveAll:
      cmds.append( 'bgp peer-mac-resolution-timeout %d' % pmrtVal )

   isEnforceFirstAsSet, enforceFirstAsVal = evaluateTristate( bgpConfig,
                                                              defaultVrfBgpConfig,
                                                              'enforceFirstAs' )
   if isEnforceFirstAsSet and not enforceFirstAsVal:
      cmds.append( 'no bgp enforce-first-as' )
   elif saveAll:
      cmds.append( 'bgp enforce-first-as' )

   isfecSkipIarDivDetSet, fecSkipIarDivDetVal = evaluateTristate( bgpConfig,
      defaultVrfBgpConfig, 'fecSkipIarDivergenceDetection' )
   isfecSkipIarSet, fecSkipIarVal = evaluateTristate( bgpConfig,
      defaultVrfBgpConfig, 'fecSkipIarTracking' )
   if isfecSkipIarDivDetSet:
      if fecSkipIarDivDetVal:
         cmds.append( 'bgp fec skip in-place update event route-changes' )
      else:
         cmds.append( 'no bgp fec skip in-place update' )
   elif isfecSkipIarSet:
      if fecSkipIarVal:
         cmds.append( 'bgp fec skip in-place update event all' )
      else:
         cmds.append( 'no bgp fec skip in-place update' )
   elif saveAll:
      cmds.append( 'no bgp fec skip in-place update' )

   isfecSkipIarDrUndrSet, fecSkipIarDrUndrVal = evaluateTristate( bgpConfig,
      defaultVrfBgpConfig, 'fecSkipIarDrainUndrain' )
   if isfecSkipIarDrUndrSet:
      if fecSkipIarDrUndrVal:
         cmds.append( 'no bgp fec in-place update event drain-undrain' )
      else:
         cmds.append( 'bgp fec in-place update event drain-undrain' )
   elif saveAll:
      cmds.append( 'no bgp fec in-place update event drain-undrain' )

   isfecSkipIarPeerInitSet, fecSkipIarPeerInitVal = evaluateTristate( bgpConfig,
      defaultVrfBgpConfig, 'fecSkipIarPeerInit' )
   if isfecSkipIarPeerInitSet:
      if fecSkipIarPeerInitVal:
         cmds.append( 'no bgp fec in-place update event peer-init' )
      else:
         cmds.append( 'bgp fec in-place update event peer-init' )
   elif saveAll:
      cmds.append( 'no bgp fec in-place update event peer-init' )

   disabledVal = bgpConfig.fecIarTimeoutDisabled
   defVal = bgpConfig.fecIarTimeoutDefault
   iarTimeout = bgpConfig.fecIarTimeout
   isSet = ( bgpConfig.fecIarTimeout != bgpConfig.fecIarTimeoutDefault )
   if iarTimeout == defVal:
      # IAR timeout is not explicitly configured at BGP VRF level. Inherit
      # the value from default VRF
      iarTimeout = defaultVrfBgpConfig.fecIarTimeout
   if iarTimeout == bgpConfig.fecIarTimeoutDefault:
      if saveAll:
         cmds.append( 'no bgp fec in-place update timeout' )
   elif isSet or saveAll:
      if iarTimeout == disabledVal:
         cmds.append( 'bgp fec in-place update timeout disabled' )
      else:
         cmds.append( 'bgp fec in-place update timeout %d' % iarTimeout )

   if bgpConfig.routeInstallMap != bgpConfig.routeInstallMapDefault:
      cmds.append( 'bgp route install-map %s' % 
                       bgpConfig.routeInstallMap )
   elif saveAll:
      cmds.append( 'no bgp route install-map' )

   aclVrfName = vrfName
   if vrfName == '':
      aclVrfName = 'default'
   for aclType in [ 'ip', 'ipv6' ]:
      serviceAclConfig = aclCpConfig.cpConfig[ aclType ].serviceAcl
      if ( aclVrfName in serviceAclConfig and
           'bgpsacl' in serviceAclConfig[ aclVrfName ].service and
           serviceAclConfig[ aclVrfName ].service[ 'bgpsacl' ].aclName ):
         cmds.append( '%s access-group %s' % ( aclType,
                      serviceAclConfig[ aclVrfName ].service[ 'bgpsacl' ].aclName ) )
      elif saveAll:
         cmds.append( 'no %s access-group' % aclType )

   if bgpConfig.listenPort != bgpConfig.listenPortInvalid:
      cmds.append( 'bgp transport listen-port %d' % bgpConfig.listenPort )
   elif saveAll:
      cmds.append( 'no bgp transport listen-port' ) 

   if bgpConfig.sockMaxSegSizeIpv4 != bgpConfig.sockMaxSegSizeInvalid:
      cmds.append( 'bgp transport ipv4 mss %u' % bgpConfig.sockMaxSegSizeIpv4 )
   elif saveAll:
      cmds.append( 'no bgp transport ipv4 mss' )

   if bgpConfig.sockMaxSegSizeIpv6 != bgpConfig.sockMaxSegSizeInvalid:
      cmds.append( 'bgp transport ipv6 mss %u' % bgpConfig.sockMaxSegSizeIpv6 )
   elif saveAll:
      cmds.append( 'no bgp transport ipv6 mss' )

   if bgpConfig.pathMtuDiscoverySet:
      if bgpConfig.pathMtuDiscovery == 'isTrue':
         cmds.append( 'bgp transport pmtud' )
      else:
         cmds.append( 'bgp transport pmtud disabled' )
   elif saveAll:
      cmds.append( 'no bgp transport pmtud' )

   if not vrfName and bgpConfig.dscp != bgpConfig.dscpDefault:
      cmds.append( 'bgp transport qos dscp %d' % bgpConfig.dscp )
   elif saveAll and not vrfName:
      cmds.append( 'bgp transport qos dscp %d' % bgpConfig.dscp )

   if bgpConfig.defaultMetricPresent != False:
      cmds.append( 'default-metric %d' % bgpConfig.defaultMetric )
   elif saveAll:
      cmds.append( 'no default-metric' )
 
   isCmpMedSet, cmpMedVal = evaluateTristate( bgpConfig, defaultVrfBgpConfig, 
                                              'alwaysCompareMed' )
   if isCmpMedSet or saveAll:
      cmds.append( ( '' if cmpMedVal else 'no ' ) + 'bgp always-compare-med' )

   isMedWrstSet, medWrstVal = evaluateTristate( bgpConfig, defaultVrfBgpConfig, 
                                                'medMissingAsWorst' )
   if isMedWrstSet or saveAll:
      cmds.append( ( '' if medWrstVal else 'no ' ) + \
                          'bgp bestpath med missing-as-worst' )

   isMedCfdSet, medCfdVal = evaluateTristate( bgpConfig, defaultVrfBgpConfig, 
                                              'medConfed' )
   isMedCfdMissWrstSet, medCfdMissWrstVal = evaluateTristate( bgpConfig, 
                                                         defaultVrfBgpConfig, 
                                                         'medConfedMissingAsWorst' )
   if isMedCfdSet or isMedCfdMissWrstSet or saveAll:
      cmds.append( ( '' if medCfdVal or medCfdMissWrstVal else 'no ') + \
                          'bgp bestpath med confed' + \
                          ( ' missing-as-worst' if isMedCfdMissWrstSet else '') )

   if bgpConfig.reflectedRouteAttr != bgpConfig.reflectedRouteAttrDefault:
      cmd = 'bgp route-reflector preserve-attributes'
      if bgpConfig.reflectedRouteAttr == reflectedRouteAttrStateEnum.alwaysPreserved:
         cmd += ' always'
      cmds.append( cmd )
   elif saveAll:
      cmds.append( 'no bgp route-reflector preserve-attributes' )

   isMpSet, mpVal = evaluateValue( bgpConfig, defaultVrfBgpConfig, 'maxPaths' )
   isEcmpSet, ecmpVal = evaluateValue( bgpConfig, defaultVrfBgpConfig, 'maxEcmp',
                                       routingHwStatus.maxEcmp )
   if isMpSet or isEcmpSet or saveAll:
      cmd = 'maximum-paths %d' % mpVal
      if isEcmpSet or saveAll:
         if isEcmpSet or (not isMpSet):
            ecmpVal = ecmpVal if ecmpVal != getattr( defaultVrfBgpConfig,
               'maxEcmpInvalid' ) else routingHwStatus.maxEcmp
         else:
            # Default VRF -> ecmp not configured, set it to hwMax.
            # Non-Default VRF -> ecmp and also max-path not configured, set ecmp to
            # hwMax.
            ecmpVal = routingHwStatus.maxEcmp

         if ecmpVal:
            cmd += ' ecmp %s' % ecmpVal
      cmds.append( cmd )

   # Print additional-paths command in RouterBgpMode only if there's
   # no AF mode under bgp; this command is for IPv4 Unicast AF.
   if not hasAfConfig( bgpConfig ):
      if bgpConfig.picIPv4Uni:
         if not bgpConfig.picEcmpPrimaryIPv4Uni:
            cmds.append( "bgp additional-paths install" )
         else:
            cmds.append( "bgp additional-paths install ecmp-primary" )
      elif saveAll:
         cmds.append( "no bgp additional-paths install" )

   # Add-path receive command
   isAPRecvSet, apRecvVal = evaluateTristate( bgpConfig, defaultVrfBgpConfig,
                                              'apRecv' )
   if isAPRecvSet:
      if apRecvVal:
         cmds.append( 'bgp additional-paths receive' )
      else:
         cmds.append( 'no bgp additional-paths receive' )
   elif saveAll:
      # Add-path receive is enabled by default
      cmds.append( 'bgp additional-paths receive' )

   # Add-path send command
   cmd = ''
   for app in [ 'appAny' ]:
      appConfig = bgpConfig.apSend.get( app )
      if appConfig is not None:
         if appConfig.enable:
            cmd = 'bgp '
         else:
            cmd = 'no bgp '
         cmd += addPathSendCommon( app, appConfig=appConfig )
         break
   if saveAll and cmd == '':
      cmd = 'no bgp '
      cmd += addPathSendCommon( 'appAny', saveAll=saveAll )
   if cmd != '':
      cmds.append( cmd )

   isLlSet, val = evaluateValue( bgpConfig, defaultVrfBgpConfig, 'listenLimit' )
   if isLlSet or saveAll:
      cmd = 'dynamic peer max %d' % val
      cmds.append( cmd )

   listenRange = set( bgpConfig.listenRange )
   for peergroupName in sorted( list( listenRange ) ):
      peergroup = bgpConfig.listenRange.get( peergroupName )
      # TODO .stringValue should not be necessary below, BUG25925
      for prefix in peergroup.prefixList:
         peerSpec = peergroup.prefixList[prefix]
         if peerSpec.hasAsn:
            spec = 'remote-as %s' % bgpFormatAsn( peerSpec.asn, asdotConfigured )
         else:
            spec = 'peer-filter %s' % peerSpec.peerFilter
         peerId = ''
         if peerSpec.includeRouterId:
            peerId = ' peer-id include router-id'
         cmd = 'bgp listen range %s%s peer-group %s %s' % \
               ( prefix.stringValue,
                 peerId,
                 peergroup.peergroupName, 
                 spec )
         cmds.append( cmd )

   isSet, val = evaluateTristate( bgpConfig, defaultVrfBgpConfig,
                                  'asPathMultiPathRelax' )
   _, matchVal = evaluateValue( bgpConfig, defaultVrfBgpConfig,
                                         'asPathMultiPathRelaxMatch' )
   if isSet or matchVal != bgpConfig.asPathMultiPathRelaxMatchDefault or saveAll:
      if matchVal != bgpConfig.asPathMultiPathRelaxMatchInvalid and \
         matchVal != bgpConfig.asPathMultiPathRelaxMatchDefault:
         cmd = ( 'bgp bestpath as-path multipath-relax match %d' %
                 bgpConfig.asPathMultiPathRelaxMatch )
      else:
         cmd = ( ( '' if val else 'no ' ) +
                 'bgp bestpath as-path multipath-relax' )
      cmds.append( cmd )

   isSet, val = evaluateTristate( bgpConfig, defaultVrfBgpConfig, 'ignoreAsPathLen' )
   
   if isSet or saveAll:
      cmds.append( ( '' if val else 'no ' ) + 'bgp bestpath as-path ignore' )

   # bestpath d-path configuration applies only to default-vrf.
   if not vrfName:
      val = defaultVrfBgpConfig.includeDomainPathLen == 'isTrue'
      if val or saveAll:
         cmds.append( ( '' if val else 'no ' ) + 'bgp bestpath d-path' )

   isSet, val = evaluateTristate( bgpConfig, defaultVrfBgpConfig,
         'nexthopResolutionNhgTunnelEcmpDisabled' )

   if isSet or saveAll:
      cmds.append( ( '' if val else 'no ' ) +
         'bgp bestpath next-hop resolution tunnel nexthop-group ecmp disabled' )

   isSet, val = evaluateTristate( bgpConfig, defaultVrfBgpConfig,
         'skipNextHopIgpCost' )
   
   if isSet or saveAll:
      cmds.append( ( '' if val else 'no ' ) +
                   'bgp bestpath skip next-hop igp-cost' )

   isSet, val = evaluateTristate( bgpConfig, defaultVrfBgpConfig,
         'skipAigpMetric' )

   if RoutingLibToggleLib.toggleAccumulatedIgpMetricEnabled() and \
      ( isSet or saveAll ):
      cmds.append( ( '' if val else 'no ' ) +
                   'bgp bestpath skip aigp-metric' )

   if IpRibLibToggleLib.toggleMixedEcmpEnabled():
      isSet, val = evaluateTristate( bgpConfig, defaultVrfBgpConfig,
                                     'compatibleEBgpIBgp' )

      if isSet or saveAll:
         cmds.append( ( '' if val else 'no ' ) +
                      'bgp bestpath skip peer type ebgp ibgp' )

   isSet, val = evaluateTristate( bgpConfig, defaultVrfBgpConfig, 'asPathCmpIncNh' )
   if isSet or saveAll:
      cmds.append( ( '' if val else 'no ' ) + 'bgp aspath-cmp-include-nexthop' )

   isSet, val = evaluateTristate( bgpConfig, defaultVrfBgpConfig, 'ecmpFast' )
   if isSet and not val:
      cmds.append( 'no bgp bestpath ecmp-fast' )
   elif saveAll:
      cmds.append( 'bgp bestpath ecmp-fast' )

   isSet, val = evaluateTristate( bgpConfig, defaultVrfBgpConfig, 'ecmpTbOnAge' )
   if isSet or saveAll:
      cmds.append( ( '' if val else 'no ' ) + 'bgp bestpath tie-break age' )

   isSet, val = evaluateTristate( bgpConfig, defaultVrfBgpConfig,
                                  'ecmpTbOnRouterId' )
   if isSet or saveAll:
      cmds.append( ( '' if val else 'no ' ) + \
                          'bgp bestpath tie-break router-id' )

   isSet, val = evaluateTristate( bgpConfig, defaultVrfBgpConfig,
                                  'ecmpTbOnOrigId' )
   if isSet or saveAll:
      cmds.append( ( '' if val else 'no ' ) + \
                          'bgp bestpath tie-break originator-id' )

   isSet, val = evaluateTristate( bgpConfig, defaultVrfBgpConfig,
                                  'ecmpTbOnClusterListLen' )
   if isSet or saveAll:
      cmds.append( ( '' if val else 'no ' ) + \
                          'bgp bestpath tie-break cluster-list-length' )

   isSet, val = evaluateTristate( bgpConfig, defaultVrfBgpConfig,
                                  'advertiseInactive' )
   if isSet or saveAll:
      cmds.append( ( '' if val else 'no ' ) + 'bgp advertise-inactive' )

   isSet, val = evaluateTristate( bgpConfig, defaultVrfBgpConfig,
                                  'autoLocalAddr' )
   if isSet or saveAll:
      cmds.append( ( '' if val else 'no ' ) + 'bgp auto-local-addr' )

   isSet, val = evaluateTristate( bgpConfig, defaultVrfBgpConfig,
                                  'nexthopUnchanged' )
   if isSet or saveAll:
      cmds.append( ( '' if val else 'no ' ) + 'bgp next-hop-unchanged' )

   val = bgpConfig.aggregateCommInheritLoose
   if val or saveAll:
      cmds.append( ( '' if val else 'no ' ) +
                   'bgp aggregate-route community inheritance loose' )

   isSet, val = evaluateTristate( bgpConfig, defaultVrfBgpConfig,
                                  'includeOriginAsValidity' )
   if isSet:
      cmds.append( 'bgp bestpath origin-as validity' +
                   ( '' if val else ' disabled' ) )
   elif saveAll:
      cmds.append( 'no bgp bestpath origin-as validity' )


   for hook in bgpMonitoringHook.extensions():
      cmd = hook( bgpConfig, saveAll )
      if cmd:
         cmds.append( cmd )

   neighborDefaultSendCommunityCmd = 'neighbor default send-community'
   if bgpConfig.sendCommunityPresent:
      if ( bgpConfig.sendCommunity or
           bgpConfig.sendStandardCommunity or
           bgpConfig.sendExtendedCommunity or
           bgpConfig.sendLargeCommunity ):
         if bgpConfig.sendStandardCommunity:
            neighborDefaultSendCommunityCmd += ' standard'
         if bgpConfig.sendExtendedCommunity:
            neighborDefaultSendCommunityCmd += ' extended'
         if bgpConfig.sendLargeCommunity:
            neighborDefaultSendCommunityCmd += ' large'
         cmds.append( neighborDefaultSendCommunityCmd )
      else:
         cmds.append( getExplicitNoCmd( neighborDefaultSendCommunityCmd,
                                        getattr( defaultVrfBgpConfig,
                                                 'noEqualsDefaultMode' ) ) )

   def peerConfigSave( peerKey ):
      nedMode = getattr( defaultVrfBgpConfig, 'noEqualsDefaultMode' )
      v6Addr = peerKey.type == 'peerIpv6'
      v4Addr = peerKey.type == 'peerIpv4'

      addrOrPgName = peerKey.stringValue

      peer = bgpConfig.neighborConfig.get( peerKey )
      if not peer and saveAll:
         # Display default peer config for saveAll.
         # Using defaultPeerConfig to get the default values for the peer.
         # Assigning the peer with default config changes the entity 
         # name (addrOrPgName). 
         # Therefore, 'addrOrPgName' should be used to access neighbor ip.
         # (XXX.Shall we create a dummy peer instance instead?)
         peer = bgpConfig.defaultPeerConfig
      peerGroup = peer.isPeerGroup
      peerGroupPeer = peer.isPeerGroupPeer
      if peerGroup:
         cmds.append( 'neighbor %s peer group' % addrOrPgName )

      def addSimpleSaveCommand( attrStr, cmd, disabledKeywordInNedMode,
                                hasGlobalAttr=False, valuePrint=None, trail=None ):
         attr = getattr( peer, attrStr )
         attrDefault = getattr( peer, attrStr + 'Default' )
         attrPresent = getattr( peer, attrStr + 'Present' )

         saveCmd = 'neighbor %s %s' % ( addrOrPgName, cmd )
         if attrPresent:
            if ( attr == attrDefault ) and ( peerGroupPeer or hasGlobalAttr ):
               if disabledKeywordInNedMode:
                  cmds.append( getExplicitNoCmd( saveCmd, nedMode ) )
               else:
                  cmds.append( getExplicitNoCmd( saveCmd, nedMode, attrDefault ) )
               return
            if attrStr == 'removePrivateAs':
               if attr.always:
                  saveCmd += ' all'
               if attr.replaceAs:
                  saveCmd += ' replace-as'
            elif attrStr == 'neighborShutdown':
               shutdownMsgAttr = peer.neighborShutdownMsg
               shutdownMsgAttrDefault = peer.neighborShutdownMsgDefault
               if shutdownMsgAttr != shutdownMsgAttrDefault:
                  saveCmd += ' reason %s' % shutdownMsgAttr
            elif attrStr in [ 'idleHoldTime', 'bgpTtlSecMaxHop' ]:
               saveCmd = '%s %d' % ( saveCmd, attr )
            elif valuePrint:
               saveCmd += valuePrint( peer, attr )
            elif not isinstance( attr, bool ):
               if cmd == 'remote-as':
                  saveCmd = saveCmd + ' ' + bgpFormatAsn( attr, asdotConfigured )
               else:
                  saveCmd = saveCmd + ' ' + str( attr )
            if trail:
               saveCmd = saveCmd + ' ' + trail
            cmds.append( saveCmd )
         elif saveAll and not peerGroupPeer:
            if disabledKeywordInNedMode:
               cmd = getExplicitNoCmd( saveCmd, nedMode )
            else:
               cmd = getExplicitNoCmd( saveCmd, nedMode, attrDefault )
            cmds.append( cmd )

      def addSimpleBoolSaveCommand( attrStr, baseCmd, positiveCmdDefault=False ):
         attr = getattr( peer, attrStr )
         attrPresent = getattr( peer, attrStr + 'Present' )

         saveCmd = 'neighbor %s %s' % ( addrOrPgName, baseCmd )
         if attrPresent:
            if attr:
               if attrStr == 'defaultOriginate':
                  if peer.defaultOriginateRouteMap:
                     saveCmd += " route-map %s" % peer.defaultOriginateRouteMap
                  if peer.defaultOriginateAlways:
                     saveCmd += " always"

            else:
               saveCmd = getExplicitNoCmd( saveCmd, nedMode )
            cmds.append( saveCmd )
         elif saveAll and not peerGroupPeer:
            if positiveCmdDefault:
               cmds.append( saveCmd )
            else:
               cmds.append( getExplicitNoCmd( saveCmd, nedMode ) )

      # Used for route-map and prefix-list in|out commands
      def addSimpleInOutSaveCommand( attrStr, baseCmd, inOrOut ):
         assert inOrOut in [ 'in', 'out' ]
         attr = getattr( peer, attrStr )
         attrPresent = getattr( peer, attrStr + 'Present' )

         saveCmd = 'neighbor %s %s' % ( addrOrPgName, baseCmd )
         if attrPresent:
            if attr:
               cmds.append( '%s %s %s' % ( saveCmd, attr, inOrOut ) )
            elif peer.isPeerGroupPeer:
               cmds.append( getExplicitNoCmd( '%s %s' % ( saveCmd, inOrOut ),
                                              nedMode ) )
         elif saveAll and not peerGroupPeer:
            cmds.append( getExplicitNoCmd( '%s %s' % ( saveCmd, inOrOut ),
                                           nedMode ) )


      # COMPATIBILITY WARNING
      # Please don't reorder the cmds.append, unless you have a *really* good
      # reason and are ready to break backward-compatibility.
      #
      # Add new attributes at the very end.
      #
      # See BUG198992.

      if peerGroupPeer:
         pgName = peer.peerGroupKey.group
         cmds.append( 'neighbor %s peer group %s' % ( addrOrPgName,
                                                      pgName ) )
      elif saveAll and not peerGroup and not v6Addr:
         cmds.append( 'no neighbor %s peer group' % addrOrPgName )

      addSimpleSaveCommand( 'asNumber', 'remote-as', True )
      addSimpleSaveCommand( 'importLocalPref', 'import-localpref', False )
      addSimpleSaveCommand( 'exportLocalPref', 'export-localpref', False )
      addSimpleSaveCommand( 'nextHopSelf', 'next-hop-self', True )
      addSimpleSaveCommand( 'nextHopPeer', 'next-hop-peer', True )
      addSimpleSaveCommand( 'nexthopUnchanged', 'next-hop-unchanged', True,
                            hasGlobalAttr=True )
      addSimpleSaveCommand( 'neighborShutdown', 'shutdown', True )
      addSimpleSaveCommand( 'removePrivateAs', 'remove-private-as', True )
      addSimpleSaveCommand( 'linkbwDelay', 'link-bandwidth update-delay', False )
      addSimpleSaveCommand( 'prependOwnDisabled', 'as-path prepend-own disabled',
                            True )
      addSimpleSaveCommand( 'replaceRemoteAs', 
                            'as-path remote-as replace out', True )

      if peer.pathMtuDiscoveryPresent:
         if peer.pathMtuDiscovery:
            addSimpleSaveCommand( 'pathMtuDiscovery', 'transport pmtud', True )
         else:
            addSimpleSaveCommand( 'pathMtuDiscovery', 'transport pmtud disabled',
                                  True )

      def localAsValuePrint( peer, localAs ):
         cmd = ' %s no-prepend replace-as' % \
            bgpFormatAsn( localAs.asn, asdotConfigured )
         if localAs.fallback:
            cmd += " fallback"
         return cmd

      addSimpleSaveCommand( 'localAs', 'local-as', True,
                            valuePrint=localAsValuePrint )

      weightCmd = "neighbor %s weight" % addrOrPgName
      if peer.weightPresent:
         if peer.weight == peer.weightDefault and peerGroupPeer:
            cmds.append( getExplicitNoCmd( weightCmd, nedMode,
                                           defaultValue=peer.weightDefault ) )
         else:
            cmds.append( '%s %s' % ( weightCmd, peer.weight ) )
      elif saveAll and not peerGroupPeer:
         # Per-neighbor weight cannot be configured both globally and per address
         # family. If it is not configured in any address family, we should render
         # the default neighbor weight cmd for save-all at the global level.
         if not weightConfiguredInAf( peer ):
            cmds.append( getExplicitNoCmd( weightCmd, nedMode,
                                           defaultValue=peer.weightDefault ) )

      outDelayCmd = "neighbor %s out-delay" % addrOrPgName
      if peer.outDelayPresent:
         if peer.outDelay == peer.outDelayDefault and peerGroupPeer:
            odCmd = getExplicitNoCmd( outDelayCmd, nedMode, 
                                      defaultValue=peer.outDelayDefault )
         else:
            odCmd = '%s %s' % ( outDelayCmd, peer.outDelay )
         if peer.outDelayApply == outDelayApplyEnum.outDelayApplyChanges:
            odCmd += ' changes'
         elif peer.outDelayApply == \
              outDelayApplyEnum.outDelayApplyChangesWithdrawals:
            odCmd += ' changes withdrawals'
         cmds.append( odCmd )

      elif saveAll and not peerGroupPeer:
         # Per-neighbor out-delay cannot be configured both globally and per address
         # family. If it is not configured in any address family, we should render
         # the default neighbor out-delay cmd for save-all at the global level.
         if not outDelayConfiguredInAf( peer ):
            cmds.append( getExplicitNoCmd( outDelayCmd, nedMode,
                                           defaultValue=peer.outDelayDefault ) )

      addSimpleSaveCommand( 'passive', 'passive', True )
      addSimpleSaveCommand( 'remotePort', 'transport remote-port', False )
      addSimpleSaveCommand( 'updateSrcIntf', 'update-source', True )
      addSimpleSaveCommand( 'dontCapabilityNegotiate', 'dont-capability-negotiate',
                            True )

      # Only use the deprecated syntax of the neighbor bfd command if the c-bit
      # option of the command is not being used, and the deprecated syntax for this
      # command was used.
      bfdCmd = "bfd"
      if ( getattr( peer, 'bfdCBitEnabled' ) ):
         bfdCmd += " c-bit"
      if BgpCommonToggleLib.toggleArBgpPeerFlapDampingEnabled():
         if ( getattr( peer, 'bfdDampingEnabled' ) ):
            bfdCmd += " damping"
         if ( getattr( peer, 'bfdDampingTimePresent' ) ):
            bfdCmd += " seconds %d" % getattr( peer, 'bfdDampingTime' )
         if ( getattr( peer, 'bfdDampingMultiplierPresent' ) ):
            bfdCmd += " max-multiplier %s" % getattr( peer, 'bfdDampingMultiplier' )

      addSimpleSaveCommand( 'bfdEnabledState',
                            bfdCmd,
                            True )

      if not v4Addr:
         addSimpleSaveCommand( 'localIp4Addr', 'local-v4-addr', True )
      if not v6Addr:
         addSimpleSaveCommand( 'localIp6Addr', 'local-v6-addr', True )
      addSimpleSaveCommand( 'autoLocalAddr', 'auto-local-addr', True )
         
      for hook in neighborMonitoringHook.extensions():
         cmd = hook( addrOrPgName, peer, saveAll )
         if cmd:
            cmds.append( cmd )

      if not v6Addr:
         addSimpleSaveCommand( 'remoteIp6Addr', 'next-hop-v6-addr', True,
                               trail='in' )

      descriptionCmd = 'neighbor %s description' % addrOrPgName
      if peer.descriptionPresent:
         if peer.description != peer.descriptionDefault:
            cmds.append( '%s %s' % ( descriptionCmd, peer.description ) )
         elif peerGroupPeer:
            cmds.append( 'no %s' % descriptionCmd )
      elif saveAll and not peerGroupPeer:
         cmds.append( 'no %s' % descriptionCmd )

      addSimpleSaveCommand( 'allowAs', 'allowas-in', True, True )

      # softreconfig inbound is default true
      sciCmd = 'neighbor %s rib-in pre-policy retain' % addrOrPgName
      if peer.softReconfigInboundPresent:
         if peer.softReconfigInbound != peer.softReconfigInboundDefault:
            if peer.softReconfigInbound == SoftReconfigInboundStateEnum.sciAll:
               cmds.append( '%s all' % sciCmd )
            else:
               cmds.append( getExplicitNoCmd( sciCmd, nedMode ) )
         elif peerGroupPeer:
            cmds.append( sciCmd )
      elif saveAll and not peerGroupPeer:
         cmds.append( sciCmd )

      # ebgpMultiHop may or may not print out a ttl
      ebgpMultiHopCmd = 'neighbor %s ebgp-multihop' % addrOrPgName
      if peer.ebgpMultiHopPresent:
         if ( peer.ebgpMultiHop == peer.ebgpMultiHopDefault ) and peerGroupPeer:
            ebgpMultiHopCmd = getExplicitNoCmd( ebgpMultiHopCmd, nedMode )
         elif peer.ebgpMultiHop != peer.ebgpMultiHopEnabledTtlDefault:
            ebgpMultiHopCmd = '%s %d' % ( ebgpMultiHopCmd, peer.ebgpMultiHop )
         cmds.append( ebgpMultiHopCmd )
      elif saveAll and not peerGroupPeer:
         cmds.append( getExplicitNoCmd( ebgpMultiHopCmd, nedMode ) ) 

      bgpTtlSecMaxHopCmdTrail = None
      if peer.bgpTtlSecMaxHopLog != peer.bgpTtlSecMaxHopLogDefault:
         bgpTtlSecMaxHopCmdTrail = 'log'
      addSimpleSaveCommand( 'bgpTtlSecMaxHop', 'ttl maximum-hops', False,
                            trail=bgpTtlSecMaxHopCmdTrail )

      # route reflector has another option meshed
      rrClientCmd = 'neighbor %s route-reflector-client' % addrOrPgName
      rrClientEnum = Tac.Type( "Routing::Bgp::ReflectorClient" )
      if peer.rrClientPresent:
         if peer.rrClient == rrClientEnum.reflectorMeshed:
            rrClientCmd = '%s meshed' % rrClientCmd
         if ( peer.rrClient == peer.rrClientDefault ) and peerGroupPeer:
            rrClientCmd = getExplicitNoCmd( rrClientCmd, nedMode )
         cmds.append( rrClientCmd )
      elif saveAll and not peerGroupPeer:
         cmds.append( getExplicitNoCmd( rrClientCmd, nedMode ) )

      timersCmd = 'neighbor %s timers' % addrOrPgName
      if peer.timersPresent:
         cmds.append( '%s %d %d' % ( timersCmd, peer.keepaliveTime, peer.holdTime ) )
      elif saveAll and not peerGroupPeer:
         timersDefaultValue = '%d %d' % ( peer.keepaliveTimeDefault,
                                          peer.holdTimeDefault )
         cmds.append( getExplicitNoCmd( timersCmd, nedMode,
                      defaultValue=timersDefaultValue ) )

      addSimpleInOutSaveCommand( 'routeMapIn', 'route-map', 'in' )

      gracefulRestartCmd = "neighbor %s graceful-restart" % addrOrPgName
      if peer.gracefulRestartPresent:
         if peer.gracefulRestart:
            cmds.append( gracefulRestartCmd )
         else:
            cmds.append( getExplicitNoCmd( gracefulRestartCmd, nedMode ) )
      elif saveAll and not peerGroupPeer:
         # Graceful restart cannot be configured both globally and per address
         # family. If it is not configured in any address family, we should render
         # the default graceful restart cmd for save-all at the global level.
         _, anyAfConfigured = isGracefulRestartGlobalOrAfConfigured( peer )
         if not anyAfConfigured:
            cmds.append( getExplicitNoCmd( gracefulRestartCmd, nedMode ) )

      grHelperCmd = 'neighbor %s graceful-restart-helper' % addrOrPgName
      # grHelperPresent is False in case of default cmd
      if peer.grHelperPresent:
         # peer.grHelper is False in case of 'no' cmd
         if peer.grHelper:
            if peer.grHelperRestartTime:
               grHelperCmd += ( ' restart-time %d' % peer.grHelperRestartTime )
            if peer.llgrHelper:
               grHelperCmd += ' long-lived'
            cmds.append( grHelperCmd )
         else:
            cmds.append( getExplicitNoCmd( grHelperCmd, nedMode ) )
      elif saveAll and not peerGroupPeer:
         cmds.append( grHelperCmd )
      addSimpleBoolSaveCommand( 'apRecv', 'additional-paths receive', True )
      
      # Add-path send command
      cmd = ''
      for app in [ 'appAny' ]:
         appConfig = peer.apSend.get( app )
         if appConfig is not None:
            if appConfig.enable:
               cmd = 'neighbor %s %s' % ( addrOrPgName,
                        addPathSendCommon( app, appConfig=appConfig ) )
            else:
               cmd = getExplicitNoCmd( 'neighbor %s %s' % ( addrOrPgName,
                        addPathSendCommon( app, appConfig=appConfig ) ), nedMode )
            break
      if saveAll and not peerGroupPeer and cmd == '':
         cmd = getExplicitNoCmd( 'neighbor %s %s' % ( addrOrPgName,
                  addPathSendCommon( 'appAny', saveAll=saveAll ) ), nedMode )
      if cmd != '':
         cmds.append( cmd )

      addSimpleBoolSaveCommand( 'routeToPeer', 'route-to-peer', True )

      addSimpleInOutSaveCommand( 'routeMapOut', 'route-map', 'out' )
      
      passwordCmd = 'neighbor %s password' % addrOrPgName
      if peer.passwordPresent:
         if peer.password != peer.passwordDefault:
            key = None if securityConfig.commonKeyEnabled else \
                  addrOrPgName + '_passwd'
            encryptedPasswd = encodeKey( peer.password, key=key, 
                                         algorithm='MD5' )
            cmds.append( '%s 7 %s' \
                  % ( passwordCmd,
                      CliSave.sanitizedOutput( options, encryptedPasswd ) ) )
         elif peer.isPeerGroupPeer:
            cmds.append( 'no ' + passwordCmd )
      elif saveAll and not peerGroupPeer:
         cmds.append( 'no ' + passwordCmd )

      addSimpleBoolSaveCommand( 'defaultOriginate', 'default-originate' )
      addSimpleBoolSaveCommand( 'enforceFirstAs', 'enforce-first-as', True )

      metricOutCmd = 'neighbor %s metric-out' % addrOrPgName
      if ( peer.metricOutState != MetricOutStateEnum.metricNotConfigured ):
         if ( peer.metricOutState == MetricOutStateEnum.metricOutSet ):
            cmds.append( '%s %d' % ( metricOutCmd, peer.metricOut ) )
         else:
            cmds.append( getExplicitNoCmd( metricOutCmd, nedMode ) ) 
      elif saveAll and not peerGroupPeer:
         cmds.append( getExplicitNoCmd( metricOutCmd, nedMode ) )

      addSimpleSaveCommand( 'idleHoldTime', 'idle-restart-timer', True )

      sendCommunityCmd = 'neighbor %s send-community' % addrOrPgName
      if peer.sendCommunityPresent:
         lbwPresenceValid = \
               peer.sendCommunityLinkBwPresent and \
               ( peer.sendExtendedCommunity or \
                not ( peer.sendLargeCommunity or peer.sendStandardCommunity ) )
         if peer.sendCommunity or \
               peer.sendStandardCommunity or \
               peer.sendExtendedCommunity or \
               peer.sendLargeCommunity:
            if peer.sendStandardCommunity:
               sendCommunityCmd += ' standard'
            if peer.sendExtendedCommunity:
               sendCommunityCmd += ' extended'
            if peer.sendLargeCommunity:
               sendCommunityCmd += ' large'
            if lbwPresenceValid:
               sendCommunityCmd += ' link-bandwidth'
               if peer.sendCommunityLinkBw == \
                      SendCommunityLinkBandwidthStateEnum.\
                      sendCommunityLinkBandwidthAdditive:
                  sendCommunityCmd += ' aggregate'
               elif peer.sendCommunityLinkBw == \
                      SendCommunityLinkBandwidthStateEnum.\
                      sendCommunityLinkBandwidthAdditiveRef:
                  sendCommunityCmd += ' aggregate %s' \
                      % peer.sendCommunityLinkBwAdditiveRefDisplay
               elif peer.sendCommunityLinkBw == \
                      SendCommunityLinkBandwidthStateEnum.\
                      sendCommunityLinkBandwidthDivideEqual:
                  sendCommunityCmd += ' divide equal'
               elif peer.sendCommunityLinkBw == \
                      SendCommunityLinkBandwidthStateEnum.\
                      sendCommunityLinkBandwidthDivideRatio:
                  sendCommunityCmd += ' divide ratio'
               else:
                  raise ValueError()
            cmds.append( sendCommunityCmd )
         else:
            cmds.append( getExplicitNoCmd( sendCommunityCmd, nedMode ) )
      elif saveAll and not peerGroupPeer:
         cmds.append( getExplicitNoCmd( sendCommunityCmd, nedMode ) ) 
      
      savePeerMaxRoutes( peer, cmds, addrOrPgName, saveAll, nedMode )
      savePeerMaxAdvRoutes( peer, cmds, addrOrPgName, saveAll, nedMode )

      # linkBw may have auto or default generation configured or not
      linkBwCmd = 'neighbor %s link-bandwidth' % addrOrPgName
      if peer.linkBwPresent:
         if peer.linkBw != \
                LinkBandwidthGenerationStateEnum.linkBandwidthNotConfigured:
            if peer.linkBw == \
                  LinkBandwidthGenerationStateEnum.linkBandwidthAutoGeneration:
               linkBwCmd += " auto"
               if peer.linkBwAutoPercentPresent:
                  linkBwCmd += " percent %s" % peer.linkBwAutoPercent
            elif peer.linkBw == \
                   LinkBandwidthGenerationStateEnum.linkBandwidthDefaultGeneration:
               linkBwCmd += " default %s" \
                   % peer.linkBwDefDisplay
            cmds.append( linkBwCmd )
         else:
            cmds.append( getExplicitNoCmd( linkBwCmd, nedMode ) )
      elif saveAll and not peerGroupPeer:
         cmds.append( getExplicitNoCmd( linkBwCmd, nedMode ) ) 

      linkBwAdjustCmd = 'neighbor %s link-bandwidth adjust auto' % addrOrPgName
      if peer.linkBwAdjustPresent:
         if peer.linkBwAdjust:
            if peer.linkBwAdjustPercentPresent:
               linkBwAdjustCmd += ' percent %s' % peer.linkBwAdjustPercent
            cmds.append( linkBwAdjustCmd )
         else:
            cmds.append( getExplicitNoCmd( linkBwAdjustCmd, nedMode ) )
      elif saveAll and not peerGroupPeer:
         cmds.append( getExplicitNoCmd( linkBwAdjustCmd, nedMode ) )

      # rtRefreshCmd is disabled by default
      rtRefreshCmd = \
         'neighbor %s route refresh demarcated stale-path removal' % addrOrPgName
      if peer.enhRtRefreshInPresent:
         if peer.enhRtRefreshIn:
            if peer.enhRtRefreshStalePathTimePresent:
               rtRefreshCmd += " timeout %d" % peer.enhRtRefreshStalePathTime
            cmds.append( rtRefreshCmd )
         else:
            if peerGroupPeer:
               cmds.append( getExplicitNoCmd( rtRefreshCmd, nedMode ) )
      elif saveAll and not peerGroupPeer:
         cmds.append( getExplicitNoCmd( rtRefreshCmd, nedMode ) )

      for hook in neighborRpkiHook.extensions():
         cmd = hook( addrOrPgName, peer, saveAll )
         if cmd:
            cmds.append( cmd )

      # END of peerConfigSave()

   # Iterate through the ordered collection bgpConfig.neighborConfig
   # and call peerConfigSave on every element.
   for peerKey in bgpConfig.neighborConfig:
      peerConfigSave( peerKey )

   # Only print network statements in RouterBgpMode if
   # no address-family configuration exists.
   if not afCfgPresent:
      saveNetworkList( cmds, bgpConfig )

   # Print [no] bgp redistribute-internal in router bgp mode only
   # if it wasn't configured in address-family mode.
   if bgpConfig.bgpRedistInternal != redistInternalStateEnum.notConfigured:
      if bgpConfig.bgpRedistInternal != \
         bgpConfig.bgpRedistInternalDefault:
         cmds.append( 'no bgp redistribute-internal' )
      elif saveAll:
         cmds.append( 'bgp redistribute-internal' )

   def saveaggregates( lst, af ):
      for agg in lst:
         cmd = 'aggregate-address %s%s' % ( agg.address, '' )
         if agg.asSet:
            cmd += ' as-set'
         if agg.summaryOnly:
            cmd += ' summary-only'
         if agg.attrMap and agg.attrMap != agg.attrMapDefault:
            cmd += ' attribute-map %s' % agg.attrMap
         if agg.matchMap and agg.matchMap != agg.matchMapDefault:
            cmd += ' match-map %s' % agg.matchMap
         if agg.advertiseOnly:
            cmd += ' advertise-only'
         cmds.append( cmd )

   def saveAutoAggDomains( lst ):
      for domain in lst:
         cmd = 'auto-aggregation-domain %s' % domain.prefix
         if domain.asSet:
            cmd += ' as-set'
         if domain.attrMap and domain.attrMap != domain.attrMapDefault:
            cmd += ' aggregate-attributes %s' % domain.attrMap
         if domain.matchMap and domain.matchMap != domain.matchMapDefault:
            cmd += ' eligible-routes %s' % domain.matchMap
         if any( [ domain.groupByAsPath, domain.groupByLocPref, domain.groupByMed,
                   domain.groupByCommunity, domain.groupByExtCommunity ] ):
            cmd += ' group-by'
            if domain.groupByAsPath:
               cmd += ' as-path'
            if domain.groupByLocPref:
               cmd += ' local-preference'
            if domain.groupByMed:
               cmd += ' med'
            if domain.groupByCommunity:
               cmd += ' community'
            if domain.groupByExtCommunity:
               cmd += ' ext-community'
         cmds.append( cmd )

   for af in ( 'ipv4', 'ipv6' ):
      saveaggregates( filter( lambda a: a.address.af == af,
                           bgpConfig.aggregateList.values() ), af )

   saveAutoAggDomains( bgpConfig.autoAggDomainList.values() )

   for proto in [ 'protoDirect', 'protoIsis', 'protoOspf', 'protoOspfAse',
                  'protoOspfNssa', 'protoOspf3', 'protoOspf3Ase', 'protoOspf3Nssa',
                  'protoStatic', 'protoRip', 'protoBgpAggregate',
                  'protoAttachedHost', 'protoDynamic', 'protoBgp' ]:
      protocol = bgpConfig.redistributeConfig.get( proto )
      if protocol:
         command = redistributeCommand( protocol )
         if protocol.routeMap:
            command += ' route-map %s' % protocol.routeMap
         if protocol.rcf:
            command += ' rcf %s()' % protocol.rcf
      elif saveAll:
         command = 'no '
         command += redistributeCommand( proto, saveAll=saveAll )
      else:
         continue
      cmds.append( command )

   # BGP 'neighbor interface' config commands
   neighIntfConfig = {}
   for intf in bgpConfig.neighIntfConfig:
      peerConfig = bgpConfig.neighIntfConfig[ intf ]
      key = ( peerConfig.asnOrPeerFilter, peerConfig.peerGroup.stringValue )
      neighIntfConfig.setdefault( key, [ ] ).append( peerConfig.intfId )

   intfRangeCmds = []
   for asnOrPeerFilter, pg in neighIntfConfig:
      intfRanges = IntfRange.intfListToCanonical(
                             neighIntfConfig[ ( asnOrPeerFilter, pg ) ] )
      for intfRange in intfRanges:
         if asnOrPeerFilter.hasAsn:
            peerSpec = 'remote-as %s' % \
                       bgpFormatAsn( asnOrPeerFilter.asn, asdotConfigured )
         else:
            peerSpec = 'peer-filter %s' % asnOrPeerFilter.peerFilter
         cmd = 'neighbor interface %s peer-group %s %s' % \
               ( intfRange, pg, peerSpec )
         intfRangeCmds.append( cmd )
   # Since we have to sort based on interface names and different interface
   # types (ethernet, vlans etc) can be grouped together as they have the
   # same asn and peer-group, we have to first construct the individual config
   # and then sort the final list of commands.
   intfRangeCmds.sort()
   cmds += intfRangeCmds
   return cmds

def saveUcmpConfig( routingHwStatus, ucmpConfig, defaultVrfUcmpConfig,
                    options=None ):
   cmds = []
   isUcmpEnabledSet, ucmpEnabledVal = evaluateTristate( ucmpConfig,
                                                        defaultVrfUcmpConfig,
                                                        'ucmpEnabled' )
   isMaxUcmpSet, maxUcmpVal = evaluateValue( ucmpConfig, defaultVrfUcmpConfig,
                                             'maxUcmp',
                                             routingHwStatus.maxUcmp )
   isUcmpDeviationSet, ucmpDeviation = evaluateValue(
      ucmpConfig, defaultVrfUcmpConfig, 'ucmpDeviation',
      ucmpConfig.ucmpDeviationDefault )

   saveAll = options.saveAll if options else False
   if isUcmpEnabledSet or saveAll:
      if ucmpEnabledVal:
         cmd = 'ucmp mode 1'
         if isMaxUcmpSet or saveAll:
            cmd += ' %s' % ( maxUcmpVal )
            if isUcmpDeviationSet or saveAll:
               cmd += ' %s' % ( ucmpDeviation )
      else:
         cmd = 'no ucmp mode'
      cmds.append( cmd )

   isUcmpLinkbwDelaySet, ucmpLinkbwDelayVal = evaluateValue( ucmpConfig,
                                                             defaultVrfUcmpConfig,
                                                             'ucmpLinkbwDelay' )
   if isUcmpLinkbwDelaySet or saveAll:
      cmds.append( 'ucmp link-bandwidth update-delay %s' % ucmpLinkbwDelayVal )

   isUcmpWeightedSet, ucmpWeightedVal = evaluateTristate( ucmpConfig,
                                                          defaultVrfUcmpConfig,
                                                          'ucmpLinkbwWeighted' )
   if isUcmpWeightedSet or saveAll:
      cmds.append( ( '' if ucmpWeightedVal else 'no ' ) +
                          'ucmp link-bandwidth encoding-weighted' )

   isUcmpTriggerFecThresholdSet, ucmpTriggerFecThresholdVal = evaluateValue(
                                   ucmpConfig, defaultVrfUcmpConfig,
                                   'ucmpTriggerFecThreshold' )
   isUcmpClearFecThresholdSet, ucmpClearFecThresholdVal = evaluateValue(
                                   ucmpConfig, defaultVrfUcmpConfig,
                                   'ucmpClearFecThreshold' )

   isSet, val = evaluateTristate( ucmpConfig, defaultVrfUcmpConfig,
                                  'ucmpLinkbwRecursive' )

   if isSet or saveAll:
      cmds.append( ( '' if val else 'no ' ) + 'ucmp link-bandwidth recursive' )

   if saveAll and not isUcmpTriggerFecThresholdSet and not\
          isUcmpClearFecThresholdSet:
      cmds.append( 'no ucmp fec threshold' )
   elif saveAll or ( ucmpTriggerFecThresholdVal !=
                        ucmpConfig.ucmpTriggerFecThresholdDefault or\
                        ucmpClearFecThresholdVal != \
                        ucmpConfig.ucmpClearFecThresholdDefault ):
      cmds.append( 'ucmp fec threshold trigger %s clear %s warning-only' %
                       ( ucmpTriggerFecThresholdVal, ucmpClearFecThresholdVal ) )
   return cmds

# pkgdeps : library MgmtSecurity

@CliSave.saver( 'Routing::Bgp::Config', 'routing/bgp/config',
                requireMounts = ( 'routing/hardware/status', \
                      'routing/bgp/asn/config',
                      'acl/cpconfig/cli',
                      'mgmt/security/config', ) )
def saveConfig( bgpConfig, root, sysdbRoot, options,
                requireMounts ):
   if bgpConfig.asNumber == 0:
      return
   if options.intfFilter:
      return

   saveAll = options.saveAll if options else False
   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]
   routingHwStatus = requireMounts[ 'routing/hardware/status' ]
   securityConfig = requireMounts[ 'mgmt/security/config' ]
   aclCpConfig = requireMounts[ 'acl/cpconfig/cli' ]
   asdotConfigured = isAsdotConfigured( asnConfig )
   bgpMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
         bgpConfig.asNumber, asdotConfigured ) )
   cmds = saveBgpConfig( bgpConfig, bgpConfig, 
                         routingHwStatus, "", 
                         securityConfig, aclCpConfig, asdotConfigured,
                         options=options )
   for cmd in cmds:
      bgpMode[ 'Bgp.config' ].addCommand( cmd )

   # Add router-bgp-vrf-default mode commands now. This mode only has VPN commands
   # for now.
   vpnCmds = []
   if bgpConfig.routeDistinguisher != 'INVALID':
      rd = bgpConfig.routeDistinguisher
      vpnCmds.append( 'rd %s' % formatRd( rd, asdotConfigured ) )
   vpnCmds.extend( saveBgpVpnConfig( DEFAULT_VRF, bgpConfig, bgpConfig,
                                     asdotConfigured, saveAll ) )
   # Add this mode only if there is some config under it.
   if vpnCmds:
      bgpDefaultVrfMode = bgpMode[ RouterBgpVrfConfigMode
            ].getOrCreateModeInstance( DEFAULT_VRF )
      for cmd in vpnCmds:
         bgpDefaultVrfMode[ 'Bgp.vrf.config' ].addCommand( cmd )

   # Add config commands for router-bgp-vrf-default-af modes
   for addrFamily in BgpLib.bgpConfigAttrAfList( DEFAULT_VRF ):
      vpnCmds = saveBgpVpnAfConfig( bgpConfig, bgpConfig, addrFamily,
                                    asdotConfigured, saveAll )
      # Add AF VPN config only if there is some config under it
      if vpnCmds:
         # The default VRF mode may get created for the first time here if there was
         # no config outside address family mode
         bgpDefaultVrfMode = bgpMode[ RouterBgpVrfConfigMode
               ].getOrCreateModeInstance( DEFAULT_VRF )
         defaultVrfAfMode = bgpDefaultVrfMode[ RouterBgpVrfAfConfigMode
               ].getOrCreateModeInstance( ( addrFamily, DEFAULT_VRF ) )
         for cmd in vpnCmds:
            defaultVrfAfMode[ 'Bgp.vrf.afConfig.%s' % addrFamily ].addCommand( cmd )

@CliSave.saver( 'Routing::Bgp::VrfConfigDir', 'routing/bgp/vrf/config',
                requireMounts = ( 'routing/bgp/config', 'routing/hardware/status', \
                                  'routing/bgp/asn/config',
                                  'acl/cpconfig/cli',
                                  'mgmt/security/config',
                                  'routing/ucmp/bgp/vrf/config',
                                  'routing/ucmp/bgp/config' ) )
def saveVrfConfig( vrfConfigDir, root, sysdbRoot, options,
                   requireMounts ):
   bgpConfig = requireMounts[ 'routing/bgp/config' ]
   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]
   securityConfig = requireMounts[ 'mgmt/security/config' ]
   asdotConfigured = isAsdotConfigured( asnConfig ) 
   routingHwStatus = requireMounts[ 'routing/hardware/status' ]
   aclCpConfig = requireMounts[ 'acl/cpconfig/cli' ]
   if bgpConfig.asNumber == 0:
      return
   if options.intfFilter:
      return

   bgpMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
         bgpConfig.asNumber, asdotConfigured, ) )
   for vrfName in vrfConfigDir.vrfConfig:
      vrfBgpConfig = vrfConfigDir.vrfConfig[ vrfName ]
      bgpVrfMode = bgpMode[ RouterBgpVrfConfigMode
            ].getOrCreateModeInstance( vrfName )
      cmds = saveBgpConfig( vrfBgpConfig, bgpConfig,
                            routingHwStatus,
                            vrfName, securityConfig, aclCpConfig,
                            asdotConfigured, options=options )
      for cmd in cmds:
         bgpVrfMode[ 'Bgp.vrf.config' ].addCommand( cmd )

@CliSave.saver( 'RoutingLib::Ucmp::UcmpConfig', 'routing/ucmp/bgp/config',
                requireMounts = ( 'routing/hardware/status',
                                  'routing/bgp/config',
                                  'routing/bgp/asn/config',
                                  'mgmt/security/config', ) )
def saveUcmpBgpConfig( ucmpConfig, root, sysdbRoot, options,
                       requireMounts ):
   bgpConfig = requireMounts[ 'routing/bgp/config' ]
   if bgpConfig.asNumber == 0:
      return
   if options.intfFilter:
      return

   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]
   asdotConfigured = isAsdotConfigured( asnConfig )
   routingHwStatus = requireMounts[ 'routing/hardware/status' ]

   bgpMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
         bgpConfig.asNumber, asdotConfigured, ) )
   cmds = saveUcmpConfig( routingHwStatus, ucmpConfig, ucmpConfig, options )
   for cmd in cmds:
      bgpMode[ 'Bgp.config' ].addCommand( cmd )

@CliSave.saver( 'RoutingLib::Ucmp::VrfUcmpConfigDir',
                 'routing/ucmp/bgp/vrf/config',
                requireMounts = ( 'routing/hardware/status',
                                  'routing/bgp/config',
                                  'routing/bgp/vrf/config',
                                  'routing/bgp/asn/config',
                                  'routing/ucmp/bgp/config' ) )
def saveVrfUcmpConfig( vrfConfigDir, root, sysdbRoot, options,
                       requireMounts ):
   routingHwStatus = requireMounts[ 'routing/hardware/status' ]
   bgpConfig = requireMounts[ 'routing/bgp/config' ]
   if bgpConfig.asNumber == 0:
      return
   if options.intfFilter:
      return

   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]
   asdotConfigured = isAsdotConfigured( asnConfig )
   ucmpConfig = requireMounts[ 'routing/ucmp/bgp/config' ]

   bgpMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
         bgpConfig.asNumber, asdotConfigured, ) )
   for vrfName in vrfConfigDir.vrfConfig:
      vrfUcmpConfig = vrfConfigDir.vrfConfig[ vrfName ]
      bgpVrfMode = bgpMode[ RouterBgpVrfConfigMode
            ].getOrCreateModeInstance( vrfName )
      cmds = saveUcmpConfig( routingHwStatus, vrfUcmpConfig, ucmpConfig, options )
      for cmd in cmds:
         bgpVrfMode[ 'Bgp.vrf.config' ].addCommand( cmd )

@CliSave.saver( 'Tac::Dir', 'ip/vrf/routeDistinguisherInputDir/config',
                requireMounts=( 'routing/bgp/config', 'routing/bgp/asn/config', ) )
def saveVrfRdConfig( entity, root, sysdbRoot, options, requireMounts ):
   bgpConfig = requireMounts[ 'routing/bgp/config' ]
   if bgpConfig.asNumber == 0:
      return
   if options.intfFilter:
      return

   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]
   asdotConfigured = isAsdotConfigured( asnConfig )
   bgpMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
         bgpConfig.asNumber, asdotConfigured, ) )
   bgpRouteDistinguisherInput = entity.get( 'bgp' )
   if not bgpRouteDistinguisherInput:
      return
   for vrfName in sorted( bgpRouteDistinguisherInput.routeDistinguisher ):
      if vrfName == DEFAULT_VRF:
         # For default VRF, we populate RD (and other VPN) config when Bgp config
         # ("routing/bgp/config") is loaded
         continue
      rd = bgpRouteDistinguisherInput.routeDistinguisher[ vrfName ]
      assert rd != 'INVALID'
      bgpVrfMode = bgpMode[ RouterBgpVrfConfigMode
            ].getOrCreateModeInstance( vrfName )
      bgpVrfMode[ 'Bgp.vrf.config' ].addCommand(
            'rd %s' % formatRd( rd, asdotConfigured ) )

def getLsImportCmds( lsImportCfg ):
   cmds = []
   for protoConfig in sorted( lsImportCfg.protoConfig.values() ):
      proto = protoConfig.proto
      if proto == 'protoIsis':
         proto = 'isis'
      for instName in sorted( protoConfig.instanceCfg ):
         cmd = "%s instance %s" % ( proto, instName )
         instanceCfg = protoConfig.instanceCfg[ instName ]
         identifierPresent = instanceCfg.identifierPresent
         if identifierPresent:
            identifier = instanceCfg.identifier
            cmd += " identifier %lu" % identifier
         cmds.append( cmd )
   return cmds

@CliSave.saver( 'Routing::Bgp::LinkStateImportConfig', 'routing/bgp/lsImportConfig',
                requireMounts=( 'routing/bgp/config', 'routing/bgp/asn/config',
                                  'mgmt/security/config', ) )
def saveLsImportConfig( lsImportCfg, root, sysdbRoot, options,
                       requireMounts ):
   bgpConfig = requireMounts[ 'routing/bgp/config' ]
   if bgpConfig.asNumber == 0:
      return
   if not lsImportCfg.protoConfig:
      return

   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]
   asdotConfigured = isAsdotConfigured( asnConfig )
   bgpMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
         bgpConfig.asNumber, asdotConfigured, ) )
   bgpAfMode = bgpMode[ RouterBgpBaseAfConfigMode ].getOrCreateModeInstance(
         'link-state' )
   cmds = getLsImportCmds( lsImportCfg )
   for cmd in cmds:
      bgpAfMode[ 'Bgp.afConfig.link-state' ].addCommand( cmd )

#------------------------------------------------------------------------------------
# address-family mode
#------------------------------------------------------------------------------------
class RouterBgpBaseAfConfigMode( RoutingBgpBaseAfMode, CliSave.Mode ):
   def __init__( self, param, label=False, srTe=False ):
      RoutingBgpBaseAfMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

# Used to generate the address-family config level commands
RouterBgpBaseConfigMode.addChildMode( RouterBgpBaseAfConfigMode )
RouterBgpBaseAfConfigMode.addCommandSequence( 'Bgp.afConfig.ipv4' )
RouterBgpBaseAfConfigMode.addCommandSequence( 'Bgp.afConfig.ipv6' )
RouterBgpBaseAfConfigMode.addCommandSequence( 'Bgp.afConfig.ipv4 labeled-unicast' )
RouterBgpBaseAfConfigMode.addCommandSequence( 'Bgp.afConfig.ipv6 labeled-unicast' )
# XXX_FIX_BGP_CORE_CLI_EVPN_CONFIG: Move this to Evpn package eventually ?
RouterBgpBaseAfConfigMode.addCommandSequence( 'Bgp.afConfig.evpn' )
RouterBgpBaseAfConfigMode.addCommandSequence( 'Bgp.afConfig.ipv4 multicast' )
RouterBgpBaseAfConfigMode.addCommandSequence( 'Bgp.afConfig.ipv6 multicast' )
RouterBgpBaseAfConfigMode.addCommandSequence( 'Bgp.afConfig.ipv4 sr-te' )
RouterBgpBaseAfConfigMode.addCommandSequence( 'Bgp.afConfig.ipv6 sr-te' )
RouterBgpBaseAfConfigMode.addCommandSequence( 'Bgp.afConfig.link-state' )
RouterBgpBaseAfConfigMode.addCommandSequence( 'Bgp.afConfig.path-selection' )
RouterBgpBaseAfConfigMode.addCommandSequence( 'Bgp.afConfig.rt-membership' )

class RouterBgpVrfAfConfigMode( RoutingBgpVrfAfMode, CliSave.Mode ):
   def __init__( self, afAndVrfName ):
      af = afAndVrfName[ 0 ]
      vrfName = afAndVrfName[ 1 ] 
      RoutingBgpVrfAfMode.__init__( self, af, vrfName )
      CliSave.Mode.__init__( self, af )

RouterBgpVrfConfigMode.addChildMode( RouterBgpVrfAfConfigMode )
RouterBgpVrfAfConfigMode.addCommandSequence( 'Bgp.vrf.afConfig.ipv4' )
RouterBgpVrfAfConfigMode.addCommandSequence( 'Bgp.vrf.afConfig.ipv6' )
RouterBgpVrfAfConfigMode.addCommandSequence( 'Bgp.vrf.afConfig.ipv4 multicast' )
RouterBgpVrfAfConfigMode.addCommandSequence( 'Bgp.vrf.afConfig.ipv6 multicast' )

def savePeerAfActivation( peer, cmds, key, afConfig, addrFamily, saveAll, nedMode ):
   if afConfig == peerAfStateEnum.afActive:
      cmds.append( 'neighbor %s activate' % key )
   elif afConfig == peerAfStateEnum.afInactive:
      if nedMode:
         cmds.append( 'neighbor %s deactivate' % key )
      else:
         cmds.append( 'no neighbor %s activate' % key )
   elif saveAll:
      cmds.append( 'default neighbor %s activate' % key )
   
   # XXX_FIX_BGP_CORE_CLI_EVPN_CONFIG: Move this to Evpn package eventually ?
   # Commented the following code because of BUG149874
   # pylint: disable-msg=W0105
   # TODO: When this code block is un-commented, it needs to be edited to support
   # no-equals-default mode
   '''
   if addrFamily == 'evpn':
      if peer.afEvpnActivateMacIpRouteTypePresent:
         if peer.afEvpnActivateMacIpRouteType:
            cmds.append( 'neighbor %s activate layer-2' % key )
         else:
            cmds.append( 'no neighbor %s activate layer-2' % key )
      elif saveAll and not peer.isPeerGroupPeer:
         cmds.append( 'default neighbor %s activate layer-2' % key )
      
      if peer.afEvpnActivateIpPrefixRouteTypePresent:
         if peer.afEvpnActivateIpPrefixRouteType:
            cmds.append( 'neighbor %s activate layer-3' % key )
         else:
            cmds.append( 'no neighbor %s activate layer-3' % key )
      elif saveAll and not peer.isPeerGroupPeer:
         cmds.append( 'default neighbor %s activate layer-3' % key )
   '''

def savePeerPerAfRouteMap( peer, cmds, key, addrFamily, bgpConfig, saveAll,
                           nedMode ):

   for direction in [ "In", "Out" ]:
      attr = "routeMap%s" % direction
      attrName = BgpLib.peerConfigAttrsAfMap[ attr ].get( addrFamily )
      if not attrName:
         # sr-te only supports inbound route-map
         return
      attrNamePresent = attrName + "Present"

      if getattr( peer, attrNamePresent ):
         if getattr( peer, attrName ):
            cmd = 'neighbor %s route-map %s %s' \
                     % ( key, getattr( peer, attrName ), direction.lower() )
         else:
            cmd = getExplicitNoCmd( 'neighbor %s route-map %s' \
                                       % ( key, direction.lower() ), nedMode )
         cmds.append( cmd )
      elif saveAll and not peer.isPeerGroupPeer:
         cmd = getExplicitNoCmd( 'neighbor %s route-map %s' \
                                    % ( key, direction.lower() ), nedMode )
         cmds.append( cmd )

def savePeerPerAfRcf( peer, cmds, key, addrFamily, bgpConfig, saveAll,
                     nedMode ):
   directions = [ "In" ]
   if RcfLibToggleLib.toggleRcfLimaEnabled():
      directions += [ "Out" ]
   for direction in directions:
      attr = "rcf%s" % direction
      attrName = BgpLib.peerConfigAttrsAfMap[ attr ].get( addrFamily )
      if not attrName:
         return
      attrNamePresent = attrName + "Present"

      if getattr( peer, attrNamePresent ):
         funcName = getattr( peer, attrName )
         if funcName:
            funcName += "()"
            cmd = 'neighbor %s rcf %s %s' \
                     % ( key, direction.lower(), funcName )
         else:
            cmd = getExplicitNoCmd( 'neighbor %s rcf %s'
                     % ( key, direction.lower() ), nedMode )
         cmds.append( cmd )
      elif saveAll and not peer.isPeerGroupPeer:
         cmd = getExplicitNoCmd( 'neighbor %s rcf %s'
                                    % ( key, direction.lower() ), nedMode )
         cmds.append( cmd )

def savePeerPerAfPrefixList( peer, cmds, key, addrFamily, bgpConfig, saveAll,
                             afCfgPresent, nedMode ):

   for direction in [ "In", "Out" ]:
      attr = "prefixList%s" % direction
      attrName = BgpLib.peerConfigAttrsAfMap[ attr ].get( addrFamily )
      attrNamePresent = attrName + "Present"

      if getattr( peer, attrNamePresent ):
         if getattr( peer, attrName ):
            cmd = 'neighbor %s prefix-list %s %s' \
                     % ( key, getattr( peer, attrName ), direction.lower() )
         else:
            cmd = getExplicitNoCmd( 'neighbor %s prefix-list %s' \
                                       % ( key, direction.lower() ), nedMode )
         cmds.append( cmd )
      elif saveAll and not peer.isPeerGroupPeer:
         cmd = getExplicitNoCmd( 'neighbor %s prefix-list %s' \
                                    % ( key, direction.lower() ), nedMode )
         cmds.append( cmd )

def savePeerAfDefOrigin( peer, cmds, key, addrFamily, saveAll, nedMode ):

   attrBase = \
         BgpLib.peerConfigAttrsAfMap[ 'defaultOriginate' ].get( addrFamily )
   attrPresent = attrBase + 'Present'
   attrRouteMap = attrBase + 'RouteMap'
   attrAlways = attrBase + 'Always'
   if getattr( peer, attrPresent ):
      if addrFamily in vpnAfTypeCliKeywords:
         cmd = "neighbor %s default-route" % key
      else:
         cmd = "neighbor %s default-originate" % key
      if getattr( peer, attrBase ):
         rm = getattr( peer, attrRouteMap )
         if rm:
            cmd += ' route-map %s' % rm
         if getattr( peer, attrAlways ):
            cmd += ' always'
      else:
         if addrFamily in vpnAfTypeCliKeywords:
            cmd += ' disabled'
         else:
            cmd = getExplicitNoCmd( cmd, nedMode )
      cmds.append( cmd )
   elif saveAll and not peer.isPeerGroupPeer:
      if addrFamily in vpnAfTypeCliKeywords:
         cmd = 'no neighbor %s default-route' % key
         cmds.append( cmd )
      else:
         cmds.append( getExplicitNoCmd( "neighbor %s default-originate" % key,
                                        nedMode ) )

def savePeerAfGracefulRestart( peer, cmds, key, addrFamily, saveAll, nedMode ):

   attrBase = BgpLib.peerConfigAttrsAfMap[ 'gracefulRestart' ].get( addrFamily )
   attrPresent = attrBase + 'Present'
   if getattr( peer, attrPresent ):
      cmd = "neighbor %s graceful-restart" % key
      if not getattr( peer, attrBase ):
         cmd = getExplicitNoCmd( cmd, nedMode )
      cmds.append( cmd )
   elif saveAll and not peer.isPeerGroupPeer:
      if addrFamily == 'ipv4 multicast' or addrFamily == 'ipv6 multicast':
         # Graceful restart is currently not supported for IPv4/IPv6 multicast mode.
         # Once the check for ipv4/ipv6 multicast address-family is removed from
         # RoutingBgpCli.configureNeighborGracefulRestart(), this if block can
         # be removed.
         return
     # Graceful restart cannot be configured both globally and per address family.
      # If it is not configured at the global level but it is configured at some
      # address-family level, we should render the default graceful restart cmd for
      # save-all for all other address-families.
      _, anyAfConfigured = isGracefulRestartGlobalOrAfConfigured( peer )
      if anyAfConfigured: 
         cmds.append( getExplicitNoCmd( "neighbor %s graceful-restart" % key,
                                        nedMode ) )


def savePeerAfAddPathRecv( peer, cmds, key, addrFamily, saveAll, nedMode ):
   attrBase = BgpLib.peerConfigAttrsAfMap[ 'apRecv' ].get( addrFamily )
   attrPresent = attrBase + 'Present'
   if getattr( peer, attrPresent ):
      if getattr( peer, attrBase ):
         cmd = "neighbor %s additional-paths receive" % key
      else:
         cmd = getExplicitNoCmd( "neighbor %s additional-paths receive" \
                                    % key, nedMode )
      cmds.append( cmd )
   elif saveAll and not peer.isPeerGroupPeer:
      # Add-path receive is enabled by default
      cmds.append( "neighbor %s additional-paths receive" % key )

# Add-path send command
def savePeerAfAddPathSend( peer, cmds, key, addrFamily, saveAll, nedMode ):
   attrBase = BgpLib.peerConfigAttrsAfMap[ 'apSend' ].get( addrFamily )
   apSendConfig = getattr( peer, attrBase )
   cmd = ''
   for app in [ 'appAny' ]:
      appConfig = apSendConfig.get( app )
      if appConfig is not None:
         if appConfig.enable:
            cmd = 'neighbor %s %s' \
                     % ( key, addPathSendCommon( app, appConfig=appConfig ) )
         else:
            cmd = getExplicitNoCmd( 'neighbor %s %s' \
                     % ( key, addPathSendCommon( app, appConfig=appConfig ) ),
                     nedMode )
         break
   if saveAll and not peer.isPeerGroupPeer and cmd == '':
      cmd = getExplicitNoCmd( 'neighbor %s %s' \
               % ( key, addPathSendCommon( 'appAny', saveAll=saveAll ) ), nedMode )
   if cmd != '':
      cmds.append( cmd )

def savePeerAfNexthopUnchanged( peer, cmds, key, addrFamily, saveAll, nedMode ):
   attrBase = \
         BgpLib.peerConfigAttrsAfMap[ 'nexthopUnchanged' ].get( addrFamily )
   attrPresent = attrBase + 'Present'
   if getattr( peer, attrPresent ):
      if getattr( peer, attrBase ):
         cmd = "neighbor %s next-hop-unchanged" % key
      else:
         cmd = getExplicitNoCmd( "neighbor %s next-hop-unchanged" % key, nedMode )
      cmds.append( cmd )
   elif saveAll and not peer.isPeerGroupPeer:
      # disabled by default
      cmds.append( getExplicitNoCmd( "neighbor %s next-hop-unchanged" % key,
                                     nedMode ) )

def savePeerAfNhSelfSrcIntf( peer, cmds, key, addrFamily, saveAll, nedMode ):
   if type( key ) is Ip6AddrType:
      key = key.stringValue
   attr = BgpLib.peerConfigAttrsAfMap[ 'nhSelfSrcIntf' ].get( addrFamily )
   if getattr( peer, '%sPresent' % attr ):
      val = getattr( peer, attr )
      if val != getattr( peer, 'nhSelfSrcIntfDefault' ):
         cmd = "neighbor %s next-hop-self source-interface %s" % ( key, val )
      elif peer.isPeerGroupPeer:
         cmd = getExplicitNoCmd( "neighbor %s next-hop-self source-interface" % key,
                                 nedMode )
      else:
         return
      cmds.append( cmd )
   elif saveAll and not peer.isPeerGroupPeer:
      # disabled by default
      cmds.append( getExplicitNoCmd( "neighbor %s next-hop-self source-interface" \
            % key, nedMode ) )

def savePeerAfWeight( peer, cmds, key, addrFamily, saveAll, nedMode ):
   attrBase = BgpLib.peerConfigAttrsAfMap[ 'weight' ].get( addrFamily )
   attrDefault = attrBase + 'Default'
   attrPresent = attrBase + 'Present'
   defaultWeight = getattr( peer, attrDefault )
   if getattr( peer, attrPresent ):
      if ( getattr( peer, attrBase ) == defaultWeight ) and peer.isPeerGroupPeer:
         cmd = getExplicitNoCmd( "neighbor %s weight" % key, nedMode,
                                 defaultValue=defaultWeight )
      else:
         cmd = "neighbor %s weight %s" % ( key, getattr( peer, attrBase ) )
      cmds.append( cmd )
   elif saveAll and not peer.isPeerGroupPeer:
      # Per-neighbor weight cannot be configured both globally and per address
      # family. If it is not configured at the global level but it is configured at
      # some address-family level, we should render the default neighbor weight cmd
      # for save-all for all other address-families.
      if weightConfiguredInAf( peer ):
         cmds.append( getExplicitNoCmd( "neighbor %s weight" % key, nedMode,
                                        defaultValue=defaultWeight ) )

def savePeerAfOutDelay( peer, cmds, key, addrFamily, saveAll, nedMode ):
   attrBase = BgpLib.peerConfigAttrsAfMap[ 'outDelay' ].get( addrFamily )
   attrDefault = attrBase + 'Default'
   attrPresent = attrBase + 'Present'
   defaultOutDelay = getattr( peer, attrDefault )
   if getattr( peer, attrPresent ):
      if ( getattr( peer, attrBase ) == defaultOutDelay ) and peer.isPeerGroupPeer:
         cmd = getExplicitNoCmd( "neighbor %s out-delay" % key, nedMode,
                                 defaultValue=defaultOutDelay )
      else:
         cmd = "neighbor %s out-delay %s" % ( key, getattr( peer, attrBase ) )
      cmds.append( cmd )
   elif saveAll and not peer.isPeerGroupPeer:
      # Per-neighbor out-delay cannot be configured both globally and per address
      # family. If it is not configured at the global level but it is configured at
      # some address-family level, we should render the default neighbor outdelay cmd
      # for save-all for all other address-families.
      if outDelayConfiguredInAf( peer ):
         cmds.append( getExplicitNoCmd( "neighbor %s out-delay" % key, nedMode,
                                        defaultValue=defaultOutDelay ) )

def savePeerRouteTargetExportFilterDisabled( peer, cmds, key, addrFamily, saveAll ):
   if addrFamily not in [ 'ipv4', 'ipv6', 'all' ]:
      return
   if addrFamily == 'all':
      coll = peer.routeTargetExportFilterDisabled
   else:
      coll = peer.routeTargetExportFilterDisabledAf
   vpnNlriTypeList = [ NlriType( n ) for n in
                       [ 'evpnType5Ipv4', 'evpnType5Ipv6',
                        'mplsVpnIpv4Unicast', 'mplsVpnIpv6Unicast' ] ]
   for vpnNlriType in vpnNlriTypeList:
      tristate = coll.get( vpnNlriType, 'isInvalid' )
      if addrFamily != 'all':
         # AF collection is used for both v4 and v6. Skip entries not
         # belonging to this AF.
         ipVersion = int( addrFamily[ -1 ] )
         if vpnNlriType.ipVersion != ipVersion:
            continue
      if tristate == 'isInvalid' and ( not saveAll or peer.isPeerGroupPeer ):
         continue
      cmds.append( 'neighbor %s route-target export %s filter%s' %
                   ( key, vpnNlriTypeKeyword( vpnNlriType ),
                     ' disabled' if tristate == 'isTrue' else '' ) )

def savePeerAfNextHopAfV6( peer, cmds, key, saveAll, nedMode ):
   attrBase = 'v6NextHopAfV4Uni'
   attrPresent = attrBase + 'Present'
   if getattr( peer, attrPresent ):
      if getattr( peer, attrBase ) == \
                              ExtendedNextHopCapabilityEnum.isEnabledWithOriginate:
         cmds.append( 'neighbor %s next-hop address-family ipv6 originate' % key )
      elif getattr( peer, attrBase ) == ExtendedNextHopCapabilityEnum.isEnabled:
         cmds.append( 'neighbor %s next-hop address-family ipv6' % key )
      else:
         cmds.append( getExplicitNoCmd( 'neighbor %s next-hop address-family ipv6' \
                                           % key, nedMode ) )
   elif saveAll and not peer.isPeerGroupPeer:
      cmds.append( getExplicitNoCmd( 'neighbor %s next-hop address-family ipv6' \
                                        % key, nedMode ) )

def savePeerSixPeActivation( peer, cmds, addrOrPgName, afConfig, addrFamily,
                             saveAll, nedMode ):
   if peer.sixPe:
      if peer.afV6Uni == 'afActive':
         cmds.append(
            "neighbor %s activate 6pe ipv6-unicast receive" % addrOrPgName )
      else:
         cmds.append( "neighbor %s activate 6pe" % addrOrPgName )
   else:
      savePeerAfActivation( peer, cmds, addrOrPgName, afConfig, addrFamily,
                            saveAll, nedMode )

def savePeerMaxRoutes( peer, cmds, addrOrPgName, saveAll, nedMode,
                       addrFamily='all' ):
   attr = BgpLib.peerConfigAttrsAfMap[ 'maxRoutesConfig' ].get( addrFamily )
   if ( ( addrFamily == 'all' and not peer.isPeerGroupPeer ) or
        getattr( peer, attr + 'Present' ) ):
      config = getattr( peer, attr )
      maxRoutesCmd = 'neighbor %s maximum-routes %d' % (
            addrOrPgName, config.maxRoutes )
      if config.maxRoutesThresholdPercentagePresent:
         valueType = ' percent'
         threshold = str( config.maxRoutesThresholdPercentage )
      else:
         valueType = ''
         threshold = str( config.maxRoutesThreshold )
      if threshold != '0':
         maxRoutesCmd += ' warning-limit %s%s' % ( threshold, valueType )
      if config.maxRoutesWarningOnly != getattr(
            peer, attr + 'Default' ).maxRoutesWarningOnly:
         maxRoutesCmd += ' warning-only'

      cmds.append( maxRoutesCmd )

   attr = BgpLib.peerConfigAttrsAfMap[ 'maxAcceptedRoutesConfig' ].get(
      addrFamily )
   present = getattr( peer, attr + 'Present' )
   saveAll = saveAll and addrFamily == 'all'
   if present or ( saveAll and not peer.isPeerGroupPeer ):
      config = getattr( peer, attr )
      configDefault = getattr( peer, attr + 'Default' )
      numRoutes = config.maxAcceptedRoutes
      threshold = config.maxAcceptedRoutesThreshold
      cmd = 'neighbor %s maximum-accepted-routes' % addrOrPgName
      if ( not present or
           ( numRoutes == configDefault.maxAcceptedRoutes and
             threshold == configDefault.maxAcceptedRoutesThreshold and
             peer.isPeerGroupPeer ) ):
         cmd = getExplicitNoCmd( cmd, nedMode, configDefault.maxAcceptedRoutes )
      else:
         threshold = ' warning-limit %d' % threshold if \
               threshold != configDefault.maxAcceptedRoutesThreshold else ''
         cmd += ' %s%s' % ( numRoutes, threshold )

      cmds.append( cmd )

def savePeerMaxAdvRoutes( peer, cmds, addrOrPgName, saveAll, nedMode,
                       addrFamily='all' ):
   attr = 'maxAdvRoutesConfig'
   if addrFamily == 'all':
      present = getattr( peer, attr + 'Present' )
   else:
      afiSafi = Tac.Value( 'Routing::Bgp::AfiSafi',
                           *BgpLib.addressFamilies.get( addrFamily ) )
      present = afiSafi in peer.maxAdvRoutesAfiSafiConfig
   saveAll = saveAll and addrFamily == 'all'
   if present or saveAll:
      config = getattr( peer, attr ) if addrFamily == 'all' else \
               peer.maxAdvRoutesAfiSafiConfig.get( afiSafi )
      configDefault = getattr( peer, attr + 'Default' )
      numRoutes = config.maxAdvRoutes
      threshold = config.maxAdvRoutesThreshold
      cmd = 'neighbor %s maximum-advertised-routes' % addrOrPgName
      if not present:
         # Hardcode nedMode=True until BUG433700 is resolved
         cmd = getExplicitNoCmd( cmd, True, configDefault.maxAdvRoutes )
      else:
         cmd += ' %s' % numRoutes
         if config.maxAdvRoutesThresholdPercentagePresent:
            valueType = ' percent'
            threshold = str( config.maxAdvRoutesThresholdPercentage )
         else:
            valueType = ''
            threshold = str( config.maxAdvRoutesThreshold )
         if threshold != '0':
            cmd += ' warning-limit %s%s' % ( threshold, valueType )

      cmds.append( cmd )

def savePeerAfAigpSession( peer, cmds, key, addrFamily, saveAll, nedMode ):
   attrBase = BgpLib.peerConfigAttrsAfMap[ 'aigpSession' ].get( addrFamily )
   attrPresent = attrBase + 'Present'
   if getattr( peer, attrPresent ):
      if getattr( peer, attrBase ):
         cmd = "neighbor %s aigp-session" % key
      else:
         cmd = getExplicitNoCmd( "neighbor %s aigp-session" % key, nedMode )
      cmds.append( cmd )
   elif saveAll and not peer.isPeerGroupPeer:
      cmds.append( "default neighbor %s aigp-session" % key )

def savePeerAfNexthopLuOriginate( peer, cmds, addrOrPgName, saveAll, nedMode,
                                  addrFamily ):
   attrBase = BgpLib.peerConfigAttrsAfMap[ 'nexthopLuOriginate' ].get( addrFamily )
   attrPresent = attrBase + 'Present'
   cmd = "neighbor %s next-hop labeled-unicast originate" % addrOrPgName
   if getattr( peer, attrPresent ):
      if getattr( peer, attrBase ):
         cmds.append( cmd )
      else:
         cmds.append( "%s disabled" % cmd )
   elif saveAll and not peer.isPeerGroupPeer:
      cmds.append( "default %s" % cmd )

def savePeerAfConfig( cmds, bgpConfig, defaultVrfBgpConfig, peerKey, saveAll,
                      addrFamily, afCfgPresent ):
   nedMode = getattr( defaultVrfBgpConfig, 'noEqualsDefaultMode' )
   
   addrOrPgName = peerKey.stringValue

   peer = bgpConfig.neighborConfig.get( peerKey )
   if not peer and saveAll:
      peer = bgpConfig.defaultPeerConfig

   if peer:
      afConfigAttrName = BgpLib.peerConfigAttrsAfMap[ 'af' ].get( addrFamily )
      afConfig = getattr( peer, afConfigAttrName )

      if addrFamily == 'ipv6' and peerKey.type in [ 'peerIpv4', 'peerGroup' ]:
         savePeerSixPeActivation( peer, cmds, addrOrPgName, afConfig, addrFamily,
                                  saveAll, nedMode )
      else:
         savePeerAfActivation( peer, cmds, addrOrPgName, afConfig, addrFamily,
                               saveAll, nedMode )

      # Only neighbor activate and neighbor route-map in command are supported
      # in sr-te sub-mode
      if 'sr-te' in addrFamily:
         if addrFamily in BgpLib.peerConfigAttrsAfMap[ 'routeMapIn' ]:
            savePeerPerAfRouteMap( peer, cmds, addrOrPgName, addrFamily, bgpConfig,
                                   saveAll, nedMode )
         return

      if addrFamily in BgpLib.peerConfigAttrsAfMap[ 'apRecv' ]:
         savePeerAfAddPathRecv( peer, cmds, addrOrPgName, addrFamily, saveAll,
                                nedMode )

      if addrFamily in BgpLib.peerConfigAttrsAfMap[ 'gracefulRestart' ] and \
            ( BgpCommonToggleLib.toggleArBgpLuGracefulRestartEnabled() or
              ( addrFamily != 'ipv4 labeled-unicast' and
                addrFamily != 'ipv6 labeled-unicast' ) ):
         savePeerAfGracefulRestart( peer, cmds, addrOrPgName, addrFamily, saveAll,
                                    nedMode )

      if addrFamily in BgpLib.peerConfigAttrsAfMap[ 'routeMapIn' ] and \
         addrFamily in BgpLib.peerConfigAttrsAfMap[ 'routeMapOut' ]:
         savePeerPerAfRouteMap( peer, cmds, addrOrPgName, addrFamily, bgpConfig,
                                saveAll, nedMode )

      if addrFamily in BgpLib.peerConfigAttrsAfMap[ 'rcfIn' ] or \
         addrFamily in BgpLib.peerConfigAttrsAfMap[ 'rcfOut' ]:
         savePeerPerAfRcf( peer, cmds, addrOrPgName, addrFamily, bgpConfig, saveAll,
                           nedMode )

      if addrFamily in BgpLib.peerConfigAttrsAfMap[ 'weight' ]:
         savePeerAfWeight( peer, cmds, addrOrPgName, addrFamily, saveAll, nedMode )

      if addrFamily in BgpLib.peerConfigAttrsAfMap[ 'outDelay' ]:
         savePeerAfOutDelay( peer, cmds, addrOrPgName, addrFamily, saveAll, nedMode )

      if addrFamily in BgpLib.peerConfigAttrsAfMap[ 'prefixListIn' ] and \
         addrFamily in BgpLib.peerConfigAttrsAfMap[ 'prefixListOut' ] :
         savePeerPerAfPrefixList( peer, cmds, addrOrPgName, addrFamily, bgpConfig,
                                  saveAll, afCfgPresent, nedMode )
      
      if addrFamily in BgpLib.peerConfigAttrsAfMap[ 'defaultOriginate' ]:
         savePeerAfDefOrigin( peer, cmds, addrOrPgName, addrFamily, saveAll,
                              nedMode )

      if addrFamily in BgpLib.peerConfigAttrsAfMap[ 'apSend' ]:
         savePeerAfAddPathSend( peer, cmds, addrOrPgName, addrFamily, saveAll,
                              nedMode )

      if addrFamily == 'ipv4':
         savePeerAfNextHopAfV6( peer, cmds, addrOrPgName, saveAll, nedMode )

      if addrFamily in BgpLib.peerConfigAttrsAfMap[ 'nexthopUnchanged' ]:
         savePeerAfNexthopUnchanged( peer, cmds, addrOrPgName, addrFamily, saveAll,
                                     nedMode )

      if addrFamily in BgpLib.peerConfigAttrsAfMap[ 'nhSelfSrcIntf' ]:
         savePeerAfNhSelfSrcIntf( peer, cmds, addrOrPgName, addrFamily, saveAll,
                                  nedMode )

      if addrFamily in BgpLib.peerConfigAttrsAfMap[ 'maxRoutesConfig' ]:
         savePeerMaxRoutes( peer, cmds, addrOrPgName, saveAll, nedMode,
            addrFamily )

      savePeerMaxAdvRoutes( peer, cmds, addrOrPgName, saveAll, nedMode, addrFamily )

      if RoutingLibToggleLib.toggleAccumulatedIgpMetricEnabled():
         if addrFamily in BgpLib.peerConfigAttrsAfMap[ 'aigpSession' ]:
            savePeerAfAigpSession( peer, cmds, addrOrPgName, addrFamily, saveAll,
                                   nedMode )

      if addrFamily == 'evpn':
         saveEvpnEncapType( cmds, peer, saveAll, addrOrPgName )

      if addrFamily == 'rt-membership':
         savePeerRtMembershipConfig( cmds, peer, saveAll, addrOrPgName )

      if BgpCommonToggleLib.toggleArBgpNexthopLuOriginateEnabled():
         if addrFamily in BgpLib.peerConfigAttrsAfMap[ 'nexthopLuOriginate' ]:
            savePeerAfNexthopLuOriginate( peer, cmds, addrOrPgName, saveAll, nedMode,
                                          addrFamily )

def evpnEncapTacValueToKeyword( encap ):
   if encap == 'encapVxlan':
      return 'vxlan'
   if encap == 'encapMpls':
      return 'mpls'
   raise Exception( 'Unexpected encapsulation %s' % ( encap ) )

def saveEvpnEncapType( cmds, config, saveAll, peer='default' ):
   # 'neighbor default encapsulation ...' is saved only when encap is not default
   # 'neighbor PEER encapsulation ...' is saved whenever peer configuration exists
   defaultPeer = ( peer == 'default' )
   if config.afEvpnEncap != config.afEvpnEncapDefault or not defaultPeer:
      if not defaultPeer and not config.afEvpnEncapPresent:
         return
      additionalArgs = ''
      cliEncapType = evpnEncapTacValueToKeyword( config.afEvpnEncap )
      if ( cliEncapType == 'mpls' and
            config.afEvpnMplsNexthopSelfSrcIntf !=
            config.afEvpnMplsNexthopSelfSrcIntfDefault ):
         if peer == 'default' or config.afEvpnMplsNexthopSelfSrcIntfPresent:
            additionalArgs = 'next-hop-self source-interface %s' % (
                  config.afEvpnMplsNexthopSelfSrcIntf )
      cmds.append( 'neighbor %s encapsulation %s %s' % ( peer, cliEncapType,
         additionalArgs ) )
   elif saveAll:
      cliEncapType = evpnEncapTacValueToKeyword( config.afEvpnEncapDefault )
      cmds.append( 'neighbor %s encapsulation %s' % ( peer, cliEncapType ) )

def savePeerRtMembershipConfig( cmds, config, saveAll, addrOrPgName ):
   if not BgpCommonToggleLib.toggleBgpRtMembershipEnabled():
      return

   # neighbor PEER default-route-target [ only | disabled ]
   cmd = 'neighbor %s default-route-target' % ( addrOrPgName )
   if ( config.rtMembershipDefaultRouteTarget ==
        'rtMembershipDefaultRouteTargetEnabledWithOnly' ):
      cmd += ' only'
   elif ( config.rtMembershipDefaultRouteTarget ==
          'rtMembershipDefaultRouteTargetDisabled' ):
      cmd += ' disabled'
   if ( config.rtMembershipDefaultRouteTarget !=
        config.rtMembershipDefaultRouteTargetDefault ):
      cmds.append( cmd )
   elif saveAll:
      cmds.append( 'no ' + cmd )

   # neighbor PEER default-route-target encoding origin-as omit [ disabled ]
   cmd = 'neighbor %s default-route-target encoding origin-as omit' % (
      addrOrPgName )
   if ( config.rtMembershipDefaultRouteTargetEncoding ==
        'rtMembershipDefaultRouteTargetEncodingOriginAsSet' ):
      cmd += ' disabled'
   if ( config.rtMembershipDefaultRouteTargetEncoding !=
        config.rtMembershipDefaultRouteTargetEncodingDefault ):
      cmds.append( cmd )
   elif saveAll:
      cmds.append( 'no ' + cmd )

def saveBgpVpnAfConfig( bgpConfig, defaultVrfBgpConfig, addrFamily, asdotConfigured,
                        saveAll ):
   """This method returns the VPN config for the given VRF and address-famliy"""
   cmds = []
   routeTargetImportAttrName = BgpLib.bgpConfigAttrsAfMap[ 'routeTargetImport'
                                                         ].get( addrFamily )
   routeTargetExportAttrName = BgpLib.bgpConfigAttrsAfMap[ 'routeTargetExport'
                                                         ].get( addrFamily )
   if routeTargetImportAttrName and routeTargetExportAttrName:
      for importExport, attrName in [ ( 'import', routeTargetImportAttrName ),
                                      ( 'export', routeTargetExportAttrName ) ]:
         for rt, isRtSet in getattr( bgpConfig, attrName ).iteritems():
            assert isRtSet
            cmds.append( "route-target %s %s" % ( importExport,
               routeTargetToPrint( rt, asdotConfigured ) ) )
         # Display these sorted on the cli keyword for the vpnAf.
         for vpnType, vpnTypeStr in sorted(
               vpnTypeAttrMap.items(),
               key=( lambda ( vpnType, vpnTypeStr ): vpnTypeStr ) ):
            vpnTypeAttrName = attrName + vpnType
            # Getting the vpnTypeAttrName from within bgpConfig will fail for
            # some of the attrName, vpnType combos like -
            # routeTargetImportV4VpnIpv6, routeTargetImportV6VpnIpv4.
            # Hence checking for validity of vpnTypeAttr here.
            vpnTypeAttr = getattr( bgpConfig, vpnTypeAttrName, None )
            if vpnTypeAttr:
               for rt, isRtSet in vpnTypeAttr.iteritems():
                  assert isRtSet
                  cmds.append( "route-target %s %s %s" % ( importExport,
                               vpnTypeStr,
                               routeTargetToPrint( rt, asdotConfigured ) ) )

   routeMapImportAttrName = (
         BgpLib.bgpConfigAttrsAfMap[ 'routeMapImport' ].get( addrFamily ) )
   routeMapExportAttrName = (
         BgpLib.bgpConfigAttrsAfMap[ 'routeMapExport' ].get( addrFamily ) )
   if routeMapImportAttrName and routeMapExportAttrName:
      for importExport, attrName in [ ( 'import', routeMapImportAttrName ),
                                      ( 'export', routeMapExportAttrName ) ]:
         # An explicit key is required for sort method here because vpnAf is of
         # type Routing::Bgp::AfiSafi but we need to sort them based on thier
         # cli keywork here.
         for vpnAf, rmName in sorted(
               getattr( bgpConfig, attrName ).items(),
                        key=lambda item: vpnAfTypeMap.get( item[ 0 ] ) ):
            if vpnAfTypeMap.get( vpnAf ):
               cmds.append( "route-target %s %s route-map %s" % ( importExport,
                  vpnAfTypeMap[ vpnAf ], rmName ) )

   importedRouteExportAttrName = \
      BgpLib.bgpConfigAttrsAfMap[ 'allowImportedRouteToExport' ].get( addrFamily )
   if importedRouteExportAttrName:
      for importExport, attrName in [ ( 'export', importedRouteExportAttrName ) ]:
         for vpnAf, isImpRoSet in sorted(
               getattr( bgpConfig, attrName ).items(),
                        key=lambda item: vpnAfTypeMap.get( item[ 0 ] ) ):
            assert isImpRoSet
            if vpnAfTypeMap.get( vpnAf ):
               cmds.append( "route-target %s %s %s" % ( importExport,
                            vpnAfTypeMap[ vpnAf ], 'imported-route' ) )

   for peerKey in bgpConfig.neighborConfig:
      addrOrPgName = peerKey.stringValue
      peer = bgpConfig.neighborConfig.get( peerKey )
      savePeerRouteTargetExportFilterDisabled( peer, cmds, addrOrPgName,
                                               addrFamily, saveAll )

   return cmds

# saveAfConfigCallbackDict is the dictionary of callbacks keyed by address families.
# AfiSafi plugins can register and add their callbacks to it. It is used by
# doShowBgpConfig to display address family specific config in
# show bgp config active output, other than the configs shown by saveAfConfig
saveAfConfigCallbackDict = {}

# showBgpConfigInstanceCallbackDict is a list of callbacks for instance level
# submodes.  The callbacks take the bgpConfig as an arg, and return a dictionary
# of submode commands to a list of commands configured in that submode.
showBgpConfigInstanceCallbacks = []

def saveAfConfig( bgpConfig, defaultVrfBgpConfig, vrfName, addrFamily,
                  asdotConfigured=False, options=None ):
   cmds = []
   saveAll = options.saveAll if options else False
   saveAllDetail = options.saveAllDetail if options else False
   afCfgPresent = hasAfConfig( bgpConfig )

   if vrfName:
      cmds.extend( saveBgpVpnAfConfig( bgpConfig, defaultVrfBgpConfig, addrFamily,
                                       asdotConfigured, saveAll ) )

   def maybeSetSkipRibBestpath( suffix ):
      if getattr( bgpConfig, 'skipRib%s' % suffix ):
         if getattr( bgpConfig, 'skipBestpath%s' % suffix ):
            cmds.append( "bgp skip rib-install bestpath-selection" )
         else:
            cmds.append( "bgp skip rib-install" )
      elif saveAll:
         cmds.append( "no bgp skip rib-install bestpath-selection" )

   if addrFamily == 'ipv4':
      if afCfgPresent:
         if bgpConfig.picIPv4Uni:
            if not bgpConfig.picEcmpPrimaryIPv4Uni:
               cmds.append( "bgp additional-paths install" )
            else:
               cmds.append( "bgp additional-paths install ecmp-primary" )
         elif saveAll:
            cmds.append( "no bgp additional-paths install" )
      # Print bgp next-hop address-family ipv6 in address family ipv4 mode.
      if bgpConfig.v6NextHopAfV4Uni != bgpConfig.v6NextHopAfV4UniDefault:
         if bgpConfig.v6NextHopAfV4Uni == \
                            ExtendedNextHopCapabilityEnum.isEnabledWithOriginate:
            cmds.append( "bgp next-hop address-family ipv6 originate" )
         else:
            cmds.append( "bgp next-hop address-family ipv6" )
      elif saveAll:
         cmds.append( "no bgp next-hop address-family ipv6" )
      maybeSetSkipRibBestpath( 'IPv4Uni' )
   elif addrFamily == 'ipv6':
      maybeSetSkipRibBestpath( 'IPv6Uni' )
      if bgpConfig.picIPv6Uni:
         if not bgpConfig.picEcmpPrimaryIPv6Uni:
            cmds.append( "bgp additional-paths install" )
         else:
            cmds.append( "bgp additional-paths install ecmp-primary" )
      elif saveAll:
         cmds.append( "no bgp additional-paths install" )
   # update wait-for-convergence command
   convergenceNoSyncAttrName = BgpLib.bgpConfigAttrsAfMap[ 'convergenceNoSync' ].\
                                                         get( addrFamily )
   if convergenceNoSyncAttrName:
      isConvergenceNoSyncSet, convergenceNoSyncVal = evaluateTristate( bgpConfig,
                                                     bgpConfig,
                                                     convergenceNoSyncAttrName )
      if isConvergenceNoSyncSet:
         if convergenceNoSyncVal:
            cmds.append( "no update wait-for-convergence" )
         else:
            cmds.append( "update wait-for-convergence" )
   # bgp missing-policy address family specific command
   policyActions = dict(
      policyActionUnspecified='unspecified',
      policyActionPermit='permit',
      policyActionDeny='deny',
      policyActionDenyBoth='deny-in-out' )
   # Some address families like rt-membership will not influence
   # missing policy, so do not CliSave the missing policy command
   skipMissingPolicyAf = [ 'rt-membership', 'path-selection' ]
   for direction in [ "In", "Out" ]:
      tok = [ 'bgp', 'missing-policy', 'direction', direction.lower(), 'action' ]
      actionAfAttr = 'missingPolicyActionAf' + direction
      actionIncludeSubAfAttr = 'missingPolicyIncludeSubRouteMapAf' + direction
      actionIncludePfxAfAttr = 'missingPolicyIncludePrefixListAf' + direction
      actionAfConfig = getattr( bgpConfig, actionAfAttr )
      actionIncludeSubAfConfig = getattr( bgpConfig, actionIncludeSubAfAttr )
      actionIncludePfxAfConfig = getattr( bgpConfig, actionIncludePfxAfAttr )
      af = Tac.Value( "Routing::Bgp::AfiSafi", *addressFamilies[ addrFamily ] )
      if af in actionAfConfig:
         tok.append( policyActions[ actionAfConfig[ af ] ] )
         insertPos = 2
         if 'isTrue' in ( actionIncludeSubAfConfig.get( af ),
                          actionIncludePfxAfConfig.get( af ) ):
            # insert these keywords into the command token list
            tok.insert( insertPos, 'include' )
            insertPos += 1
            if actionIncludeSubAfConfig.get( af ) == 'isTrue':
               # insert these keywords into the command token list
               tok.insert( insertPos, 'sub-route-map' )
               insertPos += 1
            if actionIncludePfxAfConfig.get( af ) == 'isTrue':
               # insert these keywords into the command token list
               tok.insert( insertPos, 'prefix-list' )
               insertPos += 1
         missingPolicyCommand = ' '.join( tok )
         cmds.append( missingPolicyCommand )
      elif saveAll and addrFamily not in skipMissingPolicyAf:
         missingPolicyCommand = 'no ' + ' '.join( tok )
         cmds.append( missingPolicyCommand )
   # Add-path receive command
   apRecvAttrName = BgpLib.bgpConfigAttrsAfMap[ 'apRecv' ].get( addrFamily )
   if apRecvAttrName:
      isAPRecvSet, apRecvVal = evaluateTristate( bgpConfig, bgpConfig,
                                                 apRecvAttrName )
      if isAPRecvSet:
         if apRecvVal:
            cmds.append( 'bgp additional-paths receive' )
         else:
            cmds.append( 'no bgp additional-paths receive' )
      elif saveAll:
         # Add-path receive is enabled by default
         cmds.append( 'bgp additional-paths receive' )

   apSendAttrName = BgpLib.bgpConfigAttrsAfMap[ 'apSend' ].get( addrFamily )
   if apSendAttrName:
      apSendConfig = getattr( bgpConfig, apSendAttrName )
      cmd = ''
      for app in [ 'appAny' ]:
         appConfig = apSendConfig.get( app )
         if appConfig is not None:
            if appConfig.enable:
               cmd = 'bgp '
            else:
               cmd = 'no bgp '
            cmd += addPathSendCommon( app, appConfig=appConfig )
            break
      if saveAll and cmd == '':
         cmd = 'no bgp '
         cmd += addPathSendCommon( 'appAny', saveAll=saveAll )
      if cmd != '':
         cmds.append( cmd )

   # next-hop-unchanged command
   nhUnchangedAttrName = \
         BgpLib.bgpConfigAttrsAfMap[ 'nexthopUnchanged' ].get( addrFamily )
   if nhUnchangedAttrName:
      isNhUnchangedSet, nhUnchVal = evaluateTristate( bgpConfig, bgpConfig,
                                                      nhUnchangedAttrName )
      if isNhUnchangedSet:
         if nhUnchVal:
            cmds.append( 'bgp next-hop-unchanged' )
         else:
            cmds.append( 'no bgp next-hop-unchanged' )
      elif saveAll:
         cmds.append( 'no bgp next-hop-unchanged' )

   routeInstallMapAttrName = \
         BgpLib.bgpConfigAttrsAfMap[ 'routeInstallMap' ].get( addrFamily )
   if routeInstallMapAttrName:
      val = getattr( bgpConfig, routeInstallMapAttrName )
      defVal = getattr( bgpConfig, routeInstallMapAttrName + 'Default' )
      if val != defVal:
         cmds.append( 'bgp route install-map %s' % val )
      elif saveAll:
         cmds.append( 'no bgp route install-map' )

   if addrFamily == 'evpn':
      saveEvpnEncapType( cmds, bgpConfig, saveAll )

   if ( addrFamily == 'ipv4' or addrFamily == 'ipv6' or
        addrFamily == 'ipv4 labeled-unicast' or
        addrFamily == 'ipv6 labeled-unicast' or
        addrFamily == 'evpn' or
        addrFamily == 'vpn-ipv4' or addrFamily == 'vpn-ipv6' ):
      if addrFamily == 'ipv4':
         if bgpConfig.afV4NexthopResolutionDisabled:
            cmds.append( 'next-hop resolution disabled' )
         elif saveAll or saveAllDetail:
            cmds.append( 'no next-hop resolution disabled' )
      elif addrFamily == 'ipv6':
         if bgpConfig.afV6NexthopResolutionDisabled:
            cmds.append( 'next-hop resolution disabled' )
         elif saveAll or saveAllDetail:
            cmds.append( 'no next-hop resolution disabled' )

      pnhts = []
      pathAfiSafiNhType = Tac.Type( 'Routing::Bgp::PathAfiSafiNexthopType' )
      if addrFamily == 'ipv4':
         pnht = pathAfiSafiNhType.ipv4UniNexthopsAll
         pnhts = [ pnht ]
      elif addrFamily == 'ipv4 labeled-unicast':
         pnht = pathAfiSafiNhType.ipv4MplsNexthopsAll
         pnhts = [ pnht ]
      elif addrFamily == 'ipv6':
         pnht1 = pathAfiSafiNhType.ipv6UniNexthopsAll
         # 6pe should be last since its a specialization of ipv6
         pnht2 = pathAfiSafiNhType.sixPeNexthopsAll
         pnhts = [ pnht1, pnht2 ]
      elif addrFamily == 'ipv6 labeled-unicast':
         pnht = pathAfiSafiNhType.ipv6MplsNexthopsAll
         pnhts = [ pnht ]
      elif addrFamily == 'evpn':
         pnht1 = pathAfiSafiNhType.l2vpnEvpnNexthopsEncapMpls
         pnht2 = pathAfiSafiNhType.l2vpnEvpnNexthopsEncapVxlan
         pnhts = [ pnht1, pnht2 ]
      elif addrFamily == 'vpn-ipv4':
         pnht = pathAfiSafiNhType.ipv4MplsVpnNexthopsAll
         pnhts = [ pnht ]
      elif addrFamily == 'vpn-ipv6':
         pnht = pathAfiSafiNhType.ipv6MplsVpnNexthopsAll
         pnhts = [ pnht ]

      for pnht in pnhts:
         rmName = bgpConfig.nhResolutionRibProfileRouteMapConfig.get( pnht )
         if rmName:
            cmd = "next-hop resolution ribs route-map %s" % rmName
            cmds.append( cmd )
            continue

         # entries are expected to always remain in the collection
         profile = bgpConfig.nhResolutionRibProfileConfig.get( pnht )
         defaultProfile = bgpConfig.nhResolutionRibProfileConfigDefault.get( pnht )
         if not profile:
            continue
         if not saveAll and profile == defaultProfile:
            continue

         if pnht == pathAfiSafiNhType.sixPeNexthopsAll:
            # SPECIAL CASE:
            # vrf-ipv6 does not support 6pe (only ipv6 does)
            # but because of two things:
            #   1. nhResolutionRibProfileConfig defaults are set in BgpConfig
            #      constructor
            #   2. an instance of BgpConfig is constructed per vrf
            # this means the vrf's will have a default value for 6pe which is
            # undesirable.
            # So :( we'll just prevent it from ever being displayed here...
            if vrfName:
               continue
            cmd = "next-hop 6pe resolution ribs"
         elif pnht == pathAfiSafiNhType.l2vpnEvpnNexthopsEncapMpls:
            cmd = "next-hop mpls resolution ribs"
         elif pnht == pathAfiSafiNhType.l2vpnEvpnNexthopsEncapVxlan:
            cmd = "next-hop vxlan resolution ribs"
         else:
            cmd = "next-hop resolution ribs"
         cmd += ' ' + profile.stringValue()
         cmds.append( cmd )

   # Iterate through the ordered collection bgpConfig.neighborConfig
   # and call savePeerAfConfig on every element.
   for peerKey in bgpConfig.neighborConfig:
      savePeerAfConfig( cmds, bgpConfig, defaultVrfBgpConfig, peerKey,
                        saveAll, addrFamily, afCfgPresent )

   # Only print networks in the address-family block if we have
   # unicast address-family specific config
   if afCfgPresent and ( addrFamily in [ 'ipv4', 'ipv6' ] ):
      saveNetworkList( cmds, bgpConfig, filterNetworkAf=( addrFamily ) )
   else:
      networkListAttrName = \
            BgpLib.bgpConfigAttrsAfMap[ 'networkList' ].get( addrFamily )
      if networkListAttrName:
         saveNetworkList( cmds, bgpConfig, addrFamily=addrFamily )

   # Print '[no] bgp redistribute-internal' in addrFamily mode if it was
   # configured in either of the addrFamily modes and if saving all then
   # we print the default value when bgp redistribute-internal is configured
   # for the other addrFamily mode.
   redistInternalAttrName = \
         BgpLib.bgpConfigAttrsAfMap[ 'bgpRedistInternal' ].get( addrFamily )
   if redistInternalAttrName:
      if bgpConfig.bgpRedistInternal == redistInternalStateEnum.notConfigured:
         state = getattr( bgpConfig, redistInternalAttrName )
         if state == redistInternalStateEnum.disableRedistInt:
            cmds.append( 'no bgp redistribute-internal' )
         elif saveAll:
            cmds.append( 'bgp redistribute-internal' )

   gracefulRestartAttrName = \
         BgpLib.bgpConfigAttrsAfMap[ 'gracefulRestart' ].get( addrFamily )
   if gracefulRestartAttrName and \
         ( BgpCommonToggleLib.toggleArBgpLuGracefulRestartEnabled() or
           ( addrFamily != 'ipv4 labeled-unicast' and
             addrFamily != 'ipv6 labeled-unicast' ) ):
      isGrSet, grVal = evaluateTristate( bgpConfig, bgpConfig,
                                         gracefulRestartAttrName )
      if isGrSet:
         if grVal:
            cmds.append( "graceful-restart" )
         else:
            cmds.append( "no graceful-restart" )

   redistributeAttrName = \
         BgpLib.bgpConfigAttrsAfMap[ 'redistributeConfig' ].get( addrFamily )
   if redistributeAttrName:
      # Print 'redistribute <proto>' in addrFamily mode
      redistributeConfig = getattr( bgpConfig, redistributeAttrName )
      for protocol in sorted( redistributeConfig.values(),
                              key=lambda p: p.proto ):
         cmd = redistributeCommand( protocol )
         if protocol.routeMap:
            cmd += ' route-map %s' % protocol.routeMap
         if protocol.rcf:
            cmd += ' rcf %s()' % protocol.rcf
         cmds.append( cmd )

   if RoutingLibToggleLib.toggleAccumulatedIgpMetricEnabled():
      # aigp-session for ibgp peers (enabled by default)
      attr = 'aigpSessionIbgpDisabled'
      attrName = BgpLib.bgpConfigAttrsAfMap[ attr ].get( addrFamily )
      if attrName:
         isAttrSet, attrVal = evaluateTristate( bgpConfig, bgpConfig, attrName )
         if isAttrSet:
            if attrVal:
               cmds.append( 'no aigp-session ibgp' )
            else:
               cmds.append( 'aigp-session ibgp' )
         elif saveAll:
            cmds.append( 'aigp-session ibgp' )

      # aigp-session for confed peers (enabled by default)
      attr = 'aigpSessionConfedDisabled'
      attrName = BgpLib.bgpConfigAttrsAfMap[ attr ].get( addrFamily )
      if attrName:
         isAttrSet, attrVal = evaluateTristate( bgpConfig, bgpConfig, attrName )
         if isAttrSet:
            if attrVal:
               cmds.append( 'no aigp-session confederation' )
            else:
               cmds.append( 'aigp-session confederation' )
         elif saveAll:
            cmds.append( 'aigp-session confederation' )

      # aigp-session for ebgp peers (disabled by default)
      attr = 'aigpSessionEbgpEnabled'
      attrName = BgpLib.bgpConfigAttrsAfMap[ attr ].get( addrFamily )
      if attrName:
         isAttrSet, attrVal = evaluateTristate( bgpConfig, bgpConfig, attrName )
         if isAttrSet:
            if attrVal:
               cmds.append( 'aigp-session ebgp' )
            else:
               cmds.append( 'no aigp-session ebgp' )
         elif saveAll:
            cmds.append( 'no aigp-session ebgp' )

   return cmds


@CliSave.saver( 'Routing::Bgp::Config', 'routing/bgp/config', 
      requireMounts = ( 'routing/bgp/asn/config', ) )
def saveBaseAfConfig( bgpConfig, root, sysdbRoot, options,
      requireMounts ):
   if bgpConfig.asNumber == 0:
      return
   if options.intfFilter:
      return

   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]
   asdotConfigured = isAsdotConfigured( asnConfig )
   bgpMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
         bgpConfig.asNumber, asdotConfigured ) )
   for af in BgpLib.bgpConfigAttrAfList():
      cmds = saveAfConfig( bgpConfig, bgpConfig, "", af, asdotConfigured, options )
      if cmds:
         bgpAfMode = bgpMode[ RouterBgpBaseAfConfigMode
               ].getOrCreateModeInstance( af )
         for cmd in cmds:
            bgpAfMode[ 'Bgp.afConfig.%s' % af ].addCommand( cmd )

@CliSave.saver( 'Routing::Bgp::VrfConfigDir', 'routing/bgp/vrf/config',
                requireMounts = ( 'routing/bgp/config', 
                                  'routing/bgp/asn/config', ) )
def saveVrfAfConfig( vrfConfigDir, root, sysdbRoot, options,
                     requireMounts ):
   bgpConfig = requireMounts[ 'routing/bgp/config' ]
   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]
   if bgpConfig.asNumber == 0:
      return
   if options.intfFilter:
      return

   asdotConfigured = isAsdotConfigured( asnConfig )
   bgpMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
         bgpConfig.asNumber, asdotConfigured ) )

   for vrfName in vrfConfigDir.vrfConfig:
      vrfBgpConfig = vrfConfigDir.vrfConfig[ vrfName ]
      bgpVrfMode = bgpMode[ RouterBgpVrfConfigMode ].getOrCreateModeInstance(
            vrfName )
      for af in BgpLib.bgpConfigAttrAfList( vrfName ):
         cmds = saveAfConfig( vrfBgpConfig, bgpConfig, vrfName, af,
                              asdotConfigured, options )
         if cmds:
            bgpVrfAfMode = bgpVrfMode[ RouterBgpVrfAfConfigMode
                  ].getOrCreateModeInstance( ( af, vrfName ) )
            for cmd in cmds:
               bgpVrfAfMode[ 'Bgp.vrf.afConfig.%s' % af ].addCommand( cmd )

@CliSave.saver( 'Routing::AsnConfig', 'routing/bgp/asn/config',
                requireMounts = ( 'routing/bgp/config', ) )
def saveAsnConfig( asnConfig, root, sysdbRoot, options,
                   requireMounts ):
   bgpConfig = requireMounts[ 'routing/bgp/config' ]
   if bgpConfig.asNumber == 0:
      return
   if options.intfFilter:
      return

   bgpMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
         bgpConfig.asNumber, isAsdotConfigured( asnConfig ), ) )

   cmds = bgpMode[ 'Bgp.config' ]

   def asnNotationString( asnNotation ):
      if asnNotation == Tac.Type( 'Routing::AsnNotation' ).asnNotationAsdot:
         return "asdot"
      elif asnNotation == Tac.Type( 'Routing::AsnNotation' ).asnNotationAsplain:
         return "asplain"
      else:
         raise Exception( 'Invalid AS Number Notation Configured' )

   asnNotation = asnConfig.asnNotation
   msg = "bgp asn notation %s" % asnNotationString( asnNotation )
   if asnNotation == asnConfig.asnNotationDefault and not options.saveAll:
      return
   cmds.addCommand( msg )

@CliSave.saver( 'Routing::Bgp::Config', 'routing/bgp/config' )
def saveServiceBgpConfig( bgpConfig, root, sysdbRoot, options ):
   cmd = None
   if bgpConfig.noEqualsDefaultMode != bgpConfig.noEqualsDefaultModeDefault:
      cmd = 'service routing configuration bgp no-equals-default'
   elif options.saveAll:
      cmd = 'no service routing configuration bgp no-equals-default'
   if cmd:
      root[ 'Bgp.service' ].addCommand( cmd )
