# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from Toggles import (
   IpRibLibToggleLib,
   RoutingLibToggleLib,
   ArnetToggleLib
   )
from CliMode.RouterGeneral import RouterGeneralBaseMode
from CliMode.RouterGeneral import RouterGeneralVrfBaseMode
import CliSave
from IpLibConsts import DEFAULT_VRF
from IntfCliSave import IntfConfigMode
import Tac
# pkgdeps: rpmwith %{_libdir}/preinit/RouterGeneralCli

defaultRouterIdV4 = Tac.Value( 'Arnet::IpAddr' ).ipAddrZero
defaultRouterIdV6 = Tac.Value( 'Arnet::Ip6Addr' )

def getVrfName( vrf ):
   return DEFAULT_VRF if vrf == '' else vrf

def getRouterGeneralVrfMode( rtrGeneralMode, vrfName ):
   # Get reference to the 'vrf ...' mode
   vrfName = '' if vrfName == DEFAULT_VRF else vrfName
   rtrGeneralVrfMode = rtrGeneralMode[ RouterGeneralVrfCliSaveMode ].\
                        getOrCreateModeInstance( ( vrfName ) )
   return rtrGeneralVrfMode

def getRouterGeneralVrfModeCtx( root, vrfName ):
   # Get reference to 'router general' mode
   rtrGeneralMode = root[ RouterGeneralCliSaveMode ].\
                     getOrCreateModeInstance( ( 'generalconfig' ) )

   # Get reference to the 'vrf ...' mode
   return getRouterGeneralVrfMode( rtrGeneralMode, vrfName )

class RouterGeneralCliSaveMode( RouterGeneralBaseMode, CliSave.Mode ):
   def __init__( self, param ):
      RouterGeneralBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( RouterGeneralCliSaveMode,
                                       after=[ IntfConfigMode ] )
RouterGeneralCliSaveMode.addCommandSequence( 'config' )

class RouterGeneralVrfCliSaveMode( RouterGeneralVrfBaseMode, CliSave.Mode ):

   def enterCmd( self ):
      return 'vrf %s' % ( getVrfName( self.param_ ) )

   def __init__( self, param ):
      RouterGeneralVrfBaseMode.__init__( self, getVrfName( param ) )
      CliSave.Mode.__init__( self, param )

RouterGeneralCliSaveMode.addChildMode( RouterGeneralVrfCliSaveMode )
RouterGeneralVrfCliSaveMode.addCommandSequence( 'vrfConfig' )

#---------------------------------------------------------------------------------
#                        C L I      S A V E R S
#---------------------------------------------------------------------------------

@CliSave.saver( 'Routing::General::Config', 'routing/general/config/global' )
def saveGeneralConfig( entity, root, sysdbRoot, options ):
   saveAll = options.saveAll
   def isAllDefault( entity ):
      if len( entity.vrfConfig ):
         return False
      # holds the default values for each attributes
      attrsDefaultDict = {}
      attrsDefaultDict[ 'fecPerNexthop' ] = False
      attrsDefaultDict[ 'fecEcmpEmulated' ] = False
      attrsDefaultDict[ 'fecEcmpOrdered' ] = False
      attrsDefaultDict[ 'routerIdV4' ] = defaultRouterIdV4
      attrsDefaultDict[ 'routerIdV6' ] = defaultRouterIdV6
      attrsDefaultDict[ 'resolveOverAggregates' ] = False
      attrsDefaultDict[ 'resolveNexthopGroupVias' ] = False
      attrsDefaultDict[ 'fecEcmpMixTunnelIp' ] = False
      attrsDefaultDict[ 'ipv4Routable240ClassE' ] = False
      cfgAttrs = attrsDefaultDict.iteritems()
      return all( getattr( entity, attr ) == dflt for attr, dflt in cfgAttrs )

   # Don't print anything when nothing has changed in the configuration
   # unless it's saveAll
   if not saveAll and isAllDefault( entity ):
      return
   mode = root[ RouterGeneralCliSaveMode ].getOrCreateModeInstance( 'generalconfig' )
   cmds = mode[ 'config' ]
   if ( entity.fecPerNexthop ):
      cmds.addCommand( 'next-hops fec dedicated' )
   elif saveAll:
      cmds.addCommand( 'no next-hops fec dedicated' )

   if ( entity.fecEcmpEmulated ):
      cmds.addCommand( 'rib fib fec ecmp emulated' )
   elif saveAll:
      cmds.addCommand( 'no rib fib fec ecmp emulated' )

   if RoutingLibToggleLib.toggleOrderedEcmpEnabled():
      if ( entity.fecEcmpOrdered ):
         cmds.addCommand( 'rib fib fec ecmp ordered' )
      elif saveAll:
         cmds.addCommand( 'no rib fib fec ecmp ordered' )

   if ArnetToggleLib.toggleIpv4Routable240ClassEEnabled():
      if entity.ipv4Routable240ClassE:
         cmds.addCommand( 'ipv4 routable 240.0.0.0/4' )
      elif saveAll:
         cmds.addCommand( 'no ipv4 routable 240.0.0.0/4' )

   if entity.routerIdV4 != defaultRouterIdV4:
      cmds.addCommand( 'router-id ipv4 %s' % entity.routerIdV4 )
   elif saveAll:
      cmds.addCommand( 'no router-id ipv4' )

   if entity.routerIdV6 != defaultRouterIdV6:
      cmds.addCommand( 'router-id ipv6 %s' % entity.routerIdV6 )
   elif saveAll:
      cmds.addCommand( 'no router-id ipv6' )

   if IpRibLibToggleLib.toggleMixedEcmpEnabled():
      if entity.fecEcmpMixTunnelIp:
         cmds.addCommand( 'rib fib fec ecmp compatible tunnel ip' )
      elif saveAll:
         cmds.addCommand( 'no rib fib fec ecmp compatible tunnel ip' )

   if ( entity.resolveOverAggregates ):
      cmds.addCommand( 'next-hops resolution bgp aggregates allowed' )
   elif saveAll:
      cmds.addCommand( 'no next-hops resolution bgp aggregates allowed' )

   if RoutingLibToggleLib.toggleCheckFecResolutionStatusEnabled():
      if ( entity.resolveNexthopGroupVias ):
         cmds.addCommand( 'route static nexthop-group unresolved invalid' )
      elif saveAll:
         cmds.addCommand( 'no route static nexthop-group unresolved invalid' )

   for vrfName in sorted( entity.vrfConfig ):
      # pylint: disable=unused-variable
      vrfMode = getRouterGeneralVrfMode( mode, vrfName )
      # pylint: enable=unused-variable
      vrfCmds = vrfMode[ 'vrfConfig' ]
      vrfEntity = entity.vrfConfig[ vrfName ]
      if vrfEntity.routerIdV4 != defaultRouterIdV4:
         vrfCmds.addCommand( 'router-id ipv4 %s' % vrfEntity.routerIdV4 )
      elif saveAll:
         vrfCmds.addCommand( 'no router-id ipv4' )

      if vrfEntity.routerIdV6 != defaultRouterIdV6:
         vrfCmds.addCommand( 'router-id ipv6 %s' % vrfEntity.routerIdV6 )
      elif saveAll:
         vrfCmds.addCommand( 'no router-id ipv6' )
