#!/usr/bin/env python
# Copyright (c) 2008-2011, 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave
import RadiusConstants
import Tac, IntfCliSave
from CliMode.Radius import RadiusServerGroupMode
from IpLibConsts import DEFAULT_VRF
from CliSavePlugin.NetworkCliSave import networkConfigCmdSeq

#-------------------------------------------------------------------------------
# Object used for saving commands in "config-sg-radius" mode.
#-------------------------------------------------------------------------------
class RadiusServerGroupConfigMode( RadiusServerGroupMode, CliSave.Mode ):
   def __cmp__( self, other ):
      return cmp( self.param_, other.param_)
   def __init__( self, param ):
      RadiusServerGroupMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( RadiusServerGroupConfigMode,
                                       before=[ 'Aaa.global' ] )
RadiusServerGroupConfigMode.addCommandSequence( 'Radius.serverGroup' )

# I want the Radius commands to go after Network.config because the
# "radius-server host" command takes hostname-or-ip-addrs and attempt to
# resolve hostnames, so it's better if the nameserver config is done first.
CliSave.GlobalConfigMode.addCommandSequence( 'Radius.global',
                                             before=[ 'Aaa.global',
                                                      RadiusServerGroupConfigMode ],
                                             after=[ networkConfigCmdSeq ] )

CliSave.GlobalConfigMode.addCommandSequence( 'Radius.srcIntf', 
   after=[ IntfCliSave.IntfConfigMode ] )

# When someone types "show running-config", the Cli code walks the sysdb object
# tree, calling all the functions which registered with CliSave.saver to be
# called for a particular object. Below, I register saveRadiusConfig to be
# called for 'security/aaa/radius/config'. When I'm called, I walk our entire
# Radius config object tree, generating all non-default config.

@CliSave.saver( 'Aaa::HostGroup', 'security/aaa/config' )
def saveHostGroup( entity, root, sysdbRoot, options ):
   saveAll = options.saveAll
   if entity.groupType != 'radius':
      return
   mode = root[ RadiusServerGroupConfigMode ].getOrCreateModeInstance( entity.name )
   cmds = mode[ 'Radius.serverGroup' ]
   for m in entity.member.itervalues():
      cmd = "server %s" % ( m.spec.hostname )
      if m.spec.vrf != DEFAULT_VRF:
         cmd += " vrf %s" % m.spec.vrf 
      if m.spec.port != RadiusConstants.defaultPort or saveAll:
         cmd += " auth-port %d" % ( m.spec.port ) 
      if m.spec.acctPort != RadiusConstants.defaultAcctPort or saveAll:
         cmd += " acct-port %d" % ( m.spec.acctPort ) 
      cmds.addCommand( cmd )

@CliSave.saver( 'Radius::Config', 'security/aaa/radius/config' )
def saveRadiusConfig( cfg, root, sysdbRoot, options ):

   cmds = root[ 'Radius.global' ]
   saveAll = options.saveAll

   if cfg.key != "":
      cmd = 'radius-server key 7 %s' % ( CliSave.sanitizedOutput( options, 
                                                                  cfg.key ) )
      cmds.addCommand( cmd )
   elif saveAll:
      cmds.addCommand( 'no radius-server key' )

   if cfg.timeout != cfg.defaultTimeout or saveAll:
      cmd ='radius-server timeout %d' % ( int( cfg.timeout ) )
      cmds.addCommand( cmd )

   if cfg.retries != cfg.defaultRetries or saveAll:
      cmd ='radius-server retransmit %d' % ( int( cfg.retries ) )
      cmds.addCommand( cmd )

   if cfg.deadtime != 0:
      cmd ='radius-server deadtime %d' % ( int( cfg.deadtime / 60 ) )
      cmds.addCommand( cmd )
   elif saveAll:
      cmd = 'no radius-server deadtime'
      cmds.addCommand( cmd )

   if cfg.nasId != "":
      cmd = 'radius-server attribute 32 include-in-access-req format %s' % cfg.nasId
      cmds.addCommand( cmd )
   elif cfg.nasIdType != "disabled":
      cmd = 'radius-server attribute 32 include-in-access-req %s' % cfg.nasIdType
      cmds.addCommand( cmd )
   elif saveAll:
      cmds.addCommand( 'no radius-server attribute 32 include-in-access-req' )

   if cfg.dynAuthPort != cfg.dynAuthPortDefault:
      cmd ='radius-server dynamic-authorization port %d' % ( cfg.dynAuthPort )
      cmds.addCommand( cmd )
   elif saveAll:
      cmd = 'no radius-server dynamic-authorization port'
      cmds.addCommand( cmd )

   hosts = cfg.host
   for h in sorted( hosts.values(), key=lambda host: host.index ):
      cmd = "radius-server host " + h.hostname
      if h.vrf != DEFAULT_VRF:
         cmd += ' vrf %s' % h.vrf
      if h.port != RadiusConstants.defaultPort or saveAll:
         cmd += " auth-port %d" % ( h.port )
      if h.acctPort != RadiusConstants.defaultAcctPort or saveAll:
         cmd += " acct-port %d" % ( h.acctPort )
      if h.useTimeout:
         cmd += " timeout %d" % ( h.timeout )
      if h.useRetries:
         cmd += " retransmit %d" % ( h.retries )
      if h.useKey:
         cmd += " key 7 %s" % ( CliSave.sanitizedOutput( options, h.key ) )
      cmds.addCommand( cmd )  

   if cfg.srcIntfName:
      for vrf, srcIntf in cfg.srcIntfName.iteritems():
         vrfString = ' vrf %s' % vrf if vrf != DEFAULT_VRF else '' 
         root[ 'Radius.srcIntf' ].addCommand( 'ip radius%s source-interface %s' % \
                                                 ( vrfString, srcIntf ) )
   elif saveAll:
      root[ 'Radius.srcIntf' ].addCommand( 'no ip radius source-interface' )

   # radius-server qos dscp <dscpValue>
   if cfg.dscpValue != cfg.dscpValueDefault:
      cmd ='radius-server qos dscp %s' % cfg.dscpValue
      cmds.addCommand( cmd )
   elif saveAll:
      cmd = 'radius-server qos dscp %s' % cfg.dscpValueDefault
      cmds.addCommand( cmd )
