# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import sys
import CliSave
from AclCliSave import CpConfigMode
import QosLib
from QosLib import classNameFromCpStaticType, coppStaticClassesByPrio, coppMapType, \
      coppStaticClassFromHwStatus
from QosTypes import tacClassMapCpType, tacActionType, \
    tacClassMapCpStaticType, tacCMapNm, tacPMapNm, tacActionRateType
import QosCliSave
import Cell

@CliSave.saver( 'Qos::Input::Config', 'qos/input/config/cli',
                requireMounts = ( 'qos/hardware/status/global',
                                   Cell.path( 'qos/hardware/status/slice' ),
                                  'qos/profile', 'dcb/pfc/status',
                                  'hwEpoch/status',
                                  'interface/config/eth/phy/all',
                                  'interface/config/all', 'interface/status/all' ) )
def saveConfig( entity, root, sysdbRoot, options,
                requireMounts ):
   saveAll = options.saveAll
   hwStatus = requireMounts[ 'qos/hardware/status/global' ]
   hwEpochStatus = requireMounts[ 'hwEpoch/status' ]
   def addCommandCoppServicePolicy():
      key = Tac.newInstance( "Qos::ServicePolicyKey", coppMapType, 'input',
                             tacPMapNm.coppName )
      addCoppSpCommand = False
      addNoCoppSpCommand = False
      if key in entity.servicePolicyConfig:
         # Add the 'service-policy input copp-system-policy' command under the
         # 'control-plane' mode only if the key is found in servicePolicyConfig,
         # and in the following cases:
         # - saveAll=True, i.e, 'show running-config all' or
         # - The image doesn't have CoPP enabled by default, which means someone
         #   explicitly configured the servicePolicyConfig of copp-system-policy
         #   (of course, this is possible only if hwStatus.coppSupported is True).
         # We do not add the 'service-policy input copp-system-policy' command if
         # the image has CoPP enabled by default and saveAll=False.
         if saveAll or not hwEpochStatus.globalProtectionModeEnabled:
            addCoppSpCommand = True
      else:
         # Add the 'no service-policy input copp-system-policy' command under the
         # 'control-plane' mode only if the key is not found in servicePolicyConfig,
         # and the image has CoPP enabled by default, which means someone explicitly
         # removed the servicePolicyConfig of copp-system-policy.
         if hwEpochStatus.globalProtectionModeEnabled:
            addNoCoppSpCommand = True
      if not addCoppSpCommand and not addNoCoppSpCommand:
         return
      cpMode = root[ CpConfigMode ].getSingletonInstance()
      cmds = cpMode[ 'Qos.cp' ]
      if addCoppSpCommand:
         cmds.addCommand( 'service-policy input %s' % tacPMapNm.coppName )
      elif addNoCoppSpCommand:
         cmds.addCommand( 'no service-policy input %s' % tacPMapNm.coppName )

   if hwStatus.coppSupported:
      # Do not generate control-plane service policy commands if coppSupported
      # is false.
      addCommandCoppServicePolicy()

def addCommandCoppPMap( root, entity, hwStatus, sliceHwStatus,
                        defaultPdpPmapCfg, mapType, pmapName, saveAll ):
   if not( hwStatus.coppSupported or hwStatus.intfCoppSupported ):
      return

   pmap = entity.pmapType[ mapType ].pmap[ pmapName ]
   shared = pmap.shared

   staticClassesEdited = []
   def coppStaticClassesEdited():
      # Find out if any of the static classes is edited.
      # If any of them is, set return True.
      classEdited = False
      hwCoppStaticClass = coppStaticClassFromHwStatus( sliceHwStatus )
      if not hwCoppStaticClass:
         return
      for cmapCpStaticType in hwCoppStaticClass:
         coppStaticClass = hwCoppStaticClass[ cmapCpStaticType ]
         cmapName = QosLib.classNameFromCpStaticType( cmapCpStaticType )
         if cmapName == tacCMapNm.coppDefault:
            policyAction = pmap.classActionDefault.policyAction
         else:
            if not cmapName in pmap.classAction.keys():
               return False
            policyAction = pmap.classAction[ cmapName ].policyAction
         shape = policyAction[ tacActionType.actionSetShape ].rate
         bw = policyAction[ tacActionType.actionSetBandwidth ].rate
         if not shape.val == bw.val == tacActionRateType.invalid:
            if shape.val != coppStaticClass.defaultMax or \
                   bw.val != coppStaticClass.defaultMin:
               staticClassesEdited.append( cmapName )
               classEdited = True
      return classEdited

   if mapType == coppMapType and hwStatus.intfCoppSupported and \
          pmapName == tacPMapNm.coppName:
      # skip copp-system-policy for intf copp
      return

   if mapType == coppMapType and not hwStatus.intfCoppSupported:
      if not( saveAll or len( pmap.classPrio ) or coppStaticClassesEdited() ):
         return # don't display copp policy-map unless required

   pmapMode = root[ QosCliSave.PolicyMapConfigMode ].getOrCreateModeInstance( \
      ( mapType, pmapName, shared ) )

   def addCommandCoppPMapClass( cmapName, cmapCpStatic,
                                classPrio, isStatic=True ):
      actions = None
      if QosLib.isDefaultClass( mapType, cmapName ) == True:
         if pmap.classActionDefault != None:
            actions = pmap.classActionDefault.policyAction
      else:
         if cmapName in pmap.classAction.keys():
            actions = pmap.classAction[ cmapName ].policyAction

      if ( actions == None and hwStatus.coppActionShaperSupported ) or \
         cmapCpStatic == tacClassMapCpStaticType.cmapCpStaticDrop:
         return

      # The ordering for static class-maps is in reverse order
      # as compared to dynamic class-maps. Hence, they have been
      # assigned different signs
      # For per port CoPP, ie pmapName not equal to
      # copp-system-policy , the prio is not reversed.
      if isStatic:
         if pmapName == tacPMapNm.coppName:
            classPrio = ( 'static', -( classPrio + 1 ) )
         else:
            classPrio = ( 'static', classPrio )
      else:
         classPrio = ( 'dynamic', classPrio )
      configPmapClassMode = pmapMode[ QosCliSave.PolicyMapClassConfigMode ].\
                            getOrCreateModeInstance( ( 
                               mapType, 
                               pmapName,
                               cmapName,
                               classPrio ) )
      cmds = configPmapClassMode[ 'Qos.pmapc' ]

      def addCommandCoppPMapClassAction():
         if actions == None:
            return

         def addCommandCoppAction( actions, actionType ):
            def addCommandCoppActionBandwidth( bandwidth ):
               minPps = bandwidth.val
               rateUnit = None
               if minPps == tacActionRateType.invalid:
                  if not saveAll:
                     return
                  hwCoppStaticClass = \
                        coppStaticClassFromHwStatus( sliceHwStatus )
                  if not hwCoppStaticClass:
                     return
                  staticClass = hwCoppStaticClass[ cmapCpStatic ]
                  minPps = staticClass.defaultMin
                  rateUnit = staticClass.defaultRateUnit
               if minPps == tacActionRateType.noValue:
                  if not saveAll:
                     if coppStaticClassesEdited() and isStatic:
                        cmds.addCommand( 'no bandwidth' )
                  else:
                     cmds.addCommand( 'no bandwidth' )
               else:
                  if rateUnit == None:
                     rateUnit = bandwidth.rateUnit

                  rateUnit = QosLib.rateUnitFromEnum( rateUnit )
                  cmds.addCommand( 'bandwidth %s %d' % \
                                      ( rateUnit, minPps ) )

            def addCommandCoppActionShape( shape ):
               maxPps = shape.val
               rateUnit = None
               if maxPps == tacActionRateType.invalid:
                  if not saveAll:
                     return
                  hwCoppStaticClass = \
                        coppStaticClassFromHwStatus( sliceHwStatus )
                  if not hwCoppStaticClass:
                     return
                  staticClass = hwCoppStaticClass[ cmapCpStatic ]
                  maxPps = staticClass.defaultMax
                  rateUnit = staticClass.defaultRateUnit
               if maxPps == tacActionRateType.noValue:
                  if not saveAll:
                     if coppStaticClassesEdited() and isStatic:
                        cmds.addCommand( 'no shape' )
                  else:
                     cmds.addCommand( 'no shape' )
               else:
                  if rateUnit == None:
                     rateUnit = shape.rateUnit
                  rateUnit = QosLib.rateUnitFromEnum( rateUnit )
                  cmds.addCommand( 'shape %s %d' % \
                                      ( rateUnit, maxPps ) )

            if actionType == tacActionType.actionSetShape:
               rate = actions[ actionType ].rate
               addCommandCoppActionShape( rate )
            elif actionType == tacActionType.actionSetBandwidth:
               rate = actions[ actionType ].rate
               addCommandCoppActionBandwidth( rate )

         if hwStatus.coppActionShaperSupported:
            if actions.__len__() == 0:
               cmds.addCommand( '   no shape' )
               cmds.addCommand( '   no bandwidth' )
            else:
               for actionType in actions:
                  addCommandCoppAction( actions, actionType )

         if hwStatus.coppActionPolicerSupported:
            policer, cirUnitStr, bcUnitStr = None, None, None
            if cmapName in pmap.classAction.keys() and \
                pmap.classAction[ cmapName ].policer != None:
               policer = pmap.classAction[ cmapName ].policer
               cirUnitStr = QosLib.rateUnitFromEnum( policer.cirUnit )
               bcUnitStr = QosLib.burstUnitFromEnum( policer.bcUnit )
            elif QosLib.isDefaultClass( mapType, cmapName ) == True and \
                  pmap.classActionDefault.policer != None:
               policer = pmap.classActionDefault.policer
               cirUnitStr = QosLib.rateUnitFromEnum( policer.cirUnit )
               bcUnitStr = QosLib.burstUnitFromEnum( policer.bcUnit )
            if policer:
               assert cirUnitStr != None and bcUnitStr != None
               if policer.cmdVersion == 1:
                  cmds.addCommand( 'police cir %d %s bc %d %s' % (
                     policer.cir, cirUnitStr, policer.bc, bcUnitStr ) )
               else:
                  cmds.addCommand( 'police rate %d %s burst-size '
                                    '%d %s' % (
                     policer.cir, cirUnitStr, policer.bc, bcUnitStr ) )
      addCommandCoppPMapClassAction()

   def addCommandCoppPMapClassDynamic():
      prio = pmap.classPrio
      indices = prio.keys()
      indices.sort()
      for index in indices:
         cmapName = prio[ index ].cmapName
         addCommandCoppPMapClass(
            cmapName, tacClassMapCpStaticType.cmapCpStaticInvalid,
            index, False )

   def addCommandCoppPMapClassStatic():
      prio = pmap.coppStaticClassPrio
      indices = prio.keys()
      indices.sort()
      prioAndTypes = coppStaticClassesByPrio( sliceHwStatus )
      if pmapName == tacPMapNm.coppName:
         for classPrioAndType in prioAndTypes:
            classPrio = classPrioAndType[ 0 ]
            cmapCpStatic = classPrioAndType[ 1 ]
            cmapName = classNameFromCpStaticType( cmapCpStatic )
            if saveAll or ( cmapName in staticClassesEdited ):
               addCommandCoppPMapClass( cmapName, cmapCpStatic,
                                        classPrio, True )
      else:
         # In addCommandCoppPMapClass, arg classPrio
         # determines the order of displaying.
         for classPrio in pmap.coppStaticClassPrio:
            cmapCpStatic = classPrio
            cmapName = pmap.coppStaticClassPrio[ classPrio ].cmapName
            if saveAll or hwStatus.intfCoppSupported or \
                   ( cmapName in staticClassesEdited ):
               addCommandCoppPMapClass( cmapName, cmapCpStatic,
                                        classPrio, True )

         # Default class
         if pmap.classActionDefault.policer != None or \
               pmap.classActionDefault.policyAction.keys():
            cmapName = pmap.classDefault.name
            cmapCpStatic = tacClassMapCpStaticType.cmapCpStaticDefault
            if saveAll or hwStatus.intfCoppSupported or \
                   ( cmapName in staticClassesEdited ):
               addCommandCoppPMapClass( cmapName, cmapCpStatic,
                                        sys.maxint, True )

   if ( saveAll or ( len( pmap.classPrio ) and
                     ( hwStatus.intfCoppSupported or
                       coppStaticClassesEdited() ) ) ):
      addCommandCoppPMapClassDynamic()
      addCommandCoppPMapClassStatic()
   else:
      if len( pmap.classPrio ):
         addCommandCoppPMapClassDynamic()
      else:
         addCommandCoppPMapClassStatic()

def addCommandMapTypeCopp( hwStatus, sliceHwStatus, aclHwStatus, hwEpochStatus ):
   return hwStatus.coppSupported or hwStatus.intfCoppSupported

def addCommandCoppCMap( cmap ):
   if cmap.cpType == tacClassMapCpType.cmapCpStatic:
      return False # static class-maps not displayed
   return True

# Setup hooks in QosCliSave.py
# pylint: disable-msg=protected-access
QosCliSave._servicePolicyNoDirectionStr[ coppMapType ] = True
QosCliSave._addCommandMapTypeCallback[ coppMapType ] = addCommandMapTypeCopp
QosCliSave._addCommandCMapCallback[ coppMapType ] = addCommandCoppCMap
QosCliSave._addCommandPMapCallback[ coppMapType ] = addCommandCoppPMap
# pylint: enable-msg=protected-access
