# Copyright (c) 2006-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements saving the PVLAN CLI
#-------------------------------------------------------------------------------
import Tac, CliSave
import Vlan, Tracing
from IntfCliSave import IntfConfigMode
from EbraCliSave import VlanConfigMode
import EthIntfUtil
from EbraLib import privateVlanAllowed

traceDetail = Tracing.trace2

#-------------------------------------------------------------------------------
# Saves the state of an Interface::VlanIntfConfig object.
#-------------------------------------------------------------------------------
# pylint: disable-msg=C0322
# XXX Workaround for a pylint bug with decorators.
@CliSave.saver( 'Interface::VlanIntfConfigDir', 'interface/config/eth/vlan',
                requireMounts=( 'bridging/input/config/cli',
                                'interface/status/all',
                                'hwEpoch/status' ) )
def savePvlanIntfConfig( entity, root, sysdbRoot, options,
                         requireMounts ):
   if not privateVlanAllowed( requireMounts[ 'hwEpoch/status' ] ):
      return
   for intfConfig in entity.intfConfig.itervalues():
      if intfConfig.source != "cli":
         # don't save dynamic SVIs
         continue
      if options.intfFilter and intfConfig.intfId not in options.intfFilter:
         continue
      saveAll = options.saveAll
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intfConfig.intfId )
      cmds = mode[ 'Ebra.vlanIntf' ]

      bridgingInputCli = requireMounts[ 'bridging/input/config/cli' ]
      vlanId = int( intfConfig.intfId[ 4: ] )
      sviMappingState = bridgingInputCli.sviMapping.get( vlanId )
      if sviMappingState:
         secVlans = Vlan.vlanSetToCanonicalString(
            sviMappingState.secondaryVlan )
         if secVlans == '':
            cmds.addCommand( 'pvlan mapping remove 1-4094' )
         elif secVlans == '1-4094':
            if saveAll:
               cmds.addCommand( 'no pvlan mapping' )
         else:
            cmds.addCommand( 'pvlan mapping %s' % secVlans )
      elif saveAll:
         cmds.addCommand( 'no pvlan mapping' )

#-------------------------------------------------------------------------------
# Given a vlanConfig obj, returns the cmds for the vlan
#-------------------------------------------------------------------------------
 
def getVlanConfig( entity, saveAll, pvlanAllowed ):
   cmds = []
   if pvlanAllowed:
      if entity.vlanTypeAndPrimaryVlan.primaryVlan != 0:
         primaryVlan = entity.vlanTypeAndPrimaryVlan.primaryVlan
         vlanType = entity.vlanTypeAndPrimaryVlan.vlanType
         cmds.append( 'private-vlan %s primary vlan %s' % ( vlanType, primaryVlan ) )
      elif saveAll:
         cmds.append( 'no private-vlan' )
      
   return cmds

#-------------------------------------------------------------------------------
# Saves the config for a list of vlans
#-------------------------------------------------------------------------------

def saveVlanListConfig( root, vlanList, vlanEnabled, cmdList ):
   # The first vlan in the vlan list is included as the first parameter
   # to ensure that the vlan lists are sorted numerically
   vlanString = Vlan.vlanSetToCanonicalString( vlanList )
   mode = root[ VlanConfigMode ].getOrCreateModeInstance( ( vlanList[0],
                                                            vlanString,
                                                            vlanEnabled ) )
   cmds = mode[ 'Ebra.vlan' ]
   for cmd in cmdList:
      cmds.addCommand( cmd )

#-------------------------------------------------------------------------------
# Saves the config of Bridging::VlanConfig objects
#-------------------------------------------------------------------------------
 
def saveVlanConfig( entity, root, sysdbRoot, options,
                    requireMounts ):
   # Save the VLAN configuration.  Note that VLANs that are in 'inactive' state
   # are not explicitly saved as all the configuration for such VLANs must be at
   # their default values.
   # There are a couple of exceptions to deal with for vlan 1.  It is created
   # by default, so we don't save the 'vlan 1' command unless the name or
   # state attributes are non-default.  
   #
   # This function iterates through the vlan objects, and outputs vlan range
   # commands, when a list of vlans have the same list of commands.
   # If multiple vlan ranges have the same set of commands, then vlan ranges are
   # displayed in the same line, separated by commas. However, if there are one or
   # more vlans in between the vlan ranges, with a different set of commands,
   # then the corresponding vlan ranges are not combined into a single line.
   if options.intfFilter:
      return

   # vlan 1 is handled as a special case, before the for loop
   vlanEntity = entity.vlanConfig.get( 1 )
   traceDetail( "Processing vlan 1" )
   pvlanAllowed = privateVlanAllowed( requireMounts[ 'hwEpoch/status' ] )
   if vlanEntity:
      cmds = getVlanConfig( vlanEntity, options.saveAll, pvlanAllowed )
      if ( cmds ):
         saveVlanListConfig( root, [ 1 ], True, cmds )

   for v in sorted( entity.vlanConfig, key=int ):
      if v == 1:
         continue
      
      traceDetail( "Processing vlan", v )
      vlanEntity = entity.vlanConfig[ v ]
      cmds = getVlanConfig( vlanEntity, options.saveAll, pvlanAllowed )

      traceDetail( "vlan cfg", cmds )
      saveVlanListConfig( root, [v], True, cmds )

#-------------------------------------------------------------------------------
# Saves the state of global Bridging Config.
#-------------------------------------------------------------------------------
# pylint: disable-msg=C0322
# XXX Workaround for a pylint bug with decorators.
@CliSave.saver( 'Bridging::Input::CliConfig', 'bridging/input/config/cli',
                requireMounts = ( 'cli/config',
                                  'bridging/config',
                                  'bridging/hwcapabilities',
                                  'hwEpoch/status' ) )
def savePvlanBridgingConfig( entity, root, sysdbRoot, options,
                             requireMounts ):
   saveVlanConfig( entity, root, sysdbRoot, options,
                   requireMounts )

   # Display interface configs
   bridgingConfig = requireMounts[ 'bridging/config' ]
   if options.saveAll or bridgingConfig.defaultSwitchportMode == 'routed':
      # Get all eligible switchports, since '[no] switchport' is always displayed.
      cfgIntfNames = EthIntfUtil.allSwitchportNames( bridgingConfig,
                                                     includeEligible=True )
   else:
      cfgIntfNames = entity.switchIntfConfig
   bridgingCliConfig = entity
   for intfName in cfgIntfNames:
      if options.intfFilter and intfName not in options.intfFilter:
         continue
      intfConfig = entity.switchIntfConfig.get( intfName )
      if not intfConfig:
         if options.saveAll or bridgingConfig.defaultSwitchportMode == 'routed':
            intfConfig = Tac.newInstance( 'Bridging::Input::SwitchIntfConfig', 
                                          intfName, 'access' )
            if bridgingConfig.defaultSwitchportMode == 'routed':
               intfConfig.enabled = False
         else:
            continue  
      saveSwitchIntfConfig( intfConfig, bridgingConfig, root, sysdbRoot, options,
                            privateVlanAllowed( requireMounts[ 'hwEpoch/status' ] ),
                            bridgingCliConfig )

#-------------------------------------------------------------------------------
# Saves the state of an Bridging::SwitchIntfConfig object.
#-------------------------------------------------------------------------------
def saveSwitchIntfConfig( entity, bridgingConfig, root, sysdbRoot, options,
                          pvlanAllowed, bridgingCliConfig ):
   mode = root[ IntfConfigMode ].getOrCreateModeInstance( entity.intfId )
   cmds = mode[ 'Ebra.switchport' ]
   saveAll = options.saveAll
   saveAllDetail = options.saveAllDetail

   if '.' in entity.intfId:
      # Sub interface does not support all the following switchport configs.
      return

   # Display defaults on all ports for saveAllDetail and on L2 ports for saveAll. 
   saveDefault = saveAllDetail or ( saveAll and entity.enabled )

   if pvlanAllowed:
      if entity.trunkPrivateVlanMapping == 1:
         cmds.addCommand( 'switchport trunk private-vlan secondary' )
      elif saveDefault:
         cmds.addCommand( 'no switchport trunk private-vlan secondary' )

      if not entity.defaultPrivateVlanMapping:
         mappedPVlans = Vlan.vlanSetToCanonicalString( entity.privateVlanMapping )
         if mappedPVlans == '':
            cmds.addCommand( 'switchport pvlan mapping remove 1-4094' )
         elif mappedPVlans == '1-4094':
            if saveDefault:
               cmds.addCommand( 'no switchport pvlan mapping' )
         else:
            cmds.addCommand( 'switchport pvlan mapping %s'
                             % mappedPVlans )

      elif saveDefault:
         cmds.addCommand( 'no switchport pvlan mapping' )
