#!/usr/bin/env python
# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave
import EthIntfUtil
import Tac
import Tracing
from IntfCliSave import IntfConfigMode
from IpLibConsts import DEFAULT_VRF
from MultiRangeRule import multiRangeToCanonicalString

from CliMode.PtpUnicastNegotiation import PtpUnicastNegotiationMode

__defaultTraceHandle__ = Tracing.Handle( 'PtpCli' )
PtpMode = Tac.Type( 'Ptp::PtpMode' )
CliConstants = Tac.Type( "Ptp::Constants" )
mtn = Tac.Type( "Ptp::MessageTypeNumber" )

class UnicastNegotiationConfigMode( PtpUnicastNegotiationMode, CliSave.Mode ):
   def __init__( self, param ):
      PtpUnicastNegotiationMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( UnicastNegotiationConfigMode,
                                       before=[ IntfConfigMode ] )
UnicastNegotiationConfigMode.addCommandSequence( 'Ptp.unicastNegotiation' )
CliSave.GlobalConfigMode.addCommandSequence( 'Ptp.global',
                                             before=[ IntfConfigMode ] )
IntfConfigMode.addCommandSequence( 'Ptp.intf' )

# Mapping of enum names to CLI names
ptpModes = { 'ptpDisabled' : 'disabled',
             'ptpBoundaryClock' : 'boundary',
             'ptpEndToEndTransparentClock' : 'e2etransparent',
             'ptpOneStepEndToEndTransparentClock' : 'e2etransparent one-step',
             'ptpPeerToPeerTransparentClock' : 'p2ptransparent',
             'ptpOrdinaryMasterClock' : 'ordinarymaster',
             'ptpGeneralized' : 'gptp' }
# using None because we're setting it with [no|default] ptp profile
# There is no specific keyword to specify the default profile
ptpProfiles = { 'ptpDefaultProfile' : None,
                'ptpG8275_2' : 'g8275.2',
                'ptpG8275_1' : 'g8275.1',
              }

delayMechanisms = { 'e2e' : 'e2e',
                    'p2p' : 'p2p' }

transportModes = { 'layer2' : 'layer2',
                   'ipv4' : 'ipv4',
                   'ipv6' : 'ipv6' }

roles = { 'dynamic' : 'dynamic',
          'master' : 'master' }

EthAddr = Tac.Type( "Arnet::EthAddr" )
G82751MacAddrs = { EthAddr.ethAddrPtp : 'forwardable',
                   EthAddr.ethAddrPtpPeerDelay : 'non-forwardable' }

# Mapping of non-enum global attribute names to CLI names
globalAttrs = { 'ptpMode' : ( 'mode', ptpModes ),
                'ptpProfile' : ( 'profile', ptpProfiles ),
                'priority1' : 'priority1',
                'priority2' : 'priority2',
                'localPriority': 'local-priority',
                'clockIdentity' : 'clock-identity',
                'clockQuality' : 'clock-accuracy',
                'domainNumber' : 'domain',
                'srcIp4' : 'source ip',
                'srcIp6' : 'source ipv6',
                'ttl' : 'ttl',
                'holdPtpTimeInterval' : 'hold-ptp-time',
                'globalDscpEvent' : 'message-type event dscp',
                'globalDscpGeneral' : 'message-type general dscp' }

# Mapping of interface attribute names to CLI names
intfAttrs = { 'logAnnounceInterval' : 'announce interval',
              'logMinDelayReqInterval' : 'delay-req interval',
              'logMinPdelayReqInterval' : 'pdelay-req interval',
              'linkDelayThreshold' : 'pdelay-neighbor-threshold',
              ( 'logSyncInterval', 'syncIntervalUseDeprecatedCmd', True ) :
                 'sync interval',
              ( 'logSyncInterval', 'syncIntervalUseDeprecatedCmd', False ) :
                 'sync-message interval',
              'syncReceiptTimeout' : 'sync timeout',
              'announceReceiptTimeout' : 'announce timeout',
              'delayMechanism' : ( 'delay-mechanism', delayMechanisms ),
              'transportMode' : ( 'transport', transportModes ),
              'role' : ( 'role', roles ),
              'dscpEvent' : 'message-type event dscp',
              'dscpGeneral' : 'message-type general dscp',
              'localPriority': 'local-priority',
              'pktTxMacAddress' : ( 'profile g8275.1 destination mac-address',
                 G82751MacAddrs ),
              'vlanConfig' : 'vlan' }

intfHasGlobalCmd = { 'dscpEvent' : 'globalDscpEvent',
                     'dscpGeneral' : 'globalDscpGeneral' }

UcastNegAttrToCliToken = {
      'announceProfile': 'announce',
      'syncProfile': 'sync',
      'delayRespProfile': 'delay-resp',
      'duration': 'duration',
      'logInterval': 'interval'
}
@CliSave.saver( 'Ptp::Config', 'ptp/config', requireMounts = ( 'ptp/status',
                                                               'bridging/config' ) )
def savePtpConfig( entity, root, sysdbRoot, options, requireMounts ):

   cmds = root[ 'Ptp.global' ]
   saveAll = options.saveAll
   saveAllDetail = options.saveAllDetail

   ptpStatus = requireMounts[ 'ptp/status' ]
   bridgingConfig = requireMounts[ 'bridging/config' ]

   if not ptpStatus.ptpSupported:
      saveAll = False
      saveAllDetail = False

   for tacAttr, cliAttr in globalAttrs.items():
      if isinstance( cliAttr, tuple ):
         cliAttr, attrValues = cliAttr
      else:
         attrValues = None
      currentValue = getattr( entity, tacAttr )
      if tacAttr == "clockIdentity":
         currentValue = [ getattr( currentValue, 'v%d' % i ) for i in range( 8 ) ]
         currentValue = ':'.join( [ '%02x' % x for x in currentValue ] )
         if getattr( entity, tacAttr + "Configured" ):
            defaultValue = 'x'
         else:
            defaultValue = currentValue
      elif tacAttr == "clockQuality":
         # only for clock accuracy for now
         if cliAttr == "clock-accuracy":
            currentValue = getattr( currentValue, "clockAccuracy" )
            if getattr( entity, tacAttr + "Configured" ):
               defaultValue = getattr( CliConstants, "clockAccuracyDefault" )
            else:
               defaultValue = currentValue
         else:
            continue
      elif tacAttr.startswith( 'srcIp' ) and entity.unicastNegotiation:
         continue
      else:
         defaultValue = getattr( CliConstants, tacAttr + "Default" )
      if ( currentValue != defaultValue ) or saveAllDetail or \
         ( saveAll and entity.ptpMode != 'ptpDisabled' ):
         if attrValues:
            currentValue = attrValues[ currentValue ]
         # At the moment, we only have dscp commands that are both per interface
         # and global. Need to special case these commands as the value is in
         # the middle of the command.
         if tacAttr.startswith( 'global' ):
            cmds.addCommand( 'ptp %s %s default' % ( cliAttr, str( currentValue ) ) )
         elif currentValue is None:
            cmds.addCommand( 'no ptp %s' % cliAttr )
         else:
            cmds.addCommand( 'ptp %s %s' % ( cliAttr, str( currentValue ) ) )

   key = Tac.Value( "Acl::AclTypeAndVrfName", 'ip', DEFAULT_VRF )
   aclName = entity.serviceAclTypeVrfMap.aclName.get( key )
   if aclName:
      cmds.addCommand( 'ptp ip access-group %s in' % aclName )

   forceSave = ( saveAll and entity.ptpMode != 'ptpDisabled' ) or saveAllDetail
   if entity.forwardPtpV1:
      cmds.addCommand( 'ptp forward-v1' )
   elif forceSave:
      cmds.addCommand( 'no ptp forward-v1' )
   if entity.forwardUnicast:
      cmds.addCommand( 'ptp forward-unicast' )
   elif forceSave:
      cmds.addCommand( 'no ptp forward-unicast' )
   if entity.managementEnabled:
      cmds.addCommand( 'ptp management all' )
   elif forceSave:
      cmds.addCommand( 'no ptp management all' )
   if entity.netSyncMonitor:
      cmds.addCommand( 'ptp netsync-monitor delay-request' )
   elif forceSave:
      cmds.addCommand( 'no ptp netsync-monitor delay-request' )
   if not entity.monitorEnabled:
      cmds.addCommand( 'no ptp monitor' )
   elif forceSave:
      cmds.addCommand( 'ptp monitor' )
   if ( entity.offsetFromMasterThreshold !=
        CliConstants.invalidOffsetFromMasterThreshold ):
      cmds.addCommand( 'ptp monitor threshold offset-from-master %d' %
                          entity.offsetFromMasterThreshold )
   elif forceSave:
      cmds.addCommand( 'no ptp monitor threshold offset-from-master' )

   if entity.meanPathDelayThreshold != CliConstants.invalidMeanPathDelayThreshold:
      cmds.addCommand( 'ptp monitor threshold mean-path-delay %d' %
                          entity.meanPathDelayThreshold )
   elif forceSave:
      cmds.addCommand( 'no ptp monitor threshold mean-path-delay' )

   if entity.skewThreshold != CliConstants.invalidSkewThreshold:
      cmds.addCommand( 'ptp monitor threshold skew %g' % entity.skewThreshold )
   elif forceSave:
      cmds.addCommand( 'no ptp monitor threshold skew' )
   if entity.skewDropThreshold != CliConstants.invalidSkewDropThreshold:
      cmds.addCommand( 'ptp monitor threshold skew %g drop' %
                       entity.skewDropThreshold )
   elif forceSave:
      cmds.addCommand( 'no ptp monitor threshold skew drop' )
   if entity.meanPathDelayDropThreshold !=\
      CliConstants.invalidMeanPathDelayDropThreshold:
      cmds.addCommand( 'ptp monitor threshold mean-path-delay %d nanoseconds drop' %
                       entity.meanPathDelayDropThreshold )
   elif forceSave:
      cmds.addCommand( 'no ptp monitor threshold mean-path-delay nanoseconds drop' )
   if entity.offsetFromMasterDropThreshold !=\
      CliConstants.invalidOffsetFromMasterDropThreshold:
      cmds.addCommand(
            'ptp monitor threshold offset-from-master %d nanoseconds drop' %
            entity.offsetFromMasterDropThreshold )
   elif forceSave:
      cmds.addCommand( 
         'no ptp monitor threshold offset-from-master nanoseconds drop' )
   if entity.logMissingMessages != CliConstants.logMissingMessagesDefault:
      cmds.addCommand( 'no ptp monitor sequence-id' )
   elif forceSave:
      cmds.addCommand( 'ptp monitor sequence-id' )
   for msgType, msgName in [ ( mtn.messageSync, "sync" ),
                             ( mtn.messageFollowUp, "follow-up" ),
                             ( mtn.messageAnnounce, "announce" ) ]:
      msgTimer = entity.logMissingMessageTimer.get( msgType,
                 CliConstants.logMissingMessagesDefault )
      msgThreshold = entity.missingTimerThresholds.get( msgType,
                     CliConstants.missingMessageThresholdDefault )
      if msgTimer != CliConstants.logMissingMessagesDefault:
         cmds.addCommand(
               'ptp monitor threshold missing-message {} {} intervals'.format(
               msgName, msgThreshold ) )
      elif forceSave:
         cmds.addCommand(
               'no ptp monitor threshold missing-message {} intervals'.format(
               msgName ) )
   for msgType, msgName in [ ( mtn.messageSync, "sync" ),
                             ( mtn.messageFollowUp, "follow-up" ),
                             ( mtn.messageDelayResp, "delay-resp" ),
                             ( mtn.messageAnnounce, "announce" ) ]:
      msgSeqId = entity.logMissingMessageSeqId.get( msgType,
                 CliConstants.logMissingMessagesDefault )
      msgThreshold = entity.missingSequenceThresholds.get( msgType,
                     CliConstants.missingMessageThresholdDefault )
      if msgSeqId != CliConstants.logMissingMessagesDefault:
         cmds.addCommand(
               'ptp monitor threshold missing-message {} {} sequence-ids'.format(
               msgName, msgThreshold ) )
      elif forceSave:
         cmds.addCommand(
               'no ptp monitor threshold missing-message {} sequence-ids'.format(
               msgName ) )
   if forceSave:
      allIntfs = set( EthIntfUtil.allSwitchportNames( bridgingConfig,
                                                      includeEligible=True ) )
      cfgIntfs = allIntfs.union( set ( entity.intfConfig.keys() ) )
   else:
      cfgIntfs = entity.intfConfig

   profiles = entity.ucastNegProfile.keys()
   for profileName in sorted( profiles, key=str.lower ):
      savePtpUcastNegProfile( entity, root, profileName, saveAll )

   for intf in cfgIntfs:
      savePtpIntfConfig( entity, root, intf, saveAll )

def savePtpUcastNegProfile( ptpConfig, root, name, saveAll ):
   profile = ptpConfig.ucastNegProfile[ name ]
   mode = root[ UnicastNegotiationConfigMode ].getOrCreateModeInstance( name )
   cmds = mode[ 'Ptp.unicastNegotiation' ]

   msgProfiles = [ 'announceProfile', 'syncProfile', 'delayRespProfile' ]
   profileAttrs = [ 'logInterval', 'duration' ]

   for msgProfile in msgProfiles:
      for profileAttr in profileAttrs:
         value = getattr( getattr( profile, msgProfile ), profileAttr )
         cmds.addCommand( '{} {} {}'.format( UcastNegAttrToCliToken[ msgProfile ],
                                             UcastNegAttrToCliToken[ profileAttr ],
                                             value ) )

def savePtpIntfConfig( ptpConfig, root, intf, saveAll ):
   if intf in ptpConfig.intfConfig:
      intfConfig = ptpConfig.intfConfig[ intf ]
   else:
      # If creating a new instance, adding default native vlan to preserve the old
      # behaviour on a trunk port
      intfConfig = Tac.newInstance( 'Ptp::IntfConfig', 'Ethernet1', False )
      intfConfig.vlanConfig.newMember( 0, True, True )
      if ptpConfig.ptpMode == PtpMode.ptpGeneralized:
         ptpType = '802Dot1AS'
      else:
         ptpType = '1588V2'
      for attr in [ 'logSyncInterval', 'logAnnounceInterval',
                    'logMinPdelayReqInterval' ]:
         defaultValue = getattr( CliConstants, attr + 'Default' + ptpType )
         setattr( intfConfig, attr, defaultValue )

   mode = None
   intfCmds = None

   if saveAll or intfConfig.enabled:
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intf )
      intfCmds = mode[ 'Ptp.intf' ]

      if intfConfig.enabled:
         intfCmds.addCommand( 'ptp enable' )
      elif saveAll:
         intfCmds.addCommand( 'no ptp enable' )

   for tacAttr, cliAttr in intfAttrs.items():
      if isinstance( tacAttr, tuple ):
         tacAttr, attrName, attrValue = tacAttr
         if getattr( intfConfig, attrName ) != attrValue:
            continue
      if isinstance( cliAttr, tuple ):
         cliAttr, cliAttrValues = cliAttr
      else:
         cliAttrValues = None

      ptpMode = ''
      if tacAttr in [ 'logSyncInterval', 'logAnnounceInterval',
                      'logMinPdelayReqInterval' ]:
         if ptpConfig.ptpMode == PtpMode.ptpGeneralized:
            ptpMode = '802Dot1AS'
         else:
            ptpMode = '1588V2'

      if tacAttr == 'vlanConfig':
         # Default value is 0, added when intfConfig is created, to preserve the
         # old default behaviour ( sending untagged packet on a trunk port )
         defaultValue = '0'
         intfVlanConfig = getattr( intfConfig, tacAttr )
         configuredVlanSet = \
            set( [ vlan.vlanId for vlan in intfVlanConfig.values() ] )
         currentValue = multiRangeToCanonicalString( configuredVlanSet )
         # No PTP VLAN information to add, just skip
         if currentValue == defaultValue:
            continue
      else:
         intfVerCmdConfiged = False
         defaultValue = None
         if tacAttr in intfHasGlobalCmd:
            intfVerCmdConfiged = getattr( intfConfig, tacAttr + "Configured" )
            defaultValue = getattr( CliConstants, intfHasGlobalCmd[ tacAttr ] + \
                                    'Default' + ptpMode )
         else:
            defaultValue = getattr( CliConstants, tacAttr + 'Default' + ptpMode )
         currentValue = getattr( intfConfig, tacAttr )

      # for cmds that have global counterparts, we don't want to save them unless
      # they are configured.
      globConfig = saveAll or intfVerCmdConfiged or ( defaultValue != currentValue )
      if globConfig:
         if mode is None or intfCmds is None:
            mode = root[ IntfConfigMode ].getOrCreateModeInstance( intf )
            intfCmds = mode[ 'Ptp.intf' ]

         if cliAttrValues:
            cliValue = cliAttrValues[ currentValue ]
         else:
            cliValue = str( currentValue )

         intfCmds.addCommand( 'ptp %s %s' % ( cliAttr, cliValue ) )

   # Save Unicast Negotiation grantor/grantee associated profiles
   if mode is None or intfCmds is None:
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intf )
      intfCmds = mode[ 'Ptp.intf' ]
   grantorIpToProfileMap = intfConfig.ucastNegGrantorIpToProfileMap
   granteeIpToProfileMap = intfConfig.ucastNegGranteeIpToProfileMap
   for cliToken, profileMap in [ ( 'candidate-grantor',
                                   grantorIpToProfileMap.items() ),
                                 ( 'remote-grantee',
                                   granteeIpToProfileMap.items() ) ]:
      for ipGenAddr, profileName in profileMap:
         profileToken = 'profile'
         profileNameToken = profileName
         # If we use the default profile, we don't need to specify the 'profile'
         # token in the CLI cmd
         if profileName == 'default':
            profileToken = ''
            profileNameToken = ''
         cmd = "ptp unicast-negotiation {} {} {} {}".format( cliToken,
                                                             ipGenAddr,
                                                             profileToken,
                                                             profileNameToken )
         intfCmds.addCommand( cmd )
