# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

from operator import attrgetter

from ArPyUtils.Decorators import (
   traced,
   tracedFuncName,
)
from BgpLib import getVpwsName
from CliMode.Pseudowire import (
   FxcMode,
   LdpPseudowireMode,
   MplsLdpPseudowiresMode,
   PatchMode,
   PatchPanelMode,
)
import CliSave
from CliSavePlugin.BgpMacVrfConfigCliSave import BgpMacVrfConfigSaveMode
from CliSavePlugin.LdpCliSave import LdpConfigMode
from CliSavePlugin.RoutingBgpCliSave import RouterBgpBaseConfigMode
from PseudowireLib import (
   ConnectorType,
   FlowLabelMode,
)
from RouteMapLib import isAsdotConfigured
import Tracing
from TypeFuture import TacLazyType
from natsort import natsorted

th = Tracing.Handle( "PseudowireCliSave" )
t0 = th.t0
t8 = th.t8

TunnelRibNameIdMap = TacLazyType( "Tunnel::TunnelTable::TunnelRibNameIdMap" )
VidNormalization = TacLazyType( "Pseudowire::Fxc::VidNormalization" )

#----------------------------------------------------------
# Helper config mode classes
#----------------------------------------------------------

class PatchPanelConfigMode( PatchPanelMode, CliSave.Mode ):
   def __init__( self, param ):
      PatchPanelMode.__init__( self )
      CliSave.Mode.__init__( self, param )

class PatchConfigMode( PatchMode, CliSave.Mode ):
   def __init__( self, patchName ):
      PatchMode.__init__( self, patchName )
      CliSave.Mode.__init__( self, patchName )

class FxcConfigMode( FxcMode, CliSave.Mode ):
   def __init__( self, patchName ):
      FxcMode.__init__( self, patchName )
      CliSave.Mode.__init__( self, patchName )

class MplsLdpPseudowiresConfigMode( MplsLdpPseudowiresMode, CliSave.Mode ):
   def __init__( self, param ):
      MplsLdpPseudowiresMode.__init__( self )
      CliSave.Mode.__init__( self, param )

class LdpPseudowireConfigMode( LdpPseudowireMode, CliSave.Mode ):
   def __init__( self, pwName ):
      LdpPseudowireMode.__init__( self, pwName )
      CliSave.Mode.__init__( self, pwName )

LdpConfigMode.addChildMode( MplsLdpPseudowiresConfigMode, after=[ 'Ldp.config' ] )
MplsLdpPseudowiresConfigMode.addCommandSequence( "Pseudowire.mplsLdpPwsConfig",
                                                 before=[ LdpPseudowireConfigMode ] )

MplsLdpPseudowiresConfigMode.addChildMode( LdpPseudowireConfigMode )
LdpPseudowireConfigMode.addCommandSequence( "Pseudowire.ldpPwConfig" )

CliSave.GlobalConfigMode.addChildMode( PatchPanelConfigMode,
                                       after=[ LdpConfigMode ] )
PatchPanelConfigMode.addCommandSequence( "Pseudowire.patchPanelConfig",
                                         before=[ PatchConfigMode, FxcConfigMode ] )

PatchPanelConfigMode.addChildMode( FxcConfigMode )
FxcConfigMode.addCommandSequence( "Pseudowire.fxcConfig" )

PatchPanelConfigMode.addChildMode( PatchConfigMode, after=[ FxcConfigMode ] )
PatchConfigMode.addCommandSequence( "Pseudowire.patchConfig" )

#----------------------------------------------------------
# Patch panel CliSave
#----------------------------------------------------------

def fxcConnectors( patch ):
   """
   Natural sort connectors in patch
   """
   return natsorted( patch.fxcConnector.values(), key=attrgetter( 'name' ) )

def patchConnectors( patch ):
   return sorted( patch.connector.values(), key=attrgetter( 'name' ) )

@tracedFuncName( trace=t8 )
def savePatchConfig( parentMode, pwConfig, options, patchName, funcName=None ):
   patch = pwConfig.patch.get( patchName )
   if not patch:
      return

   if patch.fxc:
      t0( funcName, "save FXC", patchName )
      mode = parentMode[ FxcConfigMode ].getOrCreateModeInstance( patchName )
      cmds = mode[ "Pseudowire.fxcConfig" ]
   else:
      t0( funcName, "save patch", patchName )
      mode = parentMode[ PatchConfigMode ].getOrCreateModeInstance( patchName )
      cmds = mode[ "Pseudowire.patchConfig" ]

   if not patch.enabled:
      cmds.addCommand( "shutdown" )
   elif options.saveAll:
      cmds.addCommand( "no shutdown" )

   if patch.fxc:
      noNormalization = VidNormalization.noNormalization
      singleVid = VidNormalization.singleVid
      doubleVid = VidNormalization.doubleVid
      vidNormCmd = {
         noNormalization: 'no vlan tag normalization' if options.saveAll else None,
         singleVid: 'vlan tag normalization single',
         doubleVid: 'vlan tag normalization double',
      }[ patch.vidNormalization ]
      if vidNormCmd:
         t0( funcName, "vidNormCmd:", vidNormCmd )
         cmds.addCommand( vidNormCmd )

   connectors = fxcConnectors if patch.fxc else patchConnectors

   for cliConnector in connectors( patch ):
      t0( funcName, "save connector", cliConnector )
      if cliConnector.connectorKey.connectorType == ConnectorType.local:
         if cliConnector.connectorKey.isLegacyTagged():
            cmds.addCommand( "connector %s interface %s dot1q vlan %d" %
                             ( cliConnector.name,
                               cliConnector.connectorKey.localIntfId(),
                               cliConnector.connectorKey.legacyDot1qTag() ) )
         else:
            cmds.addCommand( "connector %s interface %s" %
                             ( cliConnector.name,
                               cliConnector.connectorKey.localIntfId() ) )

      elif cliConnector.connectorKey.connectorType == ConnectorType.ldp:
         cmds.addCommand( "connector %s pseudowire ldp %s" % 
                          ( cliConnector.name,
                            cliConnector.connectorKey.name ) )

      elif cliConnector.connectorKey.connectorType == ConnectorType.bgp:
         cmds.addCommand( "connector %s pseudowire bgp vpws %s pseudowire %s" %
                          ( cliConnector.name,
                            cliConnector.connectorKey.name,
                            cliConnector.connectorKey.vpwsName ) )

@traced( trace=t8 )
def defaultPatchConfig( pwConfig ):
   '''Returns true if all attributes related to patch panel have default values.
   This is used to determine whether the patch panel config should be CliSaved.'''
   if pwConfig.patch:
      return False

   if pwConfig.intfReviewDelayRange != pwConfig.intfReviewDelayRangeDefault:
      return False

   return True

@tracedFuncName( trace=t8 )
def savePatchPanelConfig( pwConfig, root, sysdbRoot, options, funcName=None ):
   mode = root[ PatchPanelConfigMode ].getOrCreateModeInstance( None )
   cmds = mode[ "Pseudowire.patchPanelConfig" ]

   if ( pwConfig.intfReviewDelayRange != pwConfig.intfReviewDelayRangeDefault or
         options.saveAll ):
      t0( funcName, "save recovery review delay", pwConfig.intfReviewDelayRange )
      cmds.addCommand( "connector interface recovery review delay %d %d" %
            ( pwConfig.intfReviewDelayRange.min,
              pwConfig.intfReviewDelayRange.max ) )

   for patchName in sorted( pwConfig.patch ):
      mode = root[ PatchPanelConfigMode ].getOrCreateModeInstance( None )
      savePatchConfig( mode, pwConfig, options, patchName )


#----------------------------------------------------------
# MPLS LDP pseudowires CliSave
#----------------------------------------------------------

@traced( trace=t8 )
def saveLdpPseudowireConfig( parentMode, pwConfig, sysdbRoot, options, connKey,
                             hwCapability ):
   saveAll = options.saveAll

   mode = parentMode[ LdpPseudowireConfigMode ].getOrCreateModeInstance( 
                                                               connKey.name )
   cmds = mode[ "Pseudowire.ldpPwConfig" ]

   connector = pwConfig.connector[ connKey ]

   if connector.neighborAddrPresent:
      cmds.addCommand( "neighbor " + str( connector.neighborAddr ) )
   elif saveAll:
      cmds.addCommand( "no neighbor" )

   if connector.pwIdPresent:
      cmds.addCommand( "pseudowire-id %d" % connector.pwId )
   elif saveAll:
      cmds.addCommand( "no pseudowire-id" )

   if connector.mtuPresent:
      cmds.addCommand( 'mtu %d' % connector.mtu )
   elif saveAll:
      cmds.addCommand( 'no mtu' )

   if connector.controlWord:
      cmds.addCommand( 'control-word' )
   elif saveAll:
      cmds.addCommand( 'no control-word' )

   if hwCapability.flowLabelSupported:
      if connector.flowLabelMode == FlowLabelMode.transmit:
         cmds.addCommand( 'label flow transmit' )
      elif connector.flowLabelMode == FlowLabelMode.receive:
         cmds.addCommand( 'label flow receive' )
      elif connector.flowLabelMode == FlowLabelMode.both:
         cmds.addCommand( 'label flow' )
      elif saveAll:
         cmds.addCommand( 'no label flow' )

@traced( trace=t8 )
def defaultMplsLdpPseudowiresConfig( pwConfig ):
   '''Returns if all attributes related to mpls ldp pseudowires have
   default values. This is used to determine whether the mpls ldp
   pseudowire config should be CliSaved.'''
   if pwConfig.mtu != pwConfig.getMtuDefault():
      return False

   if pwConfig.mtuIgnore != pwConfig.getMtuIgnoreDefault():
      return False

   if pwConfig.connector:
      return False

   if pwConfig.tunnelRibName != TunnelRibNameIdMap.systemTunnelRibName:
      return False

   return True

@traced( trace=t8 )
def saveMplsLdpPseudowiresConfig( mode, pwConfig, root, sysdbRoot, options,
                                  hwCapability ):
   saveAll = options.saveAll

   cmds = mode[ "Pseudowire.mplsLdpPwsConfig" ]

   if saveAll or pwConfig.tunnelRibName != TunnelRibNameIdMap.systemTunnelRibName:
      cmds.addCommand( "neighbor resolution tunnel-rib %s"
                       % pwConfig.tunnelRibName )

   if saveAll or pwConfig.mtu != pwConfig.getMtuDefault():
      if pwConfig.mtu == pwConfig.getMtuDefault():
         cmds.addCommand( "no mtu" )
      else:
         cmds.addCommand( "mtu %d" % pwConfig.mtu )

   # TODO: For BUG146565, "no mtu ignore" is not allowed, and "mtu ignore" is
   # the only permissible value
   if saveAll:
      cmds.addCommand( "mtu ignore" )

   ldpConnKeys = [ connKey for connKey in pwConfig.connector
                   if connKey.connectorType == ConnectorType.ldp ]

   ldpConnKeys = sorted( ldpConnKeys )

   for connKey in ldpConnKeys:
      saveLdpPseudowireConfig( mode, pwConfig, sysdbRoot, options, connKey,
                               hwCapability )

@traced( trace=t8 )
@CliSave.saver( "Pseudowire::Config", "pseudowire/config",
                requireMounts=( 'routing/hardware/pseudowire/capability', ) )
def saveConfig( pwConfig, root, sysdbRoot, options, requireMounts ):
   saveAll = options.saveAll

   hwCapability = requireMounts[ 'routing/hardware/pseudowire/capability' ]
   if not defaultMplsLdpPseudowiresConfig( pwConfig ) or saveAll:
      mode = root[ LdpConfigMode ].getOrCreateModeInstance( 'default' )
      mode = mode[ MplsLdpPseudowiresConfigMode ].getOrCreateModeInstance( None )
      saveMplsLdpPseudowiresConfig( mode, pwConfig, root, sysdbRoot, options,
                                    hwCapability )

   if not defaultPatchConfig( pwConfig ) or saveAll:
      savePatchPanelConfig( pwConfig, root, sysdbRoot, options )

@traced( trace=t8 )
def getMockLdpResponseConfigCmd( pwaMockLdpStatus, respKey ):
   resp = pwaMockLdpStatus.response.get( respKey )
   if not resp:
      return ""

   pwId, neighbor = respKey.split( '/' )
   cmd = "debug pseudowire-id %s" % pwId
   cmd += " neighbor %s" % neighbor
   cmd += " peer-label %d" % resp.peerLabel
   cmd += " type %d" % ( 4 if resp.remotePwType == "pwType4" else 5 )
   if resp.mtu:
      cmd += " mtu %d" % resp.mtu
   if resp.controlWord:
      cmd += " control-word"
   if resp.peerFlowLabelMode != FlowLabelMode.disabled:
      cmd += " flow-label %s" % resp.peerFlowLabelMode
   if resp.requestedDot1qTag:
      cmd += " dot1q vlan %d" % resp.requestedDot1qTag
   if resp.groupId:
      cmd += " group-id %d" % resp.groupId

   return cmd

@traced( trace=t8 )
@CliSave.saver( "Pseudowire::PseudowireLdpStatus", "pseudowire/ldp/mockStatus" )
def saveDebugLdpConfig( pwaMockLdpStatus, root, sysdbRoot, options ):
   mockLdpRespKeys = sorted( pwaMockLdpStatus.response.keys() )

   if mockLdpRespKeys:
      mode = root[ LdpConfigMode ].getOrCreateModeInstance( 'default' )
      mode = mode[ MplsLdpPseudowiresConfigMode ].getOrCreateModeInstance( None )

      cmds = mode[ "Pseudowire.mplsLdpPwsConfig" ]

      for respKey in mockLdpRespKeys:
         cmds.addCommand( getMockLdpResponseConfigCmd( pwaMockLdpStatus,
                                                       respKey ) )

@traced( trace=t8 )
def getMockBgpResponseConfigCmd( pwaMockBgpStatus, respKey ):
   resp = pwaMockBgpStatus.bgpResponse.get( respKey )
   if not resp:
      return ""

   pwName = respKey.pwName
   cmd = "debug pseudowire %s" % pwName

   for peerAddr, peerInfo in resp.peerInfo.items():
      cmd += " neighbor %s" % peerAddr
      cmd += " peer-label %d" % peerInfo.label
      # peerInfo currently has only one element
      break

   if resp.mtu or resp.controlWord:
      cmd += " local"
      if resp.mtu:
         cmd += " mtu %d" % resp.mtu
      if resp.controlWord:
         cmd += " control-word"

   for peerInfo in resp.peerInfo.values():
      if peerInfo.mtu or peerInfo.controlWord:
         cmd += " peer"
         if peerInfo.mtu:
            cmd += " mtu %d" % peerInfo.mtu
         if peerInfo.controlWord:
            cmd += " control-word"
      # peerInfo currently has only one element
      break

   return cmd

@traced( trace=t8 )
@CliSave.saver( "Pseudowire::PseudowireBgpStatus", "pseudowire/bgp/mockStatus",
                requireMounts=( "routing/bgp/config",
                                "routing/bgp/asn/config" ) )
def saveDebugBgpConfig( pwaMockBgpStatus, root, sysdbRoot, options, requireMounts ):
   # This must be imported here so it does not interfere with
   # Eos/test/HwEpochVersionCliSanity.py
   bgpConfig = requireMounts[ 'routing/bgp/config' ]
   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]
   if bgpConfig.asNumber == 0:
      return

   asdotConfigured = isAsdotConfigured( asnConfig )

   parentMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
         bgpConfig.asNumber, asdotConfigured ) )

   mockBgpRespKeys = sorted( pwaMockBgpStatus.bgpResponse.keys() )

   for respKey in mockBgpRespKeys:
      vpwsName = getVpwsName( respKey.vpwsEviName )
      mode = parentMode[ BgpMacVrfConfigSaveMode ].getOrCreateModeInstance(
         ( vpwsName, False, False, True ) )
      cmds = mode[ 'Bgp.macvrf.config' ]
      cmds.addCommand( getMockBgpResponseConfigCmd( pwaMockBgpStatus, respKey ) )
