#!/usr/bin/env python
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import Tac
import CliSave
import CliSavePlugin.IntfCliSave as IntfCliSave
from CliSavePlugin.IntfCliSave import IntfConfigMode
from CliSavePlugin.SamplePolicyCliSave import SamplePolicySaveMode, SamplePolicySaver
from CliSavePlugin.TrafficPolicyCliSave import ( MatchRuleSaveMode, ActionsSaveMode )
from CliMode.PostcardTelemetry import ( PostcardTelemetryModeBase,
                                        ProfileModeBase, FEATURE )
import Toggles.PostcardTelemetryCommonToggleLib

class PostcardTelemetryConfigMode( PostcardTelemetryModeBase, CliSave.Mode ):
   def __init__( self, param ):
      PostcardTelemetryModeBase.__init__( self )
      CliSave.Mode.__init__( self, self.longModeKey )

class ProfileConfigMode( ProfileModeBase, CliSave.Mode ):
   def __init__( self, param ):
      ProfileModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, self.longModeKey )

postcardDefaults = Tac.Type( 'PostcardTelemetry::Defaults' )

CliSave.GlobalConfigMode.addCommandSequence( 'PostcardTelemetry.Config' )
IntfCliSave.IntfConfigMode.addCommandSequence( 'PostcardTelemetry.Config' )

CliSave.GlobalConfigMode.addChildMode( PostcardTelemetryConfigMode,
                                       after=[ IntfConfigMode ] )

PostcardTelemetryConfigMode.addCommandSequence( 'PostcardTelemetry.Config' )

PostcardTelemetryConfigMode.addChildMode( SamplePolicySaveMode,
                                          after=[ 'PostcardTelemetry.Config' ] )
SamplePolicySaveMode.addCommandSequence( 'PostcardTelemetry.Policy' )
SamplePolicySaveMode.addChildMode( MatchRuleSaveMode )
MatchRuleSaveMode.addCommandSequence( 'PostcardSamplePolicy.MatchRule' )
ActionsSaveMode.addCommandSequence( 'PostcardSamplePolicy.Actions' )

PostcardTelemetryConfigMode.addChildMode( ProfileConfigMode,
                                          after=[ SamplePolicySaveMode ] )
ProfileConfigMode.addCommandSequence( 'PostcardTelemetry.profile' )

class PostcardSamplePolicySaver( SamplePolicySaver ):
   trafficPoliciesMode = PostcardTelemetryConfigMode

   def __init__( self, entity, root, sysdbRoot, options, requireMounts, feature,
                 commentKey ):
      SamplePolicySaver.__init__( self, entity, root, sysdbRoot, options,
                                  requireMounts, feature, commentKey )
      self.policyMapType = 'postcard-sample-policy'
      self.options = options

   def matchModeCmds( self, matchMode ):
      return matchMode[ 'PostcardSamplePolicy.MatchRule' ]

   def actionModeCmds( self, actionsMode ):
      return actionsMode[ 'PostcardSamplePolicy.Actions' ]

   def saveTrafficPolicy( self, policyName, policiesMode ):
      if policyName == 'default' and not self.options.saveAll:
         return
      self.trafficPolicy = self.entity.pmapType.pmap[ policyName ].currCfg
      self.currPolicyName = policyName
      if not self.trafficPolicy:
         return
      param = ( self.feature, self.trafficPoliciesMode, policyName )
      trafficPolicyMode = \
         policiesMode[ self.trafficPolicyMode ].getOrCreateModeInstance( param )
      self.saveNamedCounters( trafficPolicyMode )
      self.saveMatchRules( trafficPolicyMode )

def _hasComments( commentKey, cliConfig ):
   return commentKey in cliConfig.comment

sampleRateValue = { 'sampleRate1in16k': '16384',
                    'sampleRate1in32k': '32768',
                    'sampleRate1in64k': '65536' }

@CliSave.saver( 'PostcardTelemetry::PolicyConfig', 'postcardtelemetry/policyconfig',
                requireMounts=( 'cli/config', 'postcardtelemetry/hwCapability',
                                'postcardtelemetry/samplePolicies/input/cli' ) )
def saveConfig( entity, root, sysdbRoot, options, requireMounts ):
   hwCapability = requireMounts[ 'postcardtelemetry/hwCapability' ]
   cliConfig = requireMounts[ 'cli/config' ]
   saveAll = options.saveAll
   if Toggles.PostcardTelemetryCommonToggleLib.\
      toggleFeaturePostcardTelemetryEnabled() and \
      hwCapability.postcardTelemetrySupported:
      for intfName in entity.intfToPortProfile:
         profileName = entity.intfToPortProfile[ intfName ]
         intfMode = root[ IntfCliSave.IntfConfigMode ].getOrCreateModeInstance(
            intfName )
         intfCmds = intfMode[ 'PostcardTelemetry.Config' ]
         intfCmds.addCommand( 'telemetry postcard policy profile %s' % profileName )

   def addCommandProfile( profile ):
      profileName = profile.name
      cmdList = []
      isDefaultProfile = profileName == 'default'
      if profile:
         if not saveAll and isDefaultProfile:
            return
         profileMode = postConfigMode[ ProfileConfigMode ].\
                       getOrCreateModeInstance( profileName )
         if profile.samplePolicy != '':
            cmdList.append( 'ingress sample policy %s'
                            % profile.samplePolicy )
         if cmdList:
            cmdSeq = profileMode[ 'PostcardTelemetry.profile' ]
            for cmd in cmdList:
               cmdSeq.addCommand( cmd )

   def isDefaultDestination( dstConfig ):
      return dstConfig.srcIp.v4Addr == '0.0.0.0' and \
         dstConfig.srcIp.v6Addr.stringValue == '::' and \
         dstConfig.dstIp.v4Addr == '0.0.0.0' and \
         dstConfig.dstIp.v6Addr.stringValue == '::' and \
         dstConfig.ttl == postcardDefaults.ttl and \
         dstConfig.dscp == postcardDefaults.dscp and \
         dstConfig.protocol == postcardDefaults.greProtocolType and \
         dstConfig.vrfName == postcardDefaults.vrf

   def isDefaultConfig( entity ):
      policiesCliConfig = requireMounts[
         'postcardtelemetry/samplePolicies/input/cli' ]
      return not entity.enable and len( entity.portProfile ) == 1 and \
         entity.sampleRate == 'sampleRate1in32k' and \
         isDefaultDestination( entity.destinationConfig ) and \
         policiesCliConfig.pmapType.pmapInput.keys() == [ 'default' ] and \
         not _hasComments( 'tele-postcard-policy', cliConfig )

   if Toggles.PostcardTelemetryCommonToggleLib.\
      toggleFeaturePostcardTelemetryEnabled() and \
      hwCapability.postcardTelemetrySupported and (
         not isDefaultConfig( entity ) or saveAll ):
      postConfigMode = root[ PostcardTelemetryConfigMode ].getOrCreateModeInstance(
         None )
      postCmds = postConfigMode[ 'PostcardTelemetry.Config' ]
      if entity.enable:
         postCmds.addCommand( 'no disabled' )
      elif saveAll:
         postCmds.addCommand( 'disabled' )
      if entity.sampleRate != 'sampleRate1in32k' or saveAll:
         postCmds.addCommand( 'ingress sample rate %s' % sampleRateValue[
            entity.sampleRate ] )
      if not isDefaultDestination( entity.destinationConfig ) or saveAll:
         dstConfig = entity.destinationConfig
         if dstConfig.srcIp.v6Addr.stringValue == '::' and \
            dstConfig.srcIp.v4Addr == '0.0.0.0':
            srcIp = '0.0.0.0'
         else:
            srcIp = dstConfig.srcIp
         if dstConfig.dstIp.v6Addr.stringValue == '::' and \
            dstConfig.dstIp.v4Addr == '0.0.0.0':
            dstIp = '0.0.0.0'
         else:
            dstIp = dstConfig.dstIp
         # ttl = dstConfig.ttl
         # dscp = dstConfig.dscp
         # protocol = dstConfig.protocol
         # vrfName = dstConfig.vrfName
         command = 'ingress collection gre source %s ' % srcIp
         command += 'destination %s ' % dstIp
         # command += 'dscp %d ' % dscp
         # command += 'ttl %d ' % ttl
         # command += 'protocol %s ' % protocol
         # command += 'vrf %s' % vrfName
         postCmds.addCommand( command )
      cliDumper = PostcardSamplePolicySaver( requireMounts[
         'postcardtelemetry/samplePolicies/input/cli' ],
                                             root, sysdbRoot, options,
                                             requireMounts, FEATURE,
                                             'mon-telemetry-postcard' )
      cliDumper.save()
      for portProfile in sorted( entity.portProfile ):
         addCommandProfile( entity.portProfile[ portProfile ] )
