# Copyright (c) 2009-2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave
import IntfCliSave
import Tac
import EthIntfUtil
import Tracing
t0 = Tracing.trace0

VlanId = Tac.Type( "Bridging::VlanId" )
CliSave.GlobalConfigMode.addCommandSequence( 'PortSec.global' )
IntfCliSave.IntfConfigMode.addCommandSequence( 'PortSec.switchport',
                                               after=[ 'Ebra.switchport' ] )

@CliSave.saver( 'PortSec::Config', 'portsec/config',
                requireMounts=( 'bridging/config', ) )
def savePortSecConfig( portSecConfig, root, sysdbRoot, options,
                       requireMounts ):
   savePortSecGlobalConfig( portSecConfig, root, options )
   # Get the list of interfaces on which config needs to be displayed.
   bridgingConfig = requireMounts[ 'bridging/config' ]
   if options.saveAllDetail:
      # We allow port-sec on L2 sub-interfaces (bridging ports on Sand),
      # hence include all interface names not just switchports
      cfgIntfNames = EthIntfUtil.allSwitchportNames( bridgingConfig,
                                                     includeEligible=True,
                                                     includeSubIntf=True )
      # all switchports, routed ports and bridged ports in intfConfig.
   elif options.saveAll:
      # We allow L2 configuration on routed ports. Set of intfs for config are
      switchportNames = EthIntfUtil.allSwitchportNames( bridgingConfig )
      cfgIntfNames = set( switchportNames + portSecConfig.intfConfig.keys() )
   else:
      cfgIntfNames = portSecConfig.intfConfig

   for intfName in cfgIntfNames:
      intfConfig = portSecConfig.intfConfig.get( intfName )
      if not intfConfig:
         if options.saveAll:
            # just creating a temporary instance. Mode (shutdown) is unimportant.
            intfConfig = Tac.Value( 'PortSec::IntfConfig', intfName )
         else:
            continue
      savePortSecIntfConfig( portSecConfig, intfConfig, root, options )

def savePortSecIntfConfig( portSecConfig, portSecIntfConfig, root, options ):
   ifConfigMode = root[ IntfCliSave.IntfConfigMode ]
   mode = ifConfigMode.getOrCreateModeInstance( portSecIntfConfig.intfId )
   cmds = mode[ 'PortSec.switchport' ]
   if EthIntfUtil.isSubIntf( portSecIntfConfig.intfId ):
      # mac address limit Cli Save
      if portSecIntfConfig.mode == 'protect':
         cmds.addCommand( "mac address limit violation protect" )
      elif options.saveAll or options.saveAllDetail:
         cmds.addCommand( "mac address limit violation shutdown" )
      if portSecIntfConfig.enabled:
         cmd = 'mac address limit'
         if portSecIntfConfig.maxAddrs != 1:
            cmd += " %d" % portSecIntfConfig.maxAddrs
         cmds.addCommand( cmd )
      elif options.saveAll or options.saveAllDetail:
         cmds.addCommand( "no mac address limit" )
   else:
      if portSecIntfConfig.vlanBased:
         cmds.addCommand( "switchport port-security mac-address maximum disabled" )
      elif portSecIntfConfig.maxAddrs != 1 or options.saveAll:
         # moving this above enabling port-security because it's really better
         # to set max addrs first to prevent a bunch of code to reevaluate the port
         cmd = "switchport port-security mac-address maximum %d"
         cmds.addCommand( cmd % portSecIntfConfig.maxAddrs )

      cmd = "switchport port-security vlan %d mac-address maximum %d"
      defaultCmd = "switchport port-security vlan default mac-address maximum %d"
      default = False
      for vlanId in portSecIntfConfig.vlanConfig:
         vcfg = portSecIntfConfig.vlanConfig.get( vlanId )
         if vlanId == VlanId.invalid:
            default = True
            defaultMax = vcfg.maxAddrs
         else:
            cmds.addCommand( cmd % ( vcfg.vlanId, vcfg.maxAddrs ) )
      if default:
         cmds.addCommand( defaultCmd % defaultMax )

      if portSecIntfConfig.enabled:
         if portSecIntfConfig.mode == 'shutdown':
            cmds.addCommand( "switchport port-security" )
         else:
            if portSecIntfConfig.log:
               cmds.addCommand( "switchport port-security violation protect log" )
            else:
               cmds.addCommand( "switchport port-security violation protect" )
      elif options.saveAll:
         cmds.addCommand( "no switchport port-security" )

def savePortSecGlobalConfig( portSecConfig, root, options ):
   cmds = root[ 'PortSec.global' ]

   if portSecConfig.allowSecureAddressAging:
      cmds.addCommand( "switchport port-security mac-address aging" )
   elif options.saveAllDetail or options.saveAll:
      cmds.addCommand( "no switchport port-security mac-address aging" )

   if portSecConfig.allowSecureAddressMoves:
      cmds.addCommand( "switchport port-security mac-address moveable" )
   elif portSecConfig.allowSecureAddressMovesData:
      cmds.addCommand( "switchport port-security mac-address moveable data" )
   elif portSecConfig.allowSecureAddressMovesPhone:
      cmds.addCommand( "switchport port-security mac-address moveable phone" )
   elif options.saveAllDetail or options.saveAll:
      cmds.addCommand( "no switchport port-security mac-address moveable" )

   if not portSecConfig.persistenceEnabled:
      cmds.addCommand( "switchport port-security persistence disabled" )
   elif options.saveAllDetail or options.saveAll:
      cmds.addCommand( "no switchport port-security persistence disabled" )
