# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Tracing
import CliMode
import CliMode.Poe
import CliSave
from IntfCliSave import IntfConfigMode
import Toggles.PlutoToggleLib

__defaultTraceHandle__ = Tracing.Handle( 'PlutoCliSave' )
t0 = Tracing.trace0

DEFAULT_REBOOT_ACTION = 'powerOff'
DEFAULT_INTF_POE_REBOOT_ACTION = 'unset'
DEFAULT_INTF_POE_LINK_DOWN_ACTION = 'unset'
DEFAULT_INTERFACE_SHUTDOWN_ACTION = 'maintain'
DEFAULT_INTF_POE_SHUTDOWN_ACTION = 'unset'
DEFAULT_DYNAMIC_POE_ENABLED = False

DEFAULT_POE_PORT_PRIORITY = 'defaultPriority'
DEFAULT_INTF_POE_PAIRSET = 'fourPair'
DEFAULT_POE_ENABLED = True
DEFAULT_POE_POWER_LIMIT = Tac.Value( "Units::Watts" ).value
DEFAULT_POE_POWER_LIMIT_FIXED = False
DEFAULT_POE_LLDP_ENABLED = True
DEFAULT_POE_LEGACY_DETECT = False

class PoeConfigMode( CliMode.Poe.PoeMode, CliSave.Mode ):
   def __init__( self, param ):
      CliMode.Poe.PoeMode.__init__( self )
      CliSave.Mode.__init__( self, param )

class PoeLinecardConfigMode( CliMode.Poe.PoeLinecardMode, CliSave.Mode ):
   def __init__( self, param ):
      CliMode.Poe.PoeLinecardMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( PoeConfigMode )
PoeConfigMode.addCommandSequence( 'Poe.config' )
IntfConfigMode.addCommandSequence( 'Pluto.poeConfig' )
if Toggles.PlutoToggleLib.togglePoeLinecardConfigModeEnabled():
   PoeConfigMode.addChildMode( PoeLinecardConfigMode )
   PoeLinecardConfigMode.addCommandSequence( 'PoeLinecard.config' )

def saveLinecardEntity( entity, root, saveMode, linecard, cmdKey, saveAll ):
   mode = root[ saveMode ].getOrCreateModeInstance( linecard )
   cmds = mode[ cmdKey ]
   if entity.linecardPortPriority != DEFAULT_POE_PORT_PRIORITY:
      cmds.addCommand( 'priority %s' % entity.linecardPortPriority )
   elif saveAll:
      cmds.addCommand( 'no priority' )

@CliSave.saver( 'Hardware::PLPoePort::CliConfig',
                'hardware/poe/config/cli',
                requireMounts=( 'hardware/platform/device/config/slice',
                                'hardware/poe/config', ) )
def savePoe( entity, root, sysdbRoot, options, requireMounts ):
   poePorts = set( entity.poeCliConfig.keys() )
   linecardPortPriority = entity.linecardPortPriority.keys()
   plutoConfigDir = requireMounts[ 'hardware/platform/device/config/slice' ]
   poetecConfigDir = requireMounts[ 'hardware/poe/config' ]
   if 'PoeDevice' in plutoConfigDir:
      poePorts.update( plutoConfigDir[ 'PoeDevice' ].poePort.keys() )
   elif 'poetec' in poetecConfigDir and poetecConfigDir[ 'poetec' ].poeCard:
      for card in poetecConfigDir[ 'poetec' ].poeCard.values():
         for chip in card.poeChip.values():
            poePorts.update( chip.poePort.keys() )

   if not poePorts and not linecardPortPriority:
      return
   t0( poePorts )

   actionToken = {
      'powerOff' : 'power-off',
      'maintain' : 'maintain',
   }
   pairsetToken = {
      'fourPair' : '4-pair',
      'altA' : 'alternative a',
      'altB' : 'alternative b',
   }
   if entity.rebootAction != DEFAULT_REBOOT_ACTION or options.saveAll:
      mode = root[ PoeConfigMode ].getSingletonInstance()
      cmds = mode[ 'Poe.config' ]
      cmds.addCommand( 'reboot action %s' % actionToken[ entity.rebootAction ] )

   if ( entity.intfShutdownAction != DEFAULT_INTERFACE_SHUTDOWN_ACTION or
        options.saveAll ):
      mode = root[ PoeConfigMode ].getSingletonInstance()
      cmds = mode[ 'Poe.config' ]
      cmds.addCommand( 'interface shutdown action %s' %
                       actionToken[ entity.intfShutdownAction ] )

   if Toggles.PlutoToggleLib.togglePoeLinecardConfigModeEnabled():
      for linecard in linecardPortPriority:
         mode = root[ PoeConfigMode ].getSingletonInstance()
         linecardEntity = entity.linecardPortPriority[ linecard ]
         saveLinecardEntity( linecardEntity, mode, PoeLinecardConfigMode, linecard,
                             'PoeLinecard.config', options.saveAll )
   
   if Toggles.PlutoToggleLib.toggleDynamicPoeConfigEnabled():
      if entity.dynamicPoeEnabled != DEFAULT_DYNAMIC_POE_ENABLED or options.saveAll:
         mode = root[ PoeConfigMode ].getSingletonInstance()
         cmds = mode[ 'Poe.config' ]
         cmds.addCommand( 'power budget allocation usage-based' )

   for intf in poePorts:
      t0( "intf", intf )
      if intf in entity.poeCliConfig:
         t0( "intf configured" )
         portPriority = entity.poeCliConfig[ intf ].portPriority
         portPairset = entity.poeCliConfig[ intf ].portPairset
         rebootAction = entity.poeCliConfig[ intf ].rebootAction
         linkDownAction = entity.poeCliConfig[ intf ].linkDownAction
         linkDownPoeOffTime = entity.poeCliConfig[ intf ].linkDownPoeOffTime
         intfShutdownAction = entity.poeCliConfig[ intf ].intfShutdownAction
         poeEnabled = entity.poeCliConfig[ intf ].poeEnabled
         powerLimit = entity.poeCliConfig[ intf ].powerLimit
         fixedLimit = entity.poeCliConfig[ intf ].fixedLimit
         lldpEnabled = entity.poeCliConfig[ intf ].lldpEnabled
         legacyDetect = entity.poeCliConfig[ intf ].legacyDetect
      else:
         portPriority = DEFAULT_POE_PORT_PRIORITY
         portPairset = DEFAULT_INTF_POE_PAIRSET
         rebootAction = DEFAULT_INTF_POE_REBOOT_ACTION
         linkDownAction = DEFAULT_INTF_POE_LINK_DOWN_ACTION
         poeEnabled = DEFAULT_POE_ENABLED
         intfShutdownAction = DEFAULT_INTF_POE_SHUTDOWN_ACTION
         powerLimit = DEFAULT_POE_POWER_LIMIT
         fixedLimit = DEFAULT_POE_POWER_LIMIT_FIXED
         lldpEnabled = DEFAULT_POE_LLDP_ENABLED
         legacyDetect = DEFAULT_POE_LEGACY_DETECT

      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intf )
      cmds = mode[ 'Pluto.poeConfig' ]

      if portPriority != DEFAULT_POE_PORT_PRIORITY:
         cmds.addCommand( 'poe priority %s' % portPriority )
      elif options.saveAll:
         cmds.addCommand( 'no poe priority' )

      if portPairset != DEFAULT_INTF_POE_PAIRSET:
         cmds.addCommand( 'poe pairset %s' % pairsetToken[ portPairset ] )
      elif options.saveAll:
         cmds.addCommand( 'no poe pairset' )

      if Toggles.PlutoToggleLib.togglePoeRebootActionCliCommandEnabled():
         if rebootAction != DEFAULT_INTF_POE_REBOOT_ACTION:
            cmds.addCommand( 'poe reboot action %s' % actionToken[ rebootAction ] )
         elif options.saveAll:
            cmds.addCommand( 'default poe reboot action' )

      if Toggles.PlutoToggleLib.togglePoeLinkDownActionCliCommandEnabled():
         if linkDownAction == 'powerOff':
            cmds.addCommand( 'poe link down action %s %d seconds' %
                             ( actionToken[ linkDownAction ], linkDownPoeOffTime ) )
         elif linkDownAction != DEFAULT_INTF_POE_LINK_DOWN_ACTION:
            cmds.addCommand( 'poe link down action %s' %
                             actionToken[ linkDownAction ] )
         elif options.saveAll:
            cmds.addCommand( 'default poe link down action' )

      if intfShutdownAction != DEFAULT_INTF_POE_SHUTDOWN_ACTION:
         cmds.addCommand( 'poe shutdown action %s' %
                          actionToken[ intfShutdownAction ] )
      elif options.saveAll:
         cmds.addCommand( 'default poe shutdown action' )

      if poeEnabled != DEFAULT_POE_ENABLED:
         cmds.addCommand( 'poe disabled' )
      elif options.saveAll:
         cmds.addCommand( 'poe' )

      if powerLimit != DEFAULT_POE_POWER_LIMIT:
         if fixedLimit != DEFAULT_POE_POWER_LIMIT_FIXED:
            cmds.addCommand( 'poe limit %.2f watts fixed' % powerLimit )
         else:
            cmds.addCommand( 'poe limit %.2f watts' % powerLimit )
      elif options.saveAll:
         cmds.addCommand( 'no poe limit' )

      if lldpEnabled != DEFAULT_POE_LLDP_ENABLED:
         cmds.addCommand( 'poe negotiation lldp disabled' )
      elif options.saveAll:
         cmds.addCommand( 'poe negotiation lldp' )

      if legacyDetect != DEFAULT_POE_LEGACY_DETECT:
         cmds.addCommand( 'poe legacy detect' )
      elif options.saveAll:
         cmds.addCommand( 'no poe legacy detect' )
