#!/usr/bin/env python
# Copyright (c) 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave
import Tac
import McastCommonCliLib
from CliSavePlugin.PimCliSaveLib import RouterPimSparseBaseConfigMode, \
       RouterPimSparseVrfConfigMode, getCmdRoot
from CliSavePlugin.MrouteCliSave import getCliSaveVersion
import CliParser
import Tracing
import Toggles.PimToggleLib
import IpUtils

__defaultTraceHandle__ = Tracing.Handle( "PimsmSave" )
t1 = Tracing.trace1

AddressFamily = Tac.Type( "Arnet::AddressFamily" )
PimsmConfigColl = Tac.Type( "Routing::Pim::SparseMode::ConfigColl" )
GmpSsmConfigColl = Tac.Type( "Routing::Gmp::GmpSsmConfigColl" )
PimLegacyConfig = Tac.Type( "McastCommon::LegacyConfig" )

CliSave.GlobalConfigMode.addCommandSequence( 'Ip.Pim' )

RouterPimSparseBaseConfigMode.addCommandSequence( 'Pim.config' )
RouterPimSparseBaseConfigMode.addChildMode( RouterPimSparseVrfConfigMode )
RouterPimSparseVrfConfigMode.addCommandSequence( 'Pim.vrf.config' )

pimsmVrfCliSavers = []
def pimsmVrfCliSaver( func ):
   pimsmVrfCliSavers.append( func )

ssmVrfCliSavers = []
def ssmVrfCliSaver( func ):
   ssmVrfCliSavers.append( func )

pimsmVrfNonLegacyCliSavers = []
def pimsmVrfNonLegacyCliSaver( func ):
   pimsmVrfNonLegacyCliSavers.append( func )

@ssmVrfCliSaver
def saveSsmRange( vrfConfig, cmds, saveAll, pimLegacyVersion ):
   cmd = 'ssm range'
   if pimLegacyVersion != PimLegacyConfig.ipMode:
      cmd = 'ip pim ' + cmd

   if not vrfConfig.ssmFilter:
      cmds.addCommand( "%s standard" %cmd )
   elif vrfConfig.ssmFilter != vrfConfig.ssmFilterDefault:
      cmds.addCommand( "%s %s" %( cmd, vrfConfig.ssmFilter ) )
   elif vrfConfig.ssmFilter == vrfConfig.ssmFilterDefault and saveAll:
      cmds.addCommand( "no %s" %cmd )

@pimsmVrfCliSaver
def saveSgExpiryTimer( vrfConfig, cmds, saveAll, pimLegacyVersion ):
   cmd = 'sg-expiry-timer'
   if pimLegacyVersion != PimLegacyConfig.ipMode:
      cmd = 'ip pim sparse-mode ' + cmd

   if vrfConfig.sgExpiryTimer != vrfConfig.sgExpiryTimerDefault or saveAll:
      cmds.addCommand( '%s %d' %( cmd, vrfConfig.sgExpiryTimer ) )

@pimsmVrfCliSaver
def saveFrrFilter( vrfConfig, cmds, saveAll, pimLegacyVersion ):
   cmd = 'fast-reroute'
   if pimLegacyVersion != PimLegacyConfig.ipMode:
      cmd = 'ip pim sparse-mode ' + cmd

   if ( vrfConfig.frrFilter and \
         vrfConfig.frrFilter != vrfConfig.frrFilterDefault ):
      cmds.addCommand( '%s %s' %( cmd, vrfConfig.frrFilter ) )

@pimsmVrfCliSaver
def saveSptThreshold( vrfConfig, cmds, saveAll, pimLegacyVersion ):
   cmd = 'spt threshold'
   if pimLegacyVersion != PimLegacyConfig.ipMode:
      cmd = 'ip pim spt-threshold'

   if vrfConfig.drSwitch != vrfConfig.drSwitchDefault or saveAll:
      if vrfConfig.drSwitch == 'never':
         cmds.addCommand( "%s infinity" %cmd )
      else:
         cmds.addCommand( "%s 0" %cmd )

   for acl in vrfConfig.drSwitchAcl:
      val = vrfConfig.drSwitchAcl[ acl ]
      if pimLegacyVersion != PimLegacyConfig.ipMode:
         if val == 'never':
            cmds.addCommand( "%s infinity group-list %s" % ( cmd, acl ) )
         else:
            cmds.addCommand( "%s 0 group-list %s" % ( cmd, acl ) )
      else:
         if val == 'never':
            cmds.addCommand( "%s infinity match list %s" % ( cmd, acl ) )
         else:
            cmds.addCommand( "%s 0 match list %s" % ( cmd, acl ) )

@ssmVrfCliSaver
def saveSsmConvert( vrfConfig, cmds, saveAll, pimLegacyVersion ):
   cmd = 'ssm default source'
   if pimLegacyVersion != PimLegacyConfig.ipMode:
      cmd = 'ip pim ' + cmd

   for g in sorted( vrfConfig.convertGroup ):
      for s in sorted( vrfConfig.convertGroup[ g ].groupSource,
                       cmp=IpUtils.compareIpAddress ):
         cmds.addCommand( "%s %s %s" % ( cmd, g, s ) )

@pimsmVrfCliSaver
def saveMakeBeforeBreak( vrfConfig, cmds, saveAll, pimLegacyVersion ):
   cmd = 'make-before-break'
   if pimLegacyVersion != PimLegacyConfig.ipMode:
      return
   if vrfConfig.disableMbb or saveAll:
      if vrfConfig.disableMbb:
         cmds.addCommand( "%s disabled" % cmd )
      else:
         cmds.addCommand( cmd )

@pimsmVrfNonLegacyCliSaver
def saveRouteSgInstallThreshold( vrfConfig, cmds, saveAll, af ):
   if not Toggles.PimToggleLib.togglePimSgInstallOnTrafficEnabled():
      return

   if af == AddressFamily.ipv6:
      # No support for ipv6
      return

   cmd = 'route sg installation threshold'

   if vrfConfig.sgInstallThresh != vrfConfig.sgInstallThreshDefault or saveAll:
      cmds.addCommand( '%s %s' % ( cmd, vrfConfig.sgInstallThresh ) )

# ---------------------------- #
#     PimsmConfig Saving       #
# ---------------------------- #

def saveConfig( pimsmConfigColl, root, sysdbRoot, options,
                hardwareStatus, legacyVersion, af ):
   saveAll = options.saveAll
   # Save the default config only if the platform supports multicast routing
   if not McastCommonCliLib.mcastRoutingSupported( sysdbRoot, hardwareStatus ):
      saveAll = False
   version = getCliSaveVersion( legacyVersion, saveAll, pimsmConfigColl.isDefault() )

   for vrfName, vrfConfig in pimsmConfigColl.vrfConfig.items():
      if not vrfConfig.isDefault() or saveAll:
         cmds = getCmdRoot( root, vrfName, af, version )
         for saver in pimsmVrfCliSavers:
            saver( vrfConfig, cmds, saveAll, version )

         for saver in pimsmVrfNonLegacyCliSavers:
            saver( vrfConfig, cmds, saveAll, af )

# pylint: disable-msg=E0102
@CliSave.saver( 'Routing::Pim::SparseMode::ConfigColl',
                PimsmConfigColl.mountPath( AddressFamily.ipv4 ),
      requireMounts = ( 'routing/pim/legacyconfig',
                        'routing/hardware/status' ) )
def saveIpv4RouterPimSparseConfig( pimsmConfigColl, root, sysdbRoot, options,
                requireMounts ):
   hardwareStatus = requireMounts[ 'routing/hardware/status' ]
   legacyConfig = requireMounts[ 'routing/pim/legacyconfig' ]
   legacyVersion = PimLegacyConfig.ipMode \
               if not CliParser.DEPRECATED_CMD_ALLOWED else legacyConfig.version
   saveConfig( pimsmConfigColl, root, sysdbRoot, options, hardwareStatus,
               legacyVersion, AddressFamily.ipv4 )

# pylint: disable-msg=E0102
@CliSave.saver( 'Routing::Pim::SparseMode::ConfigColl',
                PimsmConfigColl.mountPath( AddressFamily.ipv6 ),
      requireMounts = ( 'routing/pim/legacyconfig',
                        'routing6/hardware/status' ) )
def saveIpv6RouterPimSparseConfig( pimsmConfigColl, root, sysdbRoot, options,
                requireMounts ):
   hardwareStatus = requireMounts[ 'routing6/hardware/status' ]
   legacyConfig = requireMounts[ 'routing/pim/legacyconfig' ]
   legacyVersion = legacyConfig.version
   version = getCliSaveVersion( legacyVersion, options.saveAll, 
         pimsmConfigColl.isDefault() )
   if version < legacyConfig.ipMode:
      return
   legacyVersion = PimLegacyConfig.ipMode
   saveConfig( pimsmConfigColl, root, sysdbRoot, options, hardwareStatus,
               legacyVersion, AddressFamily.ipv6 )

# ---------------------------- #
#     GmpSsmConfig Saving      #
# ---------------------------- #

def saveSsmConfig( gmpSsmConfigColl, root, sysdbRoot, options,
      hardwareStatus, legacyVersion, af ):
   saveAll = options.saveAll
   # Save the default config only if the platform supports multicast routing
   if not McastCommonCliLib.mcastRoutingSupported( sysdbRoot, hardwareStatus ):
      saveAll = False
   version = getCliSaveVersion( legacyVersion,
         saveAll, gmpSsmConfigColl.isDefault() )

   for vrfName, config in gmpSsmConfigColl.config.items():
      if not config.isDefault() or saveAll:
         cmds = getCmdRoot( root, vrfName, af, version )
         for saver in ssmVrfCliSavers:
            saver( config, cmds, saveAll, version )

@CliSave.saver( 'Routing::Gmp::GmpSsmConfigColl',
      GmpSsmConfigColl.mountPath( AddressFamily.ipv4 ),
      requireMounts = ( 'routing/pim/legacyconfig',
                        'routing/hardware/status' ) )
def saveIpv4RouterPimSparseSsmConfig( gmpSsmConfigColl, root, sysdbRoot, options,
      requireMounts ):
   hardwareStatus = requireMounts[ 'routing/hardware/status' ]
   legacyConfig = requireMounts[ 'routing/pim/legacyconfig' ]
   legacyVersion = PimLegacyConfig.ipMode \
         if not CliParser.DEPRECATED_CMD_ALLOWED else legacyConfig.version
   saveSsmConfig( gmpSsmConfigColl, root, sysdbRoot, options, hardwareStatus,
         legacyVersion, AddressFamily.ipv4 )
   
@CliSave.saver( 'Routing::Gmp::GmpSsmConfigColl',
      GmpSsmConfigColl.mountPath( AddressFamily.ipv6 ),
      requireMounts = ( 'routing/pim/legacyconfig',
                        'routing6/hardware/status' ) )
def saveIpv6RouterPimSparseSsmConfig( gmpSsmConfigColl, root, sysdbRoot, options,
      requireMounts ):
   hardwareStatus = requireMounts[ 'routing6/hardware/status' ]
   legacyConfig = requireMounts[ 'routing/pim/legacyconfig' ]
   legacyVersion = legacyConfig.version
   version = getCliSaveVersion( legacyVersion, options.saveAll,
         gmpSsmConfigColl.isDefault() )
   if version < legacyConfig.ipMode:
      return
   legacyVersion = PimLegacyConfig.ipMode
   saveSsmConfig( gmpSsmConfigColl, root, sysdbRoot, options,
         hardwareStatus, legacyVersion, AddressFamily.ipv6 )
