#!/usr/bin/env python
# Copyright (c) 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave, IpUtils
import McastCommonCliLib
import Tac
import Tracing
from CliSavePlugin.PimCliSaveLib import RouterPimSparseBaseConfigMode, \
      RouterPimSparseVrfConfigMode, RouterPimSparseAfConfigMode
from CliSavePlugin.MrouteCliSave import getCliSaveVersion
from IpLibConsts import DEFAULT_VRF

__defaultTraceHandle__ = Tracing.Handle( "PimRegSave" )
t1 = Tracing.trace1

CliSave.GlobalConfigMode.addCommandSequence( 'Ip.PimReg', after=[ 'Ip.Pim' ] )
RouterPimSparseBaseConfigMode.addCommandSequence( 'PimReg.config', 
      after=[ 'Pim.config'] )
RouterPimSparseBaseConfigMode.addChildMode( RouterPimSparseVrfConfigMode )
RouterPimSparseVrfConfigMode.addCommandSequence( 'PimReg.vrf.config', 
                                             after=[ 'Pim.vrf.config' ] )
RouterPimSparseAfConfigMode.addCommandSequence( 'PimReg.vrf.af.config',
                                             after=[ 'Pim.vrf.af.config' ] )

AddressFamily = Tac.Type( "Arnet::AddressFamily" )
PimLegacyConfig = Tac.Type( "McastCommon::LegacyConfig" )

pimRegVrfCliSavers = []
def pimRegVrfCliSaver( func ):
   pimRegVrfCliSavers.append( func )

@pimRegVrfCliSaver
def saveRpAnycast( vrfConfig, cmds, saveAll, pimLegacyVersion ):
   baseCmd = 'anycast-rp'
   if pimLegacyVersion != PimLegacyConfig.ipMode:
      baseCmd = 'ip pim %s' % baseCmd

   # Need to make the sort AF independent.
   for r in sorted( vrfConfig.anycastRpConfig.values(),
                    key=lambda x: x.rpAddr, cmp=IpUtils.compareIpAddress ):
      for k in sorted( r.peer.keys(), cmp=IpUtils.compareIpAddress ):
         peer = r.peer[ k ]
         cmd = '%s %s %s' % ( baseCmd, r.rpAddr, peer.peerAddr )
         if peer.registerCount != vrfConfig.registerCountDefault or saveAll:
            if peer.registerCount == 0xFFFFFFFF:
               regCnt = 'infinity'
            else:
               regCnt = peer.registerCount
            cmd += ' register-count %s' % regCnt
         cmds.addCommand( cmd )

@pimRegVrfCliSaver
def saveRegisterSource( vrfConfig, cmds, saveAll, pimLegacyVersion ):
   if pimLegacyVersion == PimLegacyConfig.ipMode:
      baseCmd = 'register local-interface'
   else:
      baseCmd = 'ip pim register-source'

   registerSrcEnum = Tac.Type( "Routing::PimReg::RegisterSrc" )

   if vrfConfig.registerSrc == registerSrcEnum.interface:
      cmds.addCommand( '%s %s'
                        % ( baseCmd, vrfConfig.registerSrcIntf ) )
   elif vrfConfig.registerSrc == registerSrcEnum.iif:
      cmds.addCommand( '%s iif' % baseCmd )
   elif saveAll:
      cmds.addCommand( 'no %s ' % baseCmd )

def getCmdRoot( root, vrfName, af, pimLegacyVersion ):
   def getRouterMode():
      return root[ RouterPimSparseBaseConfigMode ].getSingletonInstance()

   def getRouterVrfMode( vrfName ):
      routerMode = getRouterMode()
      return routerMode[ RouterPimSparseVrfConfigMode ].getOrCreateModeInstance(
               vrfName )

   def getIpMode( vrfName, af ):
      if vrfName == DEFAULT_VRF:
         parentMode = getRouterMode()
      else:
         parentMode = getRouterVrfMode( vrfName )
      return parentMode[ RouterPimSparseAfConfigMode ].getOrCreateModeInstance(
                                                         ( vrfName, af ) )

   if pimLegacyVersion == PimLegacyConfig.ipMode:
      cmds = getIpMode( vrfName, af )[ 'PimReg.vrf.af.config' ]
   elif af == AddressFamily.ipv4:
      if vrfName == DEFAULT_VRF:
         if pimLegacyVersion == PimLegacyConfig.globalMode:
            cmds = root[ 'Ip.PimReg' ]
         else:
            cmds = getRouterMode()[ 'PimReg.config' ]
      else:
         cmds = getRouterVrfMode( vrfName )[ 'PimReg.vrf.config' ]

   return cmds

@CliSave.saver( 'Routing::PimReg::ConfigColl', 'routing/pimreg/configcoll',
                requireMounts = ( 'routing/pim/legacyconfig',
                                  'routing/hardware/status', ) )
def saveIpv4Config( pimConfigColl, root, sysdbRoot, options,
                requireMounts ):
   saveAll = options.saveAll

   # Save the default config only if the platform supports multicast routing
   if not McastCommonCliLib.mcastRoutingSupported(
         sysdbRoot,
         requireMounts[ 'routing/hardware/status' ] ):
      saveAll = False

   config = pimConfigColl.vrfConfig[ DEFAULT_VRF ]

   pimLegacyConfig = requireMounts[ 'routing/pim/legacyconfig' ]
   pimLegacyVersion = getCliSaveVersion( pimLegacyConfig.version, saveAll,
                                         pimConfigColl.isDefault() )

   if not config.isDefault() or saveAll:
      saveRoot = getCmdRoot( root, DEFAULT_VRF,
                             AddressFamily.ipv4, pimLegacyVersion )
      saveConfigImpl( config, saveRoot, saveAll, pimLegacyVersion )

   for vrfName in pimConfigColl.vrfConfig:
      config = pimConfigColl.vrfConfig[ vrfName ]

      if vrfName == DEFAULT_VRF:
         continue
      else:
         if not config.isDefault() or saveAll:
            saveRoot = \
               getCmdRoot( root, vrfName, AddressFamily.ipv4, pimLegacyVersion )
            saveConfigImpl( config, saveRoot, saveAll, pimLegacyVersion )

@CliSave.saver( 'Routing::PimReg::ConfigColl', 'routing6/pimreg/configcoll',
                requireMounts = ( 'routing/pim/legacyconfig',
                                  'routing6/hardware/status', ) )
def saveIpv6Config( pimConfigColl, root, sysdbRoot, options,
                requireMounts ):
   saveAll = options.saveAll

   # Save the default config only if the platform supports multicast routing
   if not McastCommonCliLib.mcast6RoutingSupported(
         sysdbRoot,
         requireMounts[ 'routing6/hardware/status' ] ):
      saveAll = False

   config = pimConfigColl.vrfConfig[ DEFAULT_VRF ]

   pimLegacyConfig = requireMounts[ 'routing/pim/legacyconfig' ]
   pimLegacyVersion = getCliSaveVersion( pimLegacyConfig.version, saveAll,
                                         pimConfigColl.isDefault() )

   if not config.isDefault() or saveAll:
      saveRoot = getCmdRoot( root, DEFAULT_VRF,
                             AddressFamily.ipv6, pimLegacyVersion )
      saveConfigImpl( config, saveRoot, saveAll, pimLegacyVersion )

   for vrfName in pimConfigColl.vrfConfig:
      config = pimConfigColl.vrfConfig[ vrfName ]

      if vrfName == DEFAULT_VRF:
         continue
      else:
         if not config.isDefault() or saveAll:
            saveRoot = \
               getCmdRoot( root, vrfName, AddressFamily.ipv6, pimLegacyVersion )
            saveConfigImpl( config, saveRoot, saveAll, pimLegacyVersion )

def saveConfigImpl( pimConfig, root, saveAll, pimLegacyVersion ):
   for saver in pimRegVrfCliSavers:
      saver( pimConfig, root, saveAll, pimLegacyVersion )
