#!/usr/bin/env python
# Copyright (c) 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Toggles.PimToggleLib
import CliParser, CliSave, IpUtils
from IntfCliSave import IntfConfigMode
from RoutingIntfUtils import allRoutingProtocolIntfNames
import Tac, McastCommonCliLib
import Tracing
from CliSavePlugin.MrouteCliSave import getCliSaveVersion
import PimCliSaveLib
from PimCliSaveLib import ( RouterPimBaseConfigMode,
                            RouterPimVrfConfigMode,
                            RouterPimSparseBaseConfigMode,
                            RouterPimSparseVrfConfigMode,
                            RouterPimBidirBaseConfigMode,
                            RouterPimBidirVrfConfigMode,
                            RouterPimBidirAfConfigMode,
                            RouterPimSparseAfConfigMode,
                            getCmdRoot, getCmdRootBidir )
__defaultTraceHandle__ = Tracing.Handle( "PimSave" )
t1 = Tracing.trace1

AddressFamily = Tac.Type( "Arnet::AddressFamily" )
PimLegacyConfig = Tac.Type( "McastCommon::LegacyConfig" )

IntfConfigMode.addCommandSequence( 'Pim.intf', after=[ 'Ira.ipIntf' ] )

IpPimCmdSeq = 'Ip.Pim'
CliSave.GlobalConfigMode.addCommandSequence( IpPimCmdSeq,
                                             before=[ IntfConfigMode ] )
CliSave.GlobalConfigMode.addCommandSequence( 'Ip.PimBidir',
                                             before=[ IntfConfigMode ] )
CliSave.GlobalConfigMode.addChildMode( RouterPimBaseConfigMode,
                                             after=[ IntfConfigMode ] )
CliSave.GlobalConfigMode.addChildMode( RouterPimSparseBaseConfigMode,
                                       after=[ RouterPimBaseConfigMode ] )
CliSave.GlobalConfigMode.addChildMode( RouterPimBidirBaseConfigMode,
      after=[ RouterPimSparseBaseConfigMode ] )

RouterPimBaseConfigMode.addCommandSequence( 'Pim.config' )
RouterPimBaseConfigMode.addChildMode( RouterPimVrfConfigMode )
RouterPimVrfConfigMode.addCommandSequence( 'Pim.vrf.config' )

RouterPimSparseBaseConfigMode.addCommandSequence( 'Pim.config' )
RouterPimSparseBaseConfigMode.addChildMode( RouterPimSparseVrfConfigMode )
RouterPimSparseBaseConfigMode.addChildMode( RouterPimSparseAfConfigMode )
RouterPimSparseVrfConfigMode.addCommandSequence( 'Pim.vrf.config' )
RouterPimSparseVrfConfigMode.addChildMode( RouterPimSparseAfConfigMode )
RouterPimSparseAfConfigMode.addCommandSequence( 'Pim.vrf.af.config' )

RouterPimBidirBaseConfigMode.addCommandSequence( 'PimBidir.config' )
RouterPimBidirBaseConfigMode.addChildMode( RouterPimBidirVrfConfigMode )
RouterPimBidirBaseConfigMode.addChildMode( RouterPimBidirAfConfigMode )
RouterPimBidirVrfConfigMode.addCommandSequence( 'PimBidir.vrf.config' )
RouterPimBidirVrfConfigMode.addChildMode( RouterPimBidirAfConfigMode )
RouterPimBidirAfConfigMode.addCommandSequence( 'PimBidir.vrf.af.config' )

sparseVrfCliSavers = []
def sparseVrfCliSaver( func ):
   sparseVrfCliSavers.append( func )

bidirVrfCliSavers = []
def bidirVrfCliSaver( func ):
   bidirVrfCliSavers.append( func )

def sparseAndBidirVrfCliSaver( func ):
   sparseVrfCliSavers.append( func )
   bidirVrfCliSavers.append( func )

@sparseAndBidirVrfCliSaver
def saveLogNeighbors( vrfConfig, cmds, saveAll, pimLegacyVersion, af ):
   cmd = 'log neighbors'

   if not vrfConfig.logNeighborChanges:
      cmds.addCommand( 'no ' + cmd )
   elif saveAll:
      cmds.addCommand( cmd )

@sparseVrfCliSaver
def saveBfd( vrfConfig, cmds, saveAll, pimLegacyVersion, af ):
   if vrfConfig.bfdIntfDefault:
      cmds.addCommand( 'bfd' )
   elif saveAll:
      cmds.addCommand( 'no bfd' )

@sparseVrfCliSaver
def saveSctpSrcInterface( vrfConfig, cmds, saveAll, pimLegacyVersion, af ):
   if af == 'ipv6' and not Toggles.PimToggleLib.togglePimIpv6PortEnabled():
      return
   if pimLegacyVersion != PimLegacyConfig.ipMode:
      return
   if vrfConfig.sctpSrcIntf != "":
      cmds.addCommand( "transport sctp source-interface %s" %
                       vrfConfig.sctpSrcIntf )
   elif saveAll:
      cmds.addCommand( "default transport sctp source-interface" )

@sparseVrfCliSaver
def saveAllowRp( vrfConfig, cmds, saveAll, pimLegacyVersion, af ):
   if pimLegacyVersion == PimLegacyConfig.globalMode:
      return
   cmd = 'rp allow' if pimLegacyVersion == PimLegacyConfig.ipMode \
         else 'ip pim allow-rp'
   if vrfConfig.allowRp:
      cmds.addCommand( cmd )
   elif saveAll:
      cmds.addCommand( 'no ' + cmd )

def saveIntfConfigConverted( pimIntfConfig, root, sysdbRoot, saveAll, saveAllDetail,
                             pimLegacyConfig, af ):
   pic = pimIntfConfig

   mode = root[ IntfConfigMode ].getOrCreateModeInstance( pic.intfId )
   cmds = mode[ 'Pim.intf' ]
   assert af in [ AddressFamily.ipv4, AddressFamily.ipv6 ]

   legacy = False
   if af == AddressFamily.ipv4:
      version = PimLegacyConfig.ipMode \
         if CliParser.DEPRECATED_CMD_ALLOWED is False else pimLegacyConfig.version
      legacy = version != PimLegacyConfig.ipMode

   if af == AddressFamily.ipv4:
      if legacy:
         cmdPrefix = 'ip pim '
      else:
         cmdPrefix = 'pim %s ' % af

   if af == AddressFamily.ipv6:
      cmdPrefix = 'pim %s ' % af
      legacy = False

   noCmdPrefix = 'no ' + cmdPrefix

   if pic.mode in [ 'modePimSm', 'modePimSmAndBidir' ]:
      cmds.addCommand( cmdPrefix + 'sparse-mode' )
   elif saveAll:
      cmds.addCommand( noCmdPrefix + 'sparse-mode' )

   if af == AddressFamily.ipv4:
      if pic.mode in [ 'modePimBidir', 'modePimSmAndBidir' ]:
         cmds.addCommand( cmdPrefix + 'bidirectional' )
      elif saveAll:
         cmds.addCommand( noCmdPrefix + 'bidirectional' )

   if pic.borderRouter:
      cmds.addCommand( cmdPrefix + 'border-router' )
   elif saveAll:
      cmds.addCommand( noCmdPrefix + 'border-router' )

   if pic.helloInterval != pic.helloIntervalDefault or saveAll:
      cmds.addCommand( cmdPrefix + 'hello interval %s' % pic.helloInterval )

   if pic.helloCount != pic.helloCountDefault or saveAll:
      token = 'query-count %s' if legacy else 'hello count %s'
      cmds.addCommand( cmdPrefix + token % pic.helloCount )

   if pic.helloPriority != pic.helloPriorityDefault or saveAll:
      cmds.addCommand( cmdPrefix + 'dr-priority %s' % pic.helloPriority )

   if pic.joinPruneInterval != pic.joinPruneIntervalDefault or saveAll:
      token = 'join-prune-interval %s' if legacy else 'join-prune interval %s'
      cmds.addCommand( cmdPrefix + token % pic.joinPruneInterval )

   if pic.joinPruneCount != pic.joinPruneCountDefault or saveAll:
      token = 'join-prune-count %s' if legacy else 'join-prune count %s'
      cmds.addCommand( cmdPrefix + token % pic.joinPruneCount )

   if pic.neighborFilter:
      cmds.addCommand( cmdPrefix + 'neighbor filter %s' % pic.neighborFilter )
   elif saveAll:
      cmds.addCommand( 'no ' + cmdPrefix + 'neighbor filter %s' %
                       pic.neighborFilter )

   bfdToken = 'bfd-instance' if legacy else 'bfd'
   if pic.bfd == 'bfdEnabled':
      cmds.addCommand( cmdPrefix + bfdToken )
   elif pic.bfd == 'bfdDisabled':
      cmds.addCommand( 'no ' + cmdPrefix + bfdToken )
   elif saveAll:
      cmds.addCommand( 'default ' + cmdPrefix + bfdToken )

def saveIntfAfOnlyConfig( pimIntfConfig, root, sysdbRoot, saveAll, saveAllDetail,
                          pimLegacyConfig, af ):
   ''' Commands with have only  "pim ip[4|6]" format'''
   if pimLegacyConfig.version < PimLegacyConfig.ipMode:
      # No harm in saving new format commands when calling saveAll
      if not( saveAll or saveAllDetail ):
         return

   pic = pimIntfConfig
   mode = root[ IntfConfigMode ].getOrCreateModeInstance( pic.intfId )
   cmds = mode[ 'Pim.intf' ]

   assert af in [ AddressFamily.ipv4, AddressFamily.ipv6 ]
   if af == AddressFamily.ipv4:
      afStr = "ipv4 "
   else:
      afStr = "ipv6 "
   # TODO: When in afCommon mode afStr = ""
   cmdPrefix = "pim " + afStr
   noCmdPrefix = "no  " + cmdPrefix
   # join-prune transport
   if not Toggles.PimToggleLib.togglePimIpv6PortEnabled() and \
          af == AddressFamily.ipv6:
      return
   if pic.portCapabilityFlags.sctp:
      cmds.addCommand( cmdPrefix + 'join-prune transport sctp' )
   elif saveAll:
      cmds.addCommand( noCmdPrefix + 'join-prune transport sctp' )

   if af == AddressFamily.ipv4:
      if pic.localIntfId != '':
         cmds.addCommand( cmdPrefix + 'local-interface ' + pic.localIntfId )
      elif saveAll:
         cmds.addCommand( noCmdPrefix + 'local-interface' )

      if pic.fastFailover:
         cmds.addCommand( cmdPrefix + 'non-dr install-oifs' )
      elif saveAll:
         cmds.addCommand( noCmdPrefix + 'non-dr install-oifs' )


def _sortedByIp( coll ):
   return sorted( coll, cmp=IpUtils.compareIpAddress,
                  key= lambda x: x.ipAddr )

def _routeKeyPrefixTuple( rk ):
   # rk is a Routing::Multicast::Static::RouteKeyPrefix
   TupleIpAddr = Tac.newInstance( 'Arnet::IpAddr', 0 )
   TupleIpAddr.stringValue = rk.v4Prefix.address
   IpAddrValue = TupleIpAddr.value
   return ( IpAddrValue, rk.v4Prefix.len )

def _rpfKeyTuple( rk ):
   # vk is a Routing::Multicast::Static::Rpf
   return ( rk.intfId, rk.address )

PimVrfConfig = Tac.Type( "Routing::Pim::ConfigColl" )

# pylint: disable-msg=E0102
@CliSave.saver( 'Routing::Pim::ConfigColl',
      PimVrfConfig.mountPath( AddressFamily.ipv4 ),
      requireMounts = ( 'routing/pim/pimlegacyconfig',
                        'routing/pim/legacyconfig',
                        'routing/pim/bidir/legacyconfig',
                        'routing/hardware/status',
                        'routing/pim/application/config',
                        'interface/config/all',
                        'interface/status/all',
                        'acl/cpconfig/cli' ) )
def saveConfig( pimConfigRoot, root, sysdbRoot, options, requireMounts ):
   saveAll = options.saveAll
   saveAllDetail = options.saveAllDetail
   pimLegacyConfig = requireMounts[ 'routing/pim/legacyconfig' ]
   pimBidirLegacyConfig = requireMounts[ 'routing/pim/bidir/legacyconfig' ]
   # Save the default config only if the platform supports multicast routing
   if not McastCommonCliLib.mcastRoutingSupported(
         sysdbRoot,
         requireMounts[ 'routing/hardware/status' ] ):
      saveAll = False
      saveAllDetail = False
   pimLegacyVersion = getCliSaveVersion( pimLegacyConfig.version, saveAll,
                                         pimConfigRoot.isDefault() )
   pimBidirLegacyVersion = getCliSaveVersion( pimBidirLegacyConfig.version, saveAll,
                                              pimConfigRoot.bidirConfigIsDefault() )

   if saveAllDetail:
      cfgIntfNames = allRoutingProtocolIntfNames( sysdbRoot, includeEligible=True,
                                                  requireMounts=requireMounts )
   elif saveAll:
      # Routing configuration is allowed on switchports as well.
      # Save configuration on all routing protocol interfaces and switchports
      # with non-default config.
      cfgIntfNames = set(
            allRoutingProtocolIntfNames( sysdbRoot, requireMounts=requireMounts ) +
            pimConfigRoot.intfConfig.keys() )
   else:
      cfgIntfNames = pimConfigRoot.intfConfig

   for intfId in cfgIntfNames:
      intfConfig = pimConfigRoot.intfConfig.get( intfId )
      if not intfConfig:
         if saveAll:
            intfConfig = Tac.newInstance( 'Routing::Pim::IntfConfig', intfId )
         else:
            continue
      saveIntfConfigConverted( intfConfig, root, sysdbRoot, saveAll, saveAllDetail,
                               pimLegacyConfig, AddressFamily.ipv4 )
      saveIntfAfOnlyConfig( intfConfig, root, sysdbRoot, saveAll, saveAllDetail,
                            pimLegacyConfig, AddressFamily.ipv4 )

   if pimLegacyVersion != pimLegacyConfig.ipMode:
      if pimConfigRoot.globalConfig.drNotifyDelay:
         if pimConfigRoot.globalConfig.drNotifyDelayTime != \
               pimConfigRoot.globalConfig.drNotifyDelayDefault:
            root[ IpPimCmdSeq ].addCommand( 'ip pim dr-notify-delay %s' %\
                             pimConfigRoot.globalConfig.drNotifyDelayTime )
      else:
         root[ IpPimCmdSeq ].addCommand( 'ip pim dr-notify-delay -%s' %\
                          pimConfigRoot.globalConfig.drNotifyDelayTime )
   else:
      if pimConfigRoot.globalConfig.drNotifyDelay:
         if pimConfigRoot.globalConfig.drNotifyDelayTime != \
               pimConfigRoot.globalConfig.drNotifyDelayDefault:
            saveRoot = root[ RouterPimSparseBaseConfigMode ].getSingletonInstance() \
                  [ 'Pim.config' ]
            saveRoot.addCommand( 'dr-notify-delay %s' %\
                             pimConfigRoot.globalConfig.drNotifyDelayTime )
      else:
         saveRoot = root[ RouterPimSparseBaseConfigMode ].getSingletonInstance() \
               [ 'Pim.config' ]
         saveRoot.addCommand( 'dr-notify-delay -%s' %\
                          pimConfigRoot.globalConfig.drNotifyDelayTime )

   for vrfName, pimConfig in pimConfigRoot.vrfConfig.iteritems():
      if not pimConfig.isDefault() or saveAll:
         saveRoot = getCmdRoot( root, vrfName, AddressFamily.ipv4, pimLegacyVersion )

         for saver in sparseVrfCliSavers:
            saver( pimConfig, saveRoot, saveAll, pimLegacyVersion,
                   AddressFamily.ipv4 )

   routingHwStatus = requireMounts[ 'routing/hardware/status' ]
   if routingHwStatus.pimBidirectionalSupported:
      for vrfName, pimBidirConfig in pimConfigRoot.bidirVrfConfig.iteritems():
         if not pimBidirConfig.isDefault() or saveAll:
            saveRoot = getCmdRootBidir( root, vrfName, AddressFamily.ipv4,
                                        pimBidirLegacyVersion )
            for saver in bidirVrfCliSavers:
               saver( pimBidirConfig, saveRoot, saveAll, pimBidirLegacyVersion,
                      AddressFamily.ipv4 )

   PimCliSaveLib.savePimServiceAclConfig( root, options, requireMounts )

@CliSave.saver( 'Routing::Pim::Static::RpConfigColl',
                'routing/pim/staticRpConfig/sparsemode',
                requireMounts = ( 'routing/pim/legacyconfig',
                                  'routing/hardware/status', ) )
def saveConfig( pimConfigColl, root, sysdbRoot, options, requireMounts ):
   saveAll = options.saveAll
   if not McastCommonCliLib.mcastRoutingSupported(
         sysdbRoot,
         requireMounts[ 'routing/hardware/status' ] ):
      saveAll = False
   PimCliSaveLib.saveConfig( pimConfigColl, root, sysdbRoot, saveAll,
                             requireMounts, AddressFamily.ipv4 )

@CliSave.saver( 'Routing::Pim::Static::RpConfigColl',
                'routing6/pim/staticRpConfig/sparsemode',
                requireMounts = ( 'routing/pim/legacyconfig',
                                  'routing6/hardware/status', ) )
def saveConfig6( pimConfigColl, root, sysdbRoot, options, requireMounts ):
   saveAll = options.saveAll
   if not McastCommonCliLib.mcast6RoutingSupported(
         sysdbRoot,
         requireMounts[ 'routing6/hardware/status' ] ):
      saveAll = False
   PimCliSaveLib.saveConfig( pimConfigColl, root, sysdbRoot, saveAll,
                             requireMounts, AddressFamily.ipv6 )

@CliSave.saver( 'Routing::Pim::Static::RpConfigColl',
                'routing/pim/staticRpConfig/bidir',
                requireMounts=( 'routing/hardware/status',
                                'routing/pim/config',
                                'routing/pim/bidir/legacyconfig' ) )
def savePimBidirConfig( pimConfigColl, root, sysdbRoot, options, requireMounts ):
   PimCliSaveLib.savePimBidirConfig( pimConfigColl, root, sysdbRoot, options,
                             requireMounts )

@CliSave.saver( 'Routing::Pim::RpConfigColl',
                'routing/pim/rpConfig/sparsemode',
                 requireMounts = ( 'routing/pim/legacyconfig',
                                   'routing/hardware/status', ) )
def saveRpConfig( rpConfigColl, root, sysdbRoot, options,
                requireMounts ):
   legacyConfig = requireMounts[ 'routing/pim/legacyconfig' ]
   legacyVersion = PimLegacyConfig.ipMode \
               if CliParser.DEPRECATED_CMD_ALLOWED is False else legacyConfig.version
   PimCliSaveLib.saveRpConfig( AddressFamily.ipv4, rpConfigColl, root, sysdbRoot,
         options, requireMounts, legacyVersion )

@CliSave.saver( 'Routing::Pim::RpConfigColl',
                'routing6/pim/rpConfig/sparsemode',
                 requireMounts = ( 'routing/pim/legacyconfig',
                                   'routing6/hardware/status', ) )
def saveRpConfig6( rpConfigColl, root, sysdbRoot, options,
                requireMounts ):
   legacyConfig = requireMounts[ 'routing/pim/legacyconfig' ]
   legacyVersion = PimLegacyConfig.ipMode \
               if CliParser.DEPRECATED_CMD_ALLOWED is False else legacyConfig.version
   PimCliSaveLib.saveRpConfig( AddressFamily.ipv6, rpConfigColl, root, sysdbRoot,
         options, requireMounts, legacyVersion )

@CliSave.saver( 'Routing::Pim::RpConfigColl',
                'routing/pim/rpConfig/bidir',
                requireMounts = ( 'routing/hardware/status',
                                  'routing/pim/bidir/legacyconfig',  ) )
def savePimBidirRpConfig( rpConfigColl, root, sysdbRoot, options,
                requireMounts ):
   legacyConfig = requireMounts[ 'routing/pim/bidir/legacyconfig' ]
   legacyVersion = PimLegacyConfig.ipMode \
               if CliParser.DEPRECATED_CMD_ALLOWED is False else legacyConfig.version
   PimCliSaveLib.savePimBidirRpConfig( rpConfigColl, root, sysdbRoot, options,
                             requireMounts, legacyVersion )

@CliSave.saver( 'Routing::Pim::ConfigColl',
                PimVrfConfig.mountPath( AddressFamily.ipv6 ),
                requireMounts = ( 'routing6/hardware/status',
                                  'routing/pim/legacyconfig',
                                  'routing6/pim/application/config',
                                  'interface/config/all',
                                  'interface/status/all', ) )
def saveIpv6RouterPimSparseConfig( pimConfigRoot, root, sysdbRoot,
               options, requireMounts ):
   hardwareStatus = requireMounts[ 'routing6/hardware/status' ]
   legacyConfig = requireMounts[ 'routing/pim/legacyconfig' ]

   saveAll = options.saveAll
   saveAllDetail = options.saveAllDetail

   if saveAllDetail:
      cfgIntfNames = allRoutingProtocolIntfNames( sysdbRoot, includeEligible=True,
                                                  requireMounts=requireMounts )
   elif saveAll:
      # Routing configuration is allowed on switchports as well.
      # Save configuration on all routing protocol interfaces and switchports
      # with non-default config.
      cfgIntfNames = set(
            allRoutingProtocolIntfNames( sysdbRoot, requireMounts=requireMounts ) +
            pimConfigRoot.intfConfig.keys() )
   else:
      cfgIntfNames = pimConfigRoot.intfConfig

   version = legacyConfig.version

   for intfId in cfgIntfNames:
      intfConfig = pimConfigRoot.intfConfig.get( intfId )
      if not intfConfig:
         if saveAll:
            intfConfig = Tac.newInstance( 'Routing::Pim::IntfConfig', intfId )
         else:
            continue
      saveIntfConfigConverted( intfConfig, root, sysdbRoot, saveAll, saveAllDetail,
                               legacyConfig, AddressFamily.ipv6 )
      saveIntfAfOnlyConfig( intfConfig, root, sysdbRoot, saveAll, saveAllDetail,
                            legacyConfig, AddressFamily.ipv6 )


   if version < legacyConfig.ipMode:
      return

   saveAll = options.saveAll
   if not McastCommonCliLib.mcast6RoutingSupported( sysdbRoot, hardwareStatus ):
      saveAll = False

   for vrfName, ipv6VrfConf in pimConfigRoot.vrfConfig.iteritems():
      if not ipv6VrfConf.isDefault() or saveAll:
         cmds = PimCliSaveLib.getCmdRoot( root, vrfName, AddressFamily.ipv6,
                                          version )
         for saver in sparseVrfCliSavers:
            saver( ipv6VrfConf, cmds, saveAll, version, AddressFamily.ipv6 )

@CliSave.saver( 'Routing::Pim::ConfigColl',
                PimVrfConfig.mountPath( AddressFamily.ipv6 ),
                requireMounts = ( 'routing6/hardware/status',
                                  'routing/pim/bidir/legacyconfig',
                                  'routing6/pim/application/config',
                                  'interface/config/all',
                                  'interface/status/all', ) )
def saveIpv6RouterPimBidirConfig( pimConfigRoot, root, sysdbRoot,
               options, requireMounts ):
   routingHwStatus = requireMounts[ 'routing6/hardware/status' ]
   legacyConfig = requireMounts[ 'routing/pim/bidir/legacyconfig' ]
   saveAll = options.saveAll

   if legacyConfig.version < legacyConfig.ipMode:
      return

   # Save the default config only if the platform supports multicast routing
   if not McastCommonCliLib.mcast6RoutingSupported(
                              sysdbRoot, routingHwStatus ):
      saveAll = False

   if routingHwStatus.pimBidirectionalSupported:
      for vrfName, vrfConfig in pimConfigRoot.bidirVrfConfig.iteritems():
         if not vrfConfig.isDefault() or saveAll:
            cmds = PimCliSaveLib.getCmdRootBidir( root, vrfName, AddressFamily.ipv6,
                  legacyConfig.version )
            for saver in bidirVrfCliSavers:
               saver( vrfConfig, cmds, saveAll, legacyConfig.version,
                      AddressFamily.ipv6 )
