#!/usr/bin/env python
# Copyright (c) 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#pylint disable-msg=F0401
import CliSave, IpUtils
from IntfCliSave import IntfConfigMode
import Tac, McastCommonCliLib
from McastCommonCliLib import defaultMcastPrefix, defaultMcastPrefixStr
from RoutingIntfUtils import allRoutingProtocolIntfNames
from IpLibConsts import DEFAULT_VRF as vrfDefault
from CliSavePlugin.PimCliSaveLib import RouterPimBidirBaseConfigMode, \
      RouterPimBidirVrfConfigMode, RouterPimBidirAfConfigMode
from CliSavePlugin.PimCliSaveLib import RouterPimSparseBaseConfigMode, \
      RouterPimSparseVrfConfigMode, RouterPimSparseAfConfigMode
from CliSavePlugin.MrouteCliSave import getCliSaveVersion
from CliMode.PimBsr import RoutingPimBsrMode, RoutingPimBsrVrfMode, \
      RoutingPimBsrAfMode
import Tracing
__defaultTraceHandle__ = Tracing.Handle( "PimBsrSave" )
t1 = Tracing.trace1

# xxx Not certain where in the intf output this oughta go or if it matters
IntfConfigMode.addCommandSequence( 'PimBsr.intf', after=[ 'Ira.ipIntf' ] )
AddressFamily = Tac.Type( "Arnet::AddressFamily" )
PimBsrLegacyConfig = Tac.Type( "McastCommon::LegacyConfig" )

class RouterPimBsrBaseConfigMode( RoutingPimBsrMode, CliSave.Mode ):
   def __init__( self, param ):
      RoutingPimBsrMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

class RouterPimBsrVrfConfigMode( RoutingPimBsrVrfMode, CliSave.Mode ):
   def __init__( self, param ):
      RoutingPimBsrVrfMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

class RouterPimBsrAfConfigMode( RoutingPimBsrAfMode, CliSave.Mode ):
   def __init__( self, ( vrfName, af ) ):
      RoutingPimBsrAfMode.__init__( self, vrfName, af )
      CliSave.Mode.__init__( self, ( vrfName, af ) )

   def skipIfEmpty( self ):
      return True

def getPimBsrCmdSequence( root, vrfName, af, pimBsrLegacyVersion ):
   def getRouterMode():
      return root[ RouterPimBsrBaseConfigMode ].getSingletonInstance()

   def getRouterVrfMode( vrfName ):
      routerMode = getRouterMode()
      return routerMode[ RouterPimBsrVrfConfigMode ].getOrCreateModeInstance(
                                                         vrfName )
   def getIpMode( vrfName, af ):
      if vrfName == vrfDefault:
         parentMode = getRouterMode()
      else:
         parentMode = getRouterVrfMode( vrfName )
      return parentMode[ RouterPimBsrAfConfigMode ].getOrCreateModeInstance(
                                                         ( vrfName, af ) )

   if pimBsrLegacyVersion == PimBsrLegacyConfig.ipMode:
      cmdSeq = getIpMode( vrfName, af )[ 'PimBsr.vrf.af.config' ]
   elif af == AddressFamily.ipv4:
      if vrfName == vrfDefault:
         if pimBsrLegacyVersion == PimBsrLegacyConfig.globalMode:
            cmdSeq = root[ 'Ip.PimBsr' ]
         else:
            cmdSeq = getRouterMode()[ 'PimBsr.config' ]
      else:
         cmdSeq = getRouterVrfMode( vrfName )[ 'PimBsr.vrf.config' ]
   elif af == AddressFamily.ipv6:
      if vrfName == vrfDefault:
         cmdSeq = getRouterMode()[ 'PimBsr.config' ]
      else:
         cmdSeq = getRouterVrfMode( vrfName )[ 'PimBsr.vrf.config' ]

   return cmdSeq

# Save attributes in Routing::Pim::Bsr::VrfConfig
pimBsrRootCliSavers = []
def pimBsrConfigSavers( func ):
   pimBsrRootCliSavers.append( func )

@pimBsrConfigSavers
def saveBsrHoldtime( config, cmds, saveAll, legacyVersion, af ):
   cmd = 'holdtime'
   if legacyVersion != PimBsrLegacyConfig.ipMode:
      cmd = 'ip pim bsr-' + cmd

   if config.holdTime != config.holdTimeDefault:
      cmds.addCommand( "%s %d" % ( cmd, config.holdTime ) )
   elif saveAll:
      cmds.addCommand( 'no ' + cmd )

@pimBsrConfigSavers
def saveBsrSzTimeout( config, cmds, saveAll, legacyVersion, af ):
   cmd = 'sztimeout'
   if legacyVersion != PimBsrLegacyConfig.ipMode:
      cmd = 'ip pim bsr-' + cmd

   if config.szTimeout != config.szTimeoutDefault:
      cmds.addCommand( "%s %d" % ( cmd, config.szTimeout ) )
   elif saveAll:
      cmds.addCommand( 'no ' + cmd )

@pimBsrConfigSavers
def saveBsrCandidate( config, cmds, saveAll, legacyVersion, af ):
   candidate = 'candidate'
   if legacyVersion != PimBsrLegacyConfig.ipMode:
      baseCmd = 'ip pim bsr-%s' % candidate
   else:
      baseCmd = candidate

   if config.bsrCandidateConfig:
      for intfId in config.bsrCandidateConfig:
         bsrc = config.bsrCandidateConfig[ intfId ]
         cmd = '%s %s' % ( baseCmd, intfId )
         for g in bsrc.group:
            if g.stringValue == defaultMcastPrefixStr( af ):
               cmdg = '%s priority %s hashmask %s interval %s' % \
                   ( cmd, str( bsrc.group[ g ].priority ),
                     str( bsrc.group[ g ].hashMaskLen ),
                     str( bsrc.group[ g ].interval ) )
            else:
               cmdg = '%s %s priority %s hashmask %s interval %s' % \
                   ( cmd, g.stringValue, str( bsrc.group[ g ].priority ),
                     str( bsrc.group[ g ].hashMaskLen ),
                     str( bsrc.group[ g ].interval ) )
            cmds.addCommand( cmdg )

         for acl in bsrc.acl:
            cmda = '%s access-list %s' % ( cmd, acl )
            cmda = '%s priority %s hashmask %s interval %s' % \
                ( cmda, str( bsrc.acl[ acl ].priority ),
                 str( bsrc.acl[ acl ].hashMaskLen ),
                 str( bsrc.acl[ acl ].interval ) )
            cmds.addCommand( cmda )

   elif saveAll:
      cmd = 'no %s' % baseCmd
      cmds.addCommand( cmd )

@pimBsrConfigSavers
def saveBsrCrpFilter( config, cmds, saveAll, legacyVersion, af ):
   if legacyVersion != PimBsrLegacyConfig.ipMode:
      baseCmd = 'ip pim bsr rp-candidate advertisement-filter'
   else:
      baseCmd = 'rp-candidate advertisement-filter'

   if config.crpFilterAcl:
      for acl in config.crpFilterAcl.keys():
         cmd = '%s access-list %s' % ( baseCmd, acl )
         cmds.addCommand( cmd )

   elif saveAll:
      cmd = 'no %s' % baseCmd
      cmds.addCommand( cmd )

IpPimBsrCmdSeq = 'Ip.PimBsr'
CliSave.GlobalConfigMode.addCommandSequence( IpPimBsrCmdSeq,
                                             before=[ IntfConfigMode ] )

CliSave.GlobalConfigMode.addChildMode( RouterPimBsrBaseConfigMode,
                                          after=[ RouterPimBidirBaseConfigMode ] )

RouterPimBsrBaseConfigMode.addChildMode( RouterPimBsrVrfConfigMode )
RouterPimBsrBaseConfigMode.addChildMode( RouterPimBsrAfConfigMode )
RouterPimBsrVrfConfigMode.addChildMode( RouterPimBsrAfConfigMode )

RouterPimBsrBaseConfigMode.addCommandSequence( 'PimBsr.config' )
RouterPimBsrVrfConfigMode.addCommandSequence( 'PimBsr.vrf.config' )
RouterPimBsrAfConfigMode.addCommandSequence( 'PimBsr.vrf.af.config' )

RouterPimBidirBaseConfigMode.addChildMode( RouterPimBidirVrfConfigMode )
RouterPimBidirBaseConfigMode.addChildMode( RouterPimBidirAfConfigMode )
RouterPimBidirVrfConfigMode.addChildMode( RouterPimBidirAfConfigMode )

RouterPimBidirBaseConfigMode.addCommandSequence( 'PimBsr.config' )
RouterPimBidirVrfConfigMode.addCommandSequence( 'PimBsr.vrf.config' )
RouterPimBidirAfConfigMode.addCommandSequence( 'PimBsr.vrf.af.config' )

RouterPimSparseBaseConfigMode.addChildMode( RouterPimSparseVrfConfigMode )
RouterPimSparseBaseConfigMode.addChildMode( RouterPimSparseAfConfigMode )
RouterPimSparseVrfConfigMode.addChildMode( RouterPimSparseAfConfigMode )

RouterPimSparseBaseConfigMode.addCommandSequence( 'PimBsr.config' )
RouterPimSparseVrfConfigMode.addCommandSequence( 'PimBsr.vrf.config' )
RouterPimSparseAfConfigMode.addCommandSequence( 'PimBsr.vrf.af.config' )

# pylint: disable-msg=W0612
def saveIntfConfig( pimBsrIntfConfig, root, sysdbRoot, saveAll, saveAllDetail,
                    legacyVersion, af=AddressFamily.ipv4 ):
   pic = pimBsrIntfConfig

   afStr = 'ipv4' if af == AddressFamily.ipv4 else 'ipv6'
   baseCmd = 'pim bsr %s border' % afStr

   t1( "Processing interface", pic.name, ", af", afStr )
   mode = root[ IntfConfigMode ].getOrCreateModeInstance( pic.name )
   cmds = mode[ 'PimBsr.intf' ]

   for g in pic.bsrBorderGrp:
      if pic.bsrBorderGrp[ g ]:
         if g != defaultMcastPrefix( af ):
            cmds.addCommand( "%s %s" %( baseCmd, g.stringValue ) )
         else:
            cmds.addCommand( baseCmd )
   for acl in pic.bsrBorderAcl:
      if pic.bsrBorderAcl[ acl ]:
         cmds.addCommand( '%s access-list %s' % ( baseCmd, acl ) )

   if not pic.bsrBorderGrp and not pic.bsrBorderAcl and saveAll:
      cmds.addCommand( 'no %s' % baseCmd )

def _sortedByIp( coll ):
   return sorted( coll, cmp=IpUtils.compareIpAddress,
                  key= lambda x: x.ipAddr )

rpPriority = Tac.Value( "Routing::Pim::RpPriority" )
rpIntvlDefault = Tac.Value( "Routing::Pim::Bsr::CrpAdvInterval" )

@CliSave.saver( 'Routing::Pim::Bsr::ConfigColl', 'routing/pim/bsr/configcoll',
                requireMounts = ( 'routing/pim/bsr/legacyconfig',
                                  'routing/pim/legacyconfig',
                                  'routing/pim/config',
                                  'routing/hardware/status',
                                  'interface/config/all',
                                  'interface/status/all' ) )
def saveIpv4PimBsrConfig( pimBsrConfigColl, root, sysdbRoot, options,
                          requireMounts ):
   pimConfigRoot = requireMounts[ 'routing/pim/config' ]
   routingHwStatus = requireMounts[ 'routing/hardware/status' ]
   saveConfig( pimBsrConfigColl, AddressFamily.ipv4, root, sysdbRoot, options,
               requireMounts, pimConfigRoot, routingHwStatus )

@CliSave.saver( 'Routing::Pim::Bsr::ConfigColl', 'routing6/pim/bsr/configcoll',
                requireMounts = ( 'routing/pim/bsr/legacyconfig',
                                  'routing/pim/legacyconfig',
                                  'routing6/pim/config',
                                  'routing6/hardware/status',
                                  'interface/config/all',
                                  'interface/status/all' ) )
def saveIpv6PimBsrConfig( pimBsrConfigColl, root, sysdbRoot, options,
                          requireMounts ):
   pimConfigRoot = requireMounts[ 'routing6/pim/config' ]
   routingHwStatus = requireMounts[ 'routing6/hardware/status' ]
   # Save the default config only if the platform supports multicast routing
   saveConfig( pimBsrConfigColl, AddressFamily.ipv6, root, sysdbRoot, options,
               requireMounts, pimConfigRoot, routingHwStatus )

def saveConfig( pimBsrConfigColl, af, root, sysdbRoot, options, requireMounts,
                pimConfigRoot, routingHwStatus ):
   saveAll = options.saveAll
   saveAllDetail = options.saveAllDetail
   # Save the default config only if the platform supports multicast routing
   if not McastCommonCliLib.mcastRoutingSupported(
         sysdbRoot,
         routingHwStatus ):
      saveAll = False
      saveAllDetail = False

   saveRpCandidateConfig( pimBsrConfigColl, af, root, saveAll, saveAllDetail,
                          requireMounts, pimConfigRoot, routingHwStatus )
   # Save per interface config
   if saveAllDetail:
      cfgIntfNames = allRoutingProtocolIntfNames( sysdbRoot, includeEligible=True,
                                                  requireMounts=requireMounts )
   elif saveAll:
      # Routing configuration is allowed on switchports as well.
      # Save configuration on all routing protocol interfaces and switchports
      # with non-default config.
      cfgIntfNames = set(
            allRoutingProtocolIntfNames( sysdbRoot, requireMounts=requireMounts ) +
            pimBsrConfigColl.intfConfig.keys() )
   else:
      cfgIntfNames = pimBsrConfigColl.intfConfig

   pimBsrLegacyConfig = requireMounts[ 'routing/pim/bsr/legacyconfig' ]
   legacyVersion = getCliSaveVersion( pimBsrLegacyConfig.version, saveAll,
                                      pimBsrConfigColl.isDefault() )

   for intfName in cfgIntfNames:
      intfConfig = pimBsrConfigColl.intfConfig.get( intfName )
      if not intfConfig:
         if saveAll:
            intfConfig = Tac.newInstance( 'Routing::Pim::Bsr::IntfConfig',
                                       intfName )
         else:
            continue
      saveIntfConfig( intfConfig, root, sysdbRoot, saveAll, saveAllDetail,
            legacyVersion, af )

   for vrfName, pimBsrConfig in pimBsrConfigColl.vrfConfig.iteritems() :
      t1( "saveConfig processing Vrf", vrfName, ", af", af )

      # Converted commands
      cmdSeq = getPimBsrCmdSequence( root, vrfName, af, legacyVersion )

      for configSaver in pimBsrRootCliSavers:
         configSaver( pimBsrConfig, cmdSeq, saveAll, legacyVersion, af )

      t1( "Finished processing Vrf" )

def saveRpCandidateConfig( pimBsrConfigColl, af, root, saveAll, saveAllDetail,
      requireMounts, pimConfigRoot, routingHwStatus ):
   pimLegacyConfig = requireMounts[ 'routing/pim/legacyconfig' ]

   pimLegacyVersion = getCliSaveVersion( pimLegacyConfig.version, saveAll,
                                         pimConfigRoot.isDefault() )

   for vrfName in pimBsrConfigColl.vrfConfig :

      t1( "saveRpCandidateConfig Processing Vrf", vrfName, ", af", af )
      config = pimBsrConfigColl.vrfConfig.get( vrfName )

      saveVrfRpCandidateConfig( vrfName, af, config, root, saveAll, pimLegacyVersion,
                                routingHwStatus.pimBidirectionalSupported )
      t1( "Finished processing Vrf" )

def getPimsmCmdSequence( root, vrfName, af, pimLegacyVersion ):
   def getRouterMode():
      return root[ RouterPimSparseBaseConfigMode ].getSingletonInstance()

   def getRouterVrfMode( vrfName ):
      routerMode = getRouterMode()
      return routerMode[ RouterPimSparseVrfConfigMode ].getOrCreateModeInstance(
            vrfName )

   def getIpMode( vrfName, af ):
      if vrfName == vrfDefault:
         parentMode = getRouterMode()
      else:
         parentMode = getRouterVrfMode( vrfName )
      return parentMode[ RouterPimSparseAfConfigMode ].getOrCreateModeInstance(
            ( vrfName, af ) )

   if pimLegacyVersion == PimBsrLegacyConfig.ipMode:
      return getIpMode( vrfName, af )[ 'PimBsr.vrf.af.config' ]
   elif af == AddressFamily.ipv4:
      if vrfName == vrfDefault:
         if pimLegacyVersion == PimBsrLegacyConfig.globalMode:
            return root[ 'Ip.PimBsr' ]
         else:
            return getRouterMode()[ 'PimBsr.config' ]
      else:
         return getRouterVrfMode( vrfName )[ 'PimBsr.vrf.config' ]
   elif af == AddressFamily.ipv6:
      if vrfName == vrfDefault:
         return getRouterMode()[ 'PimBsr.config' ]
      else:
         return getRouterVrfMode( vrfName )[ 'PimBsr.vrf.config' ]

def getPimBidirCmdSequence( root, vrfName, af, pimLegacyVersion ):
   def getRouterMode():
      return root[ RouterPimBidirBaseConfigMode ].getSingletonInstance()

   def getRouterVrfMode( vrfName ):
      routerMode = getRouterMode()
      return routerMode[ RouterPimBidirVrfConfigMode ].getOrCreateModeInstance(
                                                         vrfName )
   def getIpMode( vrfName, af ):
      if vrfName == vrfDefault:
         parentMode = getRouterMode()
      else:
         parentMode = getRouterVrfMode( vrfName )
      return parentMode[ RouterPimBidirAfConfigMode ].getOrCreateModeInstance(
                                                         ( vrfName, af ) )

   if pimLegacyVersion == PimBsrLegacyConfig.ipMode:
      return getIpMode( vrfName, af )[ 'PimBsr.vrf.af.config' ]
   elif af in [ AddressFamily.ipv4, AddressFamily.ipv6 ]:
      if vrfName == vrfDefault:
         return getRouterMode()[ 'PimBsr.config' ]
      else:
         return getRouterVrfMode( vrfName )[ 'PimBsr.vrf.config' ]

def saveVrfRpCandidateConfig( vrfName, af, pimBsrConfig, root, saveAll,
      pimLegacyVersion, bidirSupported ):

   if pimBsrConfig.rpCandidateConfig:
      for intfId in pimBsrConfig.rpCandidateConfig:

         if pimLegacyVersion == PimBsrLegacyConfig.ipMode:
            cmd = 'rp candidate'
         else:
            cmd = 'ip pim rp-candidate'

         rpc = pimBsrConfig.rpCandidateConfig[ intfId ]
         cmd = '%s %s' % ( cmd, intfId )

         if rpc.mode == 'modePimBidir' and not bidirSupported:
            continue

         for g in rpc.groupRange:
            if g != defaultMcastPrefix( af ) or saveAll:
               cmdg = '%s %s' % ( cmd, g.stringValue )
            else:
               cmdg = cmd

            if rpc.groupRangeOptions[ g ].priority != rpPriority.priorityBsrDefault:
               cmdg = '%s priority %s' % ( cmdg,
                     str( rpc.groupRangeOptions[ g ].priority ) )

            if rpc.advInterval != rpIntvlDefault.intervalDefault:
               cmdg = '%s interval %s' % ( cmdg, str( rpc.advInterval ) )

            if rpc.mode == 'modePimSm':
               getPimsmCmdSequence( root, vrfName, af,
                     pimLegacyVersion ).addCommand( cmdg )
            else:
               getPimBidirCmdSequence( root, vrfName,
                     af, pimLegacyVersion ).addCommand( cmdg )

         for acl in rpc.acl:
            cmda = '%s access-list %s' % ( cmd, acl )

            if rpc.aclOptions[ acl ].priority != rpPriority.priorityBsrDefault:
               cmda = '%s priority %s' % ( cmda,
                     str( rpc.aclOptions[ acl ].priority ) )

            if rpc.advInterval != rpIntvlDefault.intervalDefault:
               cmda = '%s interval %s' % ( cmda, str( rpc.advInterval ) )

            if rpc.mode == 'modePimSm':
               getPimsmCmdSequence( root, vrfName, af,
                     pimLegacyVersion ).addCommand( cmda )
            else:
               getPimBidirCmdSequence( root, vrfName,
                     af, pimLegacyVersion ).addCommand( cmda )

   elif saveAll:
      if pimLegacyVersion == PimBsrLegacyConfig.ipMode:
         cmd = 'no rp candidate'
      else:
         cmd = 'no ip pim rp-candidate'
      getPimsmCmdSequence( root, vrfName, af, pimLegacyVersion ).addCommand( cmd )

      if bidirSupported:
         getPimBidirCmdSequence( root, vrfName,
               af, pimLegacyVersion ).addCommand( cmd )

