# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave
from CliMode.OpenStack import (
      OpenStackConfigModeBase,
      RegionConfigModeBase,
   )
from CliSavePlugin.Controllerdb import ( CvxConfigMode,
                                         controllerConfigPath,
                                         getClusterName )
from OpenStackLib import coalesceList
from IpLibConsts import DEFAULT_VRF
import Tac

class OpenStackConfigSaveMode( OpenStackConfigModeBase, CliSave.Mode ):
   def __init__( self, param ):
      OpenStackConfigModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class OpenStackRegionConfigSaveMode( RegionConfigModeBase, CliSave.Mode ):
   def __init__( self, param ):
      RegionConfigModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CvxConfigMode.addChildMode( OpenStackConfigSaveMode,
                            after=[ 'cvx' ] )
OpenStackConfigSaveMode.addCommandSequence(
   'openstack.config',
   before=[ OpenStackRegionConfigSaveMode ]
)

OpenStackConfigSaveMode.addChildMode( OpenStackRegionConfigSaveMode )

OpenStackRegionConfigSaveMode.addCommandSequence( 'openstack.vlanToVniConfig' )
OpenStackRegionConfigSaveMode.addCommandSequence( 'openstack.credentials' )
OpenStackRegionConfigSaveMode.addCommandSequence( 'openstack.vlanAllocation' )
OpenStackRegionConfigSaveMode.addCommandSequence( 'openstack.keystoneAuthUrl' )
OpenStackRegionConfigSaveMode.addCommandSequence( 'openstack.mandatoryRegion' )

def getOpenStackMode( root, cvxClusterName ):
   mode = root[ CvxConfigMode ].getOrCreateModeInstance(
      CvxConfigMode.modeName( cvxClusterName ) )
   osMode = mode[ OpenStackConfigSaveMode ]
   osMode = osMode.getOrCreateModeInstance( 'cvx-openstack' )
   return osMode

def addCommandsToOpenStackMode( root, cmds, cvxClusterName ):
   if cmds:
      osMode = getOpenStackMode( root, cvxClusterName )
      osMode = osMode[ 'openstack.config' ]
      for cmd in cmds:
         osMode.addCommand( cmd )

def addCommandsToOpenStackRegionMode( root, cmdSeq, regionName, cmds,
                                      cvxClusterName ):
   if cmds:
      osMode = getOpenStackMode( root, cvxClusterName )
      osRegionMode = osMode[ OpenStackRegionConfigSaveMode ].getOrCreateModeInstance(
                        regionName )
      osRegionMode = osRegionMode[ cmdSeq ]

      for cmd in cmds:
         osRegionMode.addCommand( cmd )

# pylint: disable-msg=C0322
@CliSave.saver( 'OpenStack::Config', 'mgmt/openstack/config',
                requireMounts=( controllerConfigPath, ) )
def saveConfig( config, root, sysdbRoot, options, requireMounts ):
   cmds = []
   cvxClusterName = getClusterName( requireMounts[ controllerConfigPath ] )

   if config.enabled or options.saveAll:
      cmds.append( "%sshutdown" % ( config.enabled and "no " or "" ) )

   if config.gracePeriod != config.defaultGracePeriod or options.saveAll:
      cmds.append( "grace-period %i" % config.gracePeriod )

   if config.vlanTypeDriver != config.defaultVlanTypeDriver or options.saveAll:
      typeDriverToCli = { 'defaultVlan' : 'default',
                          'aristaVlan' : 'arista' }
      cmds.append( "network type-driver vlan %s" %
                   typeDriverToCli.get( config.vlanTypeDriver, 'default' ) )

   if config.authRole != "":
      cmds.append( "authentication role %s" % config.authRole )

   addCommandsToOpenStackMode( root, cmds, cvxClusterName )

   for credentials in config.credentials.values():
      cmds = []
      cmd = 'username %s tenant %s password 7 %s' % \
      ( credentials.username,
        credentials.tenantName,
        CliSave.sanitizedOutput( options, credentials.password ) )
      cmds.append( cmd )
      addCommandsToOpenStackRegionMode( root, 'openstack.credentials',
                                        credentials.name, cmds,
                                        cvxClusterName )

   for region in config.vlanAssignment.values():
      vlanAssignment = region.vlan.keys()
      vlanAssignment.sort()
      assignedVlanIds = coalesceList( vlanAssignment )
      cmd = [ 'resource-pool vlan %s' % ( assignedVlanIds ) ]
      addCommandsToOpenStackRegionMode( root, 'openstack.vlanAllocation',
                                        region.regionName, cmd, cvxClusterName )

   for region in config.keystoneAuthUrl.values():
      # port is necessary for backwards compatibility
      cmds = []
      cmd = 'keystone auth-url %s' % region.authUrl
      cmds.append( cmd )
      addCommandsToOpenStackRegionMode( root, 'openstack.keystoneAuthUrl',
                                        region.name, cmds,
                                        cvxClusterName )

   for region in config.mandatoryRegion.keys():
      addCommandsToOpenStackRegionMode( root, 'openstack.mandatoryRegion', region,
                                        [ 'provision sync mandatory' ],
                                        cvxClusterName )

@CliSave.saver( 'VirtualNetwork::Config', 'mgmt/virtualnetwork/config',
                requireMounts=( controllerConfigPath, ) )
def saveVirtualNetworkConfig( config, root, sysdbRoot, options, requireMounts ):
   cvxClusterName = getClusterName( requireMounts[ controllerConfigPath ] )
   cmds = []
   for vlanMapVal in config.vlanToVniMap.values():
      cmds = []
      for vlanRange, vniRange in vlanMapVal.map.items():
         if vlanRange.start != vlanRange.end:
            cmd = 'networks map vlan %d - %d vni %d - %d' % (
                     vlanRange.start,
                     vlanRange.end,
                     vniRange.start,
                     vniRange.end )
         else:
            cmd = 'networks map vlan %d vni %d' % (
                     vlanRange.start,
                     vniRange.start )
         cmds.append( cmd )

      addCommandsToOpenStackRegionMode( root, 'openstack.vlanToVniConfig',
                                        vlanMapVal.domainName.domain, cmds,
                                        cvxClusterName )

@CliSave.saver( 'OpenStack::NameResolution::Config',
                'mgmt/openstack/nameResolutionConfig',
                requireMounts=( controllerConfigPath, ) )
def saveNameResolutionConfig( config, root, sysdbRoot, options, requireMounts ):
   cmds = []

   if config.refreshPeriod != config.defaultRefreshPeriod or options.saveAll:
      cmds.append( 'name-resolution interval %s' % int( config.refreshPeriod ) )

   cvxClusterName = getClusterName( requireMounts[ controllerConfigPath ] )
   addCommandsToOpenStackMode( root, cmds, cvxClusterName )

@CliSave.saver( 'OpenStack::Config',
                'mgmt/openstack/config',
                requireMounts=( controllerConfigPath, ) )
def saveServiceAclConfig( config, root, sysdbRoot, options, requireMounts ):
   cmds = []
   for aclType in [ 'ip', 'ipv6' ]:
      # Only default vrf supported
      if config.serviceAclTypeVrfMap is None:
         continue
      key = Tac.Value( "Acl::AclTypeAndVrfName", aclType, DEFAULT_VRF )
      aclName = config.serviceAclTypeVrfMap.aclName.get( key )
      if aclName is None or aclName == '':
         continue
      cmds.append( '%s access-group %s' % ( aclType, aclName ) )
   cvxClusterName = getClusterName( requireMounts[ controllerConfigPath ] )
   addCommandsToOpenStackMode( root, cmds, cvxClusterName )
