#!/usr/bin/env python
# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave
import Intf
import Vlan
from CliMode.OpenFlow import OpenFlowMode, ControllerMode
from IntfCliSave import IntfConfigMode 
import Tac

class OpenFlowConfigMode( OpenFlowMode, CliSave.Mode ):
   def __init__( self, param ):
      OpenFlowMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class OpenFlowControllerConfigMode( ControllerMode, CliSave.Mode ):
   def __init__( self, param ):
      ControllerMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

# For using Canonical form of physical interfaces in running-config,
# it should appear after interface config mode.
CliSave.GlobalConfigMode.addChildMode( OpenFlowConfigMode,
                                       after = [ IntfConfigMode ] )
OpenFlowConfigMode.addCommandSequence( "OpenFlow" )

OpenFlowConfigMode.addChildMode( OpenFlowControllerConfigMode )
OpenFlowControllerConfigMode.addCommandSequence( "OfController.Config" )

@CliSave.saver( "OpenFlow::Config", "openflow/config" )
def saveOpenFlow( config, root, sysdbRoot, options ):
   saveAll = options.saveAll
   saveAllDetail = options.saveAllDetail
   saveDefault = saveAllDetail or ( saveAll and config.enabled )

   cmds = []
   if config.description:
      cmds.append( "description %s" % config.description )
   elif saveDefault:
      cmds.append( "no description" )

   if config.dpidType == Tac.Type( 'OpenFlow::DpidType' ).managementMac:
      cmds.append( "datapath id interface management1" )
   elif config.dpidType == Tac.Type( 'OpenFlow::DpidType' ).userDefined:
      cmds.append( "datapath id %d" % config.dpid )
   elif saveDefault:
      cmds.append( "no datapath id" )

   if int( config.keepalivePeriod ) != 10:
      cmds.append( "keepalive %d" % config.keepalivePeriod )
   elif saveDefault:
      cmds.append( "keepalive 10" )
   if config.forwardingPipeline == "flowAndNormal":
      cmds.append( "forwarding pipeline flow-and-normal" )
   else:
      if saveDefault:
         cmds.append( 'forwarding pipeline flow' )
      if config.bindMode == "bindModeVlan":
         cmds.append( "bind mode vlan" )
      elif config.bindMode == "bindModeMonitor":
         cmds.append( "bind mode monitor" )
      elif config.bindMode == "bindModeInterface" and saveDefault:
         cmds.append( 'bind mode interface' )
   vlan = Vlan.vlanSetToCanonicalString( config.bindVlan )
   if vlan:
      cmds.append( "bind vlan %s" % vlan )
   if config.minimumVersion13:
      cmds.append( "controller version minimum 1.3" )
   for bindIntf in Intf.IntfRange.intfListToCanonical( config.bindInterface ):
      cmds.append( "bind interface %s" % bindIntf )
   for vlan, rvlan in sorted( config.routedVlan.items() ):
      if vlan == 0:
         vlan = "untagged"
      cmds.append( "routing vlan %s routed-vlan %s" % (vlan, rvlan) )
   if config.recircInterface:
      cmds.append( "routing recirculation-interface %s" %
                   config.recircInterface )
   elif saveDefault:
      cmds.append( 'no routing recirculation-interface' )
   tableProfileMap = { 'tableProfileAutoMatch': 'auto',
                       'tableProfileL2Match': 'l2-match',
                       'tableProfileIpv6Match': 'ipv6-match',
                       'tableProfileIpv6ProtoPortMatch': 'ipv6-port-match',
                       'tableProfileVxlanMatch': 'vxlan-match' }
   if config.tableProfile in tableProfileMap.keys():
      cmds.append( "profile %s" % tableProfileMap[ config.tableProfile ] )
   elif saveDefault:
      cmds.append( "profile full-match" )
   if config.invalidTtlFlow:
      cmds.append( "exception ttl expired action controller" )
   if config.defaultFlowEntryActions != config.defaultFlowEntryActionsController():
      cmds.append( "default-action drop" )
   elif saveDefault:
      cmds.append( "default-action controller" )
   if config.shellCommandAllowed:
      cmds.append( "shell-command allowed" )
   elif saveDefault:
      cmds.append( "no shell-command allowed" )
   if config.enabled:
      cmds.append( "no shutdown" )
   elif saveDefault:
      cmds.append( "shutdown" )
   if bool( config.controller ):
      saveControllerConfig( config, root, saveDefault )
   elif saveDefault:
      cmds.append( "no controller" )

   # TLV Extension is a hidden CLI command, don't save default value
   if config.tracerTlv:
      cmds.append( "extension tlv tracer" )

   if cmds:
      mode = root[ OpenFlowConfigMode ].getOrCreateModeInstance( "openflow" )
      cs = mode[ "OpenFlow" ]
      for c in cmds:
         cs.addCommand( c )

def saveControllerConfig( config, root, saveDefault ):
   mode = root[ OpenFlowConfigMode ].getOrCreateModeInstance( "openflow" )
   for controller in config.controller:
      controllerMode = mode[ OpenFlowControllerConfigMode ].getOrCreateModeInstance(
            controller )
      cmds = controllerMode[ "OfController.Config" ]
      if config.controller[ controller ].enableAuxConnection:
         cmds.addCommand( "connection auxiliary udp" )
      elif saveDefault:
         cmds.addCommand( "no connection auxiliary udp" )
      controller = config.controller[ controller ]
      if controller.connectionMode != controller.defaultConnectionMode:
         cmds.addCommand( "connection mode %s" % controller.connectionMode )
      elif saveDefault:
         cmds.addCommand( "connection mode %s" % controller.defaultConnectionMode )

