# Copyright (c) 2007, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import CliSave, NtpLib
from IpLibConsts import DEFAULT_VRF
from IntfCliSave import IntfConfigMode
from RoutingIntfUtils import allIpIntfNames
import Toggles.NtpToggleLib

featureServeMultiVrfEnabled = Toggles.NtpToggleLib.toggleNtpServeMultiVrfEnabled()

CliSave.GlobalConfigMode.addCommandSequence( 'Ntp.config' )
IntfConfigMode.addCommandSequence( 'Ntp.intfConfig' )

@CliSave.saver( 'Ntp::Config', 'sys/time/ntp/config',
                requireMounts=( 'interface/config/all', 
                                'interface/status/all', 'acl/cpconfig/cli' ) )
def saveConfig( entity, root, sysdbRoot, options,
                requireMounts ):
   cmds = root[ 'Ntp.config' ]

   for key in entity.symmetricKey.itervalues():
      cmd = "ntp authentication-key %d %s 7 %s" % ( key.keyId, 
            key.keyType, 
            CliSave.sanitizedOutput( options, key.secret ) )
      cmds.addCommand( cmd )

   if entity.trustedKeys:
      cmd = "ntp trusted-key %s" % entity.trustedKeys
      cmds.addCommand( cmd )
   elif options.saveAll:
      cmd = "no ntp trusted-key"
      cmds.addCommand( cmd )

   if entity.authMode == NtpLib.authModeAll:
      cmd = "ntp authenticate"
      cmds.addCommand( cmd )
   elif entity.authMode == NtpLib.authModeServers:
      cmd = "ntp authenticate servers"
      cmds.addCommand( cmd )
   elif options.saveAll:
      cmd = "no ntp authenticate"
      cmds.addCommand( cmd )

   if entity.defaultSourceIntf.intf:
      cmd = 'ntp local-interface'

      if entity.defaultSourceIntf.vrf != DEFAULT_VRF:
         cmd = "%s vrf %s" % ( cmd, entity.defaultSourceIntf.vrf )
      cmd = "%s %s" % ( cmd, entity.defaultSourceIntf.intf )
      cmds.addCommand( cmd )

   if entity.dscpValue != entity.dscpValueInvalid:
      cmds.addCommand( "ntp qos dscp %d" % entity.dscpValue )

   servers = entity.server.values()
   for server in sorted( servers, key=lambda server: server.vrfAndHost,
                         cmp=NtpLib.compareVrfAndHost ):
      serverCmd = "ntp server"
      if server.vrf != DEFAULT_VRF:
         serverCmd += " vrf %s" % server.vrf
      serverCmd += " %s" % server.ipOrHost
      for option in [ "prefer", "burst", "iburst" ]:
         if getattr( server, option ):
            serverCmd += " " + option
      for option in [ "version", "minpoll", "maxpoll" ]:
         value = getattr( server, option )
         if value:
            serverCmd += " {option} {value}".format( **locals() )
      if server.sourceIntf:
         sourceToken = "source" if server.sourceUsed else "local-interface"
         serverCmd += " %s %s" % ( sourceToken, server.sourceIntf )
      if server.keyId != 0:
         serverCmd += " key %d" % server.keyId 
      cmds.addCommand( serverCmd )

   if entity.serverModeEnabledDefault:
      cmds.addCommand( "ntp serve all" )
   elif options.saveAll:
      cmds.addCommand( "no ntp serve all" )

   if featureServeMultiVrfEnabled:
      serveVrfs = entity.serveVrfName.keys()
      for serveVrf in serveVrfs:
         serveVrfCmd = "ntp serve all vrf " + serveVrf
         cmds.addCommand( serveVrfCmd )

   aclCpConfig = requireMounts[ 'acl/cpconfig/cli' ]
   for aclType in ( 'ip', 'ipv6' ):
      serviceAcl = aclCpConfig.cpConfig[ aclType ].serviceAcl
      for vrfName, serviceAclVrfConfig in serviceAcl.iteritems():
         serviceAclConfig = serviceAclVrfConfig.service.get( 'ntp' )
         if serviceAclConfig and serviceAclConfig.aclName:
            if vrfName == DEFAULT_VRF:
               vrfArg = ""
            else:
               vrfArg = " vrf " + vrfName
            cmds.addCommand( 'ntp serve %s access-group %s%s in' %
                             ( aclType, serviceAclConfig.aclName, vrfArg ) )

   if options.saveAll:
      cfgIntfNames = set( allIpIntfNames( sysdbRoot, requireMounts=requireMounts ) +
                          entity.serverModeEnabledIntf.keys() +
                          entity.serverModeDisabledIntf.keys() )
   else:
      cfgIntfNames = set( entity.serverModeEnabledIntf.keys() +
                          entity.serverModeDisabledIntf.keys() )

   for intfName in cfgIntfNames:
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intfName )
      cmds = mode[ "Ntp.intfConfig" ]
      if intfName in entity.serverModeDisabledIntf:
         cmds.addCommand( "no ntp serve" )
      elif intfName in entity.serverModeEnabledIntf:
         cmds.addCommand( "ntp serve" )
      elif options.saveAll:
         cmds.addCommand( "default ntp serve" )

   if entity.restartsForcedOnIntfChanges:
      cmds.addCommand( "ntp force-restarts" )
