#!/usr/bin/env python
# Copyright (c) 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import CliSave
from IntfCliSave import IntfConfigMode
from RoutingIntfUtils import allRoutingProtocolIntfNames
from IpLibConsts import DEFAULT_VRF
from CliMode.Mroute import RoutingMulticastMode, RoutingMulticastVrfMode
from CliMode.Mroute import RoutingMulticastAfMode
from McastCommonCliLib import AddressFamily, mcastRoutingSupported
from McastCommonCliLib import mcast6RoutingSupported

class RouterMulticastBaseConfigMode( RoutingMulticastMode, CliSave.Mode ):
   def __init__( self, param ):
      RoutingMulticastMode.__init__( self, param)
      CliSave.Mode.__init__( self, param )

class RouterMulticastVrfConfigMode( RoutingMulticastVrfMode, CliSave.Mode ):
   def __init__( self, param ):
      RoutingMulticastVrfMode.__init__( self, param)
      CliSave.Mode.__init__( self, param )

class RouterMulticastAfConfigMode( RoutingMulticastAfMode, CliSave.Mode ):
   def __init__( self, ( vrfName, af ) ):
      RoutingMulticastAfMode.__init__( self, vrfName, af )
      CliSave.Mode.__init__( self, ( vrfName, af ) )


CliSave.GlobalConfigMode.addCommandSequence( 'Ip.Multicast',
                                             after=[ 'Ira.routing' ] )
CliSave.GlobalConfigMode.addChildMode( RouterMulticastBaseConfigMode,
                                       after=[ 'Ip.Multicast' ] )
RouterMulticastBaseConfigMode.addCommandSequence( 'Multicast.config' )
RouterMulticastBaseConfigMode.addChildMode( RouterMulticastVrfConfigMode)
RouterMulticastBaseConfigMode.addChildMode( RouterMulticastAfConfigMode )

RouterMulticastVrfConfigMode.addCommandSequence( 'Multicast.vrf.config' )
RouterMulticastVrfConfigMode.addChildMode( RouterMulticastAfConfigMode )

RouterMulticastAfConfigMode.addCommandSequence( 'Multicast.vrf.af.config' )

IntfConfigMode.addCommandSequence( 'Mfib.intf', after=[ 'Ira.ipIntf' ] )


MfibVrfConfig = Tac.Type( 'Routing::Multicast::Fib::VrfConfig' )
LegacyConfig = Tac.Type( "Routing::Multicast::MulticastLegacyConfig" )

def getCmdRoot( root, vrfName, af, version, prefix='Multicast' ):
   def getRouterMode():
      return root[ RouterMulticastBaseConfigMode ].getSingletonInstance()

   def getRouterVrfMode( vrfName ):
      routerMode = getRouterMode()
      return routerMode[ RouterMulticastVrfConfigMode ].getOrCreateModeInstance(
                                                        vrfName )

   def getIpMode( vrfName, af ):
      if vrfName == DEFAULT_VRF:
         parentMode = getRouterMode()
      else:
         parentMode = getRouterVrfMode( vrfName )
      return parentMode[ RouterMulticastAfConfigMode ].getOrCreateModeInstance(
                                                       ( vrfName, af ) )

   if version == LegacyConfig.ipMode:
      cmds = getIpMode( vrfName, af )[ '%s.vrf.af.config' % prefix ]
   elif vrfName == DEFAULT_VRF:
      if version == LegacyConfig.globalMode:
         cmds = root[ 'Ip.%s' % prefix ]
      else:
         cmds = getRouterMode()[ '%s.config' % prefix ]
   else:
      cmds = getRouterVrfMode( vrfName )[ '%s.vrf.config' % prefix ]

   return cmds


def getCliSaveVersion( version, saveAll, configIsDefault=True ):
   # If legacy commands are deprecated, save in the new format
   return LegacyConfig.ipMode

# Save attributes in Routing::Multicast::Fib::VrfConfig
mrouteRootCliSavers = []
def mrouteRootCliSaver( func ):
   mrouteRootCliSavers.append( func )


@mrouteRootCliSaver
def saveCounterMode( config, cmds, saveAll ):
   cmd = 'counters'
   if config.counterMode != config.counterModeDefault:
      cmds.addCommand( "%s packets" % cmd )
   elif saveAll:
      cmds.addCommand( "%s bytes" % cmd )


@mrouteRootCliSaver
def saveActivityPollingInterval( config, cmds, saveAll ):
   if config.activityPollingInterval != config.activityPollingIntervalDefault or \
         saveAll:
      cmd = 'activity polling-interval'
      cmds.addCommand( "%s %d" % ( cmd, config.activityPollingInterval ) )


# Save attribures in Routing::Multicast::Fib::Config
mrouteVrfCliSavers = []
def mrouteVrfCliSaver( func ):
   mrouteVrfCliSavers.append( func )


@mrouteVrfCliSaver
def saveMulticastRouting( vrfConfig, cmds, saveAll, af ):
   cmd = 'routing'
   if vrfConfig.routing:
      cmds.addCommand( cmd )
   elif saveAll:
      cmds.addCommand( 'no %s' % cmd )

@mrouteVrfCliSaver
def saveMultipathControl( vrfConfig, cmds, saveAll, af ):
   cmd = 'multipath'
   MultipathControlType = Tac.Type(
                          "Routing::Multicast::Fib::Config::MultipathControl" )
   if vrfConfig.multipathControl == MultipathControlType.multipathNone:
      cmds.addCommand( '%s none' % cmd )
   elif vrfConfig.multipathControl == MultipathControlType.noMultipathDeterministic:
      cmds.addCommand( 'no %s deterministic' % cmd )
   elif vrfConfig.multipathControl == MultipathControlType.multipathRouterId \
         and af == AddressFamily.ipv4:
      cmds.addCommand( 'multipath deterministic router-id' )
   elif saveAll and vrfConfig.multipathControl == \
         MultipathControlType.multipathDeterministic:
      cmds.addCommand( '%s deterministic' % cmd )

@mrouteVrfCliSaver
def saveFastdrop( vrfConfig, cmds, saveAll, af ):
   cmd = 'fastdrop'
   # Hidden command disaplay only when configured
   if not vrfConfig.fastdrop:
      cmds.addCommand( "no %s" % cmd )

@mrouteVrfCliSaver
def saveMaxFastdrops( vrfConfig, cmds, saveAll, af='ipv4' ):
   cmd = 'max-fastdrops'
   if vrfConfig.maxFastdrops != vrfConfig.maxFastdropsDefault or saveAll:
      cmds.addCommand( "%s %d" % ( cmd, vrfConfig.maxFastdrops ) )


@CliSave.saver( 'Routing::Multicast::Fib::VrfConfig',
                MfibVrfConfig.mountPath( AddressFamily.ipv4 ),
                requireMounts = ( 'routing/hardware/status',
                                  'interface/config/all', 'interface/status/all',
                                  'routing/hardware/multicast/status'
                                  ) )
def saveMulticastVrfConfig( entity, root, sysdbRoot, options,
                         requireMounts ):

   saveAll = options.saveAll
   saveAllDetail = options.saveAllDetail

   # Save defaults only if the platform supports multicast routing
   if not mcastRoutingSupported(
         sysdbRoot,
         requireMounts[ 'routing/hardware/status' ] ) and \
         not saveAllDetail:
      saveAll = False

   multicastCliVersion = LegacyConfig.ipMode

   if not entity.isDefault() or saveAll:
      cmds = getCmdRoot( root, DEFAULT_VRF, AddressFamily.ipv4,
                         multicastCliVersion )
      for saver in mrouteRootCliSavers:
         saver( entity, cmds, saveAll )

      # TODO:CLI that are not converted yet
      mfibHardwareStatus = requireMounts[ 'routing/hardware/multicast/status' ]
      # TEMP hack
      cmds = getCmdRoot( root, DEFAULT_VRF, AddressFamily.ipv4,
                         LegacyConfig.routerMode )
      if not mfibHardwareStatus.mcastFlexCounterSupport:
         if entity.allCounter:
            cmds.addCommand( "ip multicast counters" )
         elif saveAll:
            cmds.addCommand( "no ip multicast counters" )

   for vrfName, config in entity.config.items():
      if not config.isDefault() or saveAll:
         cmds = getCmdRoot( root, vrfName, AddressFamily.ipv4,
                            multicastCliVersion )
         for saver in mrouteVrfCliSavers:
            saver( config, cmds, saveAll, af=AddressFamily.ipv4 )

   if saveAllDetail:
      cfgIntfNames = allRoutingProtocolIntfNames( sysdbRoot, includeEligible=True,
                                                  requireMounts=requireMounts )
   elif saveAll:
      # Routing configuration is allowed on switchports as well.
      # Interfaces for the configuration are routing protocol interfaces and
      # switchports with non-default config.
      cfgIntfNames = set(
            allRoutingProtocolIntfNames( sysdbRoot, requireMounts=requireMounts ) +
            entity.intfConfig.keys() )
   else:
      cfgIntfNames = entity.intfConfig

   for intfName in cfgIntfNames:
      intfConfig = entity.intfConfig.get( intfName )
      if not intfConfig:
         if saveAll:
            intfConfig = Tac.newInstance( 'Routing::Multicast::Fib::IntfConfig',
                                          intfName )
         else:
            continue
      saveIntfConfig( intfConfig, root, sysdbRoot, saveAll, saveAllDetail )

def saveIntfConfig( mfibIntfConfig, root, sysdbRoot, saveAll, saveAllDetail ):
   mic = mfibIntfConfig
   command = None
   if mic.noFastdrop:
      command = 'no mfib ipv4 fastdrop'
   elif saveAll:
      command = 'mfib ipv4 fastdrop'

   if command:
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( mic.name )
      cmds = mode[ 'Mfib.intf' ]
      cmds.addCommand( command )

@CliSave.saver( 'Routing::Multicast::Fib::VrfConfig',
                MfibVrfConfig.mountPath( AddressFamily.ipv6 ),
                requireMounts=( 'routing6/hardware/status', ) )
def saveIpv6RouterMulticastConfig( ipv6ConfigRoot, root, sysdbRoot, options,
                                   requireMounts ):
   hardwareStatus = requireMounts[ 'routing6/hardware/status' ]

   saveAll = options.saveAll
   saveAllDetail = options.saveAllDetail
   if not mcast6RoutingSupported( sysdbRoot, hardwareStatus ) and not saveAllDetail:
      saveAll = False

   multicastCliVersion = LegacyConfig.ipMode

   if not ipv6ConfigRoot.isDefault() or saveAll:
      cmds = getCmdRoot( root, DEFAULT_VRF, AddressFamily.ipv6, multicastCliVersion )
      for saver in mrouteRootCliSavers:
         saver( ipv6ConfigRoot, cmds, saveAll )

   for vrfName, ipv6Config in ipv6ConfigRoot.config.items():
      if saveAll or not ipv6Config.isDefault():
         cmds = getCmdRoot( root, vrfName, AddressFamily.ipv6, multicastCliVersion )
         for saver in mrouteVrfCliSavers:
            saver( ipv6Config, cmds, saveAll, af=AddressFamily.ipv6 )
