#!/usr/bin/env python
# Copyright (c) 2008, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import CliSave
from IntfCliSave import IntfConfigMode
import Arnet
from MirroringLib import metadataElementTokenDict
from IpLibConsts import DEFAULT_VRF

MetadataElementType = Tac.Type( "Mirroring::MetadataElementType" )

TruncationSize = Tac.Type( "Mirroring::TruncationSize" )

CliSave.GlobalConfigMode.addCommandSequence( 'Mirroring.global',
                                             after=[ IntfConfigMode ] )

@CliSave.saver( 'Mirroring::Config', 'mirroring/config',
                secureMonitor=True )
def saveMirroringConfig( entity, root, sysdbRoot, options ):
   cmds = root[ 'Mirroring.global' ]
   def saveMirroringSession():
      for sessionName in sorted( entity.session ):
         session = entity.session[ sessionName ]
         if ( not ( session.isInternal or session.isHiddenSession ) and
              session.secureMonitor == options.secureMonitor ):
            if session.sessionAcl != '':
               cmds.addCommand( 'monitor session %s %s access-group %s' % \
                     ( sessionName, entity.session[ sessionName ].aclType,
                       entity.session[ sessionName ].sessionAcl ) )

            for intf in Arnet.sortIntf( session.srcIntf ):
               src = session.srcIntf[ intf ]
               direction = ''
               if src.direction == 'directionRx':
                  direction = ' rx'
               elif src.direction == 'directionTx':
                  direction = ' tx'

               cmd = 'monitor session %s source %s%s' % (
                        sessionName, intf, direction )

               if not session.sessionAcl and src.srcAcl != '':
                  cmd += ' %s access-group %s' % ( src.aclType, src.srcAcl )
                  if src.aclPriority > 0:
                     cmd += ' priority %d' % src.aclPriority

               cmds.addCommand( cmd )

            for intf in Arnet.sortIntf( entity.session[ sessionName ].targetIntf ):
               if intf.startswith( "Gre" ):
                  continue
               cmds.addCommand( 'monitor session %s destination %s'
                                % ( sessionName, intf )  )

            for greTunnelKey in [ session.greTunnelKey, session.dropGreTunnelKey ]:
               if not greTunnelKey.dstIpGenAddr.isAddrZero:
                  if greTunnelKey.forwardingDrop:
                     forwardingDrop = " forwarding-drop"
                  else:
                     forwardingDrop = ""
                  tunnelParams = ""
                  if greTunnelKey.ttl != 128:
                     tunnelParams += "ttl %d " % greTunnelKey.ttl
                  if greTunnelKey.dscp:
                     tunnelParams += "dscp %d " % greTunnelKey.dscp
                  if greTunnelKey.greHdrProto != greTunnelKey.defaultGreHdrProto:
                     tunnelParams += "protocol 0x%x " % greTunnelKey.greHdrProto
                  if greTunnelKey.vrf != DEFAULT_VRF:
                     tunnelParams += "vrf %s" % greTunnelKey.vrf
                  destIntf = 'tunnel mode gre source %s destination %s %s' % \
                              ( greTunnelKey.srcIpGenAddr,
                                greTunnelKey.dstIpGenAddr, tunnelParams )
                  cmds.addCommand( 'monitor session %s%s destination %s'
                                   % ( sessionName, forwardingDrop, destIntf ) )

            if entity.session[ sessionName ].truncate:
               if entity.session[ sessionName ].truncationSize != \
                     TruncationSize.null:
                  cmds.addCommand( 'monitor session %s truncate size %d' % \
                        ( sessionName,
                          entity.session[ sessionName ].truncationSize ) )
               else:
                  cmds.addCommand( 'monitor session %s truncate' % sessionName )

            if entity.session[ sessionName ].rxSampleRate != \
               Tac.Type( 'Mirroring::SampleRateConstants' ).sampleRateUndefined:
               cmds.addCommand( 'monitor session %s sample %d' %
                                ( sessionName,
                                  entity.session[ sessionName ].rxSampleRate ) )

            if entity.session[ sessionName ].mirrorRateLimitInBps:
               cmds.addCommand( 'monitor session %s rate-limit %s %d %s' %
                  ( sessionName,
                    entity.session[ sessionName ].mirrorRateLimitChip,
                    entity.session[ sessionName ].mirrorRateLimitInBps / 1000,
                    'kbps' ) )

            if entity.session[ sessionName ].headerRemovalSize:
               cmds.addCommand( 'monitor session %s header remove size %d' %
                  ( sessionName,
                    entity.session[ sessionName ].headerRemovalSize ) )

            if entity.session[ sessionName ].txGreMetadataEnabled:
               cmds.addCommand( 'monitor session %s encapsulation gre metadata tx' %
                                sessionName )


   def saveMirroringDefaultSessionConfig():
      grePayloadType = entity.defaultSessionConfig.grePayloadType
      if grePayloadType != 'payloadTypeHwDefault':
         cmd = 'monitor session default encapsulation gre payload '
         if grePayloadType == 'payloadTypeFullPacket':
            cmd += 'full-packet'
         elif grePayloadType == 'payloadTypeInnerPacket':
            cmd += 'inner-packet'
         else:
            cmd = 'no ' + cmd
         cmds.addCommand( cmd )

      greMetadata = entity.defaultSessionConfig.greMetadata
      if greMetadata:
         cmd = 'monitor session default encapsulation gre metadata '
         
         # The metadata elements must be in correct order
         metadataType1 = [ MetadataElementType.egressPort,
                           MetadataElementType.ingressPort ]
         metadataType2 = [ MetadataElementType.ingressVlan,
                           MetadataElementType.ingressPort ]
         if set( greMetadata.keys() ) == set( metadataType1 ):
            elements = metadataType1
         elif set( greMetadata.keys() ) == set( metadataType2 ):
            elements = metadataType2
         cmd += ' '.join( metadataElementTokenDict[ el ] for el in elements )
         cmds.addCommand( cmd )

      greTimestampingEnabled = entity.defaultSessionConfig.greTimestampingEnabled
      if greTimestampingEnabled:
         cmd = 'monitor session default encapsulation gre timestamp'
         cmds.addCommand( cmd )
      
      portMirroringDisabled = entity.defaultSessionConfig.portMirroringDisabled
      if portMirroringDisabled:
         cmd = 'monitor session default-action no-mirror'
         cmds.addCommand( cmd )

   saveMirroringSession()
   saveMirroringDefaultSessionConfig()
