# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

from collections import namedtuple
import CliSave
import CliSavePlugin.IntfCliSave as IntfCliSave
import Tac

IrVxlanIpv4TunnelId = Tac.Type( "Multicast::Tunnel::IrVxlanIpv4TunnelId" )
McastTunnelIntfId = Tac.Type( "Multicast::Tunnel::McastTunnelIntfId" )
PimVxlanTunnel = Tac.Type( "Multicast::Tunnel::PimVxlanTunnel" )
McastTunnelType = Tac.Type( "Multicast::Tunnel::McastTunnelType" )

pimTunnelTypeToKeyword = { 'pimsmIpv4Tunnel': 'ipv4-pim-sm',
                           'pimssmIpv4Tunnel': 'ipv4-pim-ssm',
                           'pimsmIpv6Tunnel': 'ipv6-pim-sm',
                           'pimssmIpv6Tunnel': 'ipv6-pim-ssm' }

def intfBackwardConvert( intfId ):
   keyword = None
   tunnelEndpoint = None
   tunnelId = McastTunnelIntfId.tunnelId( intfId )
   if tunnelId.tunnelType == McastTunnelType.pimVxlan:
      keyword = 'tunnel'
      tunnelEndpoint = pimTunnelTypeToKeyword[
            PimVxlanTunnel.makePimTunnelType( intfId ) ]
   elif tunnelId.tunnelType == McastTunnelType.irVxlanIpv4:
      keyword = 'vtep'
      tunnelEndpoint = IrVxlanIpv4TunnelId.vtepIp( intfId )
   return keyword, tunnelEndpoint

def getIntfCmdsHelper( vtepCmd, tunnelCmd, portCmd, etCmd, demuxIntfs ):
   intfCmds = []
   if demuxIntfs.vtepIntfs:
      vtepCmd += ' '.join( sorted( demuxIntfs.vtepIntfs ) )
      intfCmds.append( vtepCmd )
   if demuxIntfs.tunnelIntfs:
      tunnelCmd += ' '.join( sorted( demuxIntfs.tunnelIntfs ) )
      intfCmds.append( tunnelCmd )
   if demuxIntfs.portIntfs:
      portCmd += ','.join( sorted( demuxIntfs.portIntfs ) )
      intfCmds.append( portCmd )
   if demuxIntfs.etIntfs:
      etCmd += ','.join( sorted( demuxIntfs.etIntfs ) )
      intfCmds.append( etCmd )
   return intfCmds

def demuxByInterfaceTypes( intfs ):
   demuxIntfs = namedtuple( 'demuxIntfs',
                            'vtepIntfs tunnelIntfs portIntfs etIntfs' )
   vtepIntfs = []
   tunnelIntfs = []
   portIntfs = []
   etIntfs = []
   for intf in intfs:
      if McastTunnelIntfId.isMcastTunnelIntfId( intf ):
         keyword, tunnelEndpoint = intfBackwardConvert( intf )
         if keyword == 'vtep':
            vtepIntfs.append( tunnelEndpoint )
         else:
            tunnelIntfs.append( tunnelEndpoint )
      elif 'Port-Channel' in intf:
         portIntfs.append( intf )
      else:
         etIntfs.append( intf )
   return demuxIntfs( vtepIntfs, tunnelIntfs, portIntfs, etIntfs )

def getIntfCmds( membershipJoinStatus ):
   intfCmds = []
   vlanMembershipJoin = membershipJoinStatus.vlan
   for vlanId in sorted( vlanMembershipJoin ):
      routerVtepCmd = 'vxlan vlan %d member vtep ' % vlanId
      routerTunnelCmd = 'vxlan vlan %d member tunnel ' % vlanId
      routerPortCmd = 'vxlan vlan %d member ' % vlanId
      routerEtCmd = 'vxlan vlan %d member ' % vlanId
      demuxIntfs = demuxByInterfaceTypes(
            sorted( vlanMembershipJoin[ vlanId ].routerIntf ) )
      intfCmds += getIntfCmdsHelper( routerVtepCmd, routerTunnelCmd,
                                     routerPortCmd, routerEtCmd,
                                     demuxIntfs )

      for groupAddr in sorted( vlanMembershipJoin[ vlanId ].group ):
         sourceMembershipJoin = vlanMembershipJoin[ vlanId ] \
                                .group[ groupAddr ].source
         for sourceAddr in sorted( sourceMembershipJoin ):
            includeVtepCmd = "vxlan vlan %d member %s %s vtep " \
                             % ( vlanId, groupAddr, sourceAddr )
            includeTunnelCmd = "vxlan vlan %d member %s %s tunnel " \
                               % ( vlanId, groupAddr, sourceAddr )
            includePortCmd = "vxlan vlan %d member %s %s " \
                             % ( vlanId, groupAddr, sourceAddr )
            includeEtCmd = "vxlan vlan %d member %s %s " \
                             % ( vlanId, groupAddr, sourceAddr )
            demuxIntfs = demuxByInterfaceTypes(
                  sorted( sourceMembershipJoin[ sourceAddr ].includeIntf ) )
            intfCmds += getIntfCmdsHelper( includeVtepCmd, includeTunnelCmd,
                                           includePortCmd, includeEtCmd,
                                           demuxIntfs )

            excludeVtepCmd = "vxlan vlan %d member exclude %s %s vtep " \
                             % ( vlanId, groupAddr, sourceAddr )
            excludeTunnelCmd = "vxlan vlan %d member exclude %s %s tunnel " \
                               % ( vlanId, groupAddr, sourceAddr )
            excludePortCmd = "vxlan vlan %d member exclude %s %s " \
                             % ( vlanId, groupAddr, sourceAddr )
            excludeEtCmd = "vxlan vlan %d member exclude %s %s " \
                             % ( vlanId, groupAddr, sourceAddr )
            demuxIntfs = demuxByInterfaceTypes(
                  sorted( sourceMembershipJoin[ sourceAddr ].excludeIntf ) )
            intfCmds += getIntfCmdsHelper( excludeVtepCmd, excludeTunnelCmd,
                                           excludePortCmd, excludeEtCmd,
                                           demuxIntfs )
   return intfCmds

@CliSave.saver( 'Irb::Multicast::Gmp::MembershipJoinStatusCli',
                'multicast/ipv4/irb/membership/join/cli' )
def saveIpv4MembershipJoinStatus( membershipJoinStatus, root, sysdbRoot, options ):
   intfCmds = getIntfCmds( membershipJoinStatus )
   if intfCmds:
      mode = root[ IntfCliSave.IntfConfigMode ].getOrCreateModeInstance( 'Vxlan1' )
      cmds = mode[ 'McastVpn.VxlanIntf' ]
      for cmd in intfCmds:
         cmds.addCommand( cmd )

@CliSave.saver( 'Irb::Multicast::Gmp::MembershipJoinStatusCli',
                'multicast/ipv6/irb/membership/join/cli' )
def saveIpv6MembershipJoinStatus( membershipJoinStatus, root, sysdbRoot, options ):
   intfCmds = getIntfCmds( membershipJoinStatus )
   if intfCmds:
      mode = root[ IntfCliSave.IntfConfigMode ].getOrCreateModeInstance( 'Vxlan1' )
      cmds = mode[ 'McastVpn.VxlanIntf' ]
      for cmd in intfCmds:
         cmds.addCommand( cmd )
