#!/usr/bin/env python
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
import CliSave
from RoutingIntfUtils import allRoutingProtocolIntfNames
from McastCommonCliLib import mcastRoutingSupported
from IntfCliSave import IntfConfigMode
from CliSavePlugin.MrouteCliSave import getCliSaveVersion
import Tac

IntfConfigMode.addCommandSequence( 'McastCommon.intf', after=[ 'Ira.ipIntf' ] )

AddressFamily = Tac.Type( "Arnet::AddressFamily" )
LegacyConfig = Tac.Type( "McastCommon::LegacyConfig" )
VlanIntfId = Tac.Type( "Arnet::VlanIntfId" )
sourceInjectDefaultPref = Tac.enumValue( "HostInject::Constants", "defaultPref" )

def saveMcastCommonConfig( mcastHostInjectConfig, root, sysdbRoot, options,
                           requireMounts, af ):
   saveAll = options.saveAll
   saveAllDetail = options.saveAllDetail

   # Save the default config only if the platform supports multicast routing
   if af == AddressFamily.ipv4:
      hwStatus = requireMounts[ 'routing/hardware/status' ]
   else:
      hwStatus = requireMounts[ 'routing6/hardware/status' ]
   if not mcastRoutingSupported(
         sysdbRoot,
         hwStatus ):
      saveAll = False
      saveAllDetail = False

   mcastLegacyConfig = requireMounts[ 'routing/multicast/legacyconfig' ]

   if saveAllDetail:
      cfgIntfNames = allRoutingProtocolIntfNames(
            sysdbRoot, includeEligible=True, requireMounts=requireMounts )
   elif saveAll:
      # Routing configuration is allowed on switchports as well.
      # Save configuration on all routing protocol interfaces and switchports
      # with non-default config.
      rpIntfNames = allRoutingProtocolIntfNames(
            sysdbRoot, requireMounts=requireMounts )
      intfIdSet = [ i.intfId for i in mcastHostInjectConfig.config.keys() ]
      cfgIntfNames = rpIntfNames + intfIdSet
   else:
      cfgIntfNames = [ i.intfId for i in mcastHostInjectConfig.config.keys() ]

   for intfId in cfgIntfNames:
      cKey = Tac.newInstance( "HostInject::DefaultConfigKey", intfId )
      ic = mcastHostInjectConfig.config.get( cKey )
      if not ic and not saveAll:
         continue
      saveIntfConfig( ic, intfId, root, sysdbRoot, saveAll, saveAllDetail,
                      mcastLegacyConfig, af )

def saveIntfConfig( ic, intfName, root, sysdbRoot, saveAll, saveAllDetail,
                    mcastLegacyConfig, af ):
   mode = root[ IntfConfigMode ].getOrCreateModeInstance( intfName )
   cmds = mode[ 'McastCommon.intf' ]

   version = getCliSaveVersion( mcastLegacyConfig.version, saveAll )
   legacy = version != LegacyConfig.ipMode

   if af == AddressFamily.ipv4:
      if legacy:
         cmdPrefix = 'ip multicast '
      else:
         cmdPrefix = 'multicast %s ' % af
   elif af == AddressFamily.ipv6:
      cmdPrefix = 'multicast %s ' % af

   saveAllPrefix = 'no multicast %s ' % af

   if VlanIntfId.isVlanIntfId( Arnet.IntfId( intfName ) ):
      cmd = 'source route export'
      if ic:
         if ic.pref != sourceInjectDefaultPref:
            cmds.addCommand( cmdPrefix + cmd + ' %d' % ic.pref )
         else:
            cmds.addCommand( cmdPrefix + cmd )
      elif saveAll:
         cmds.addCommand( saveAllPrefix + cmd )

@CliSave.saver( 'HostInject::HostInjectConfig', 'routing/attached-host/mcast/config',
                requireMounts = ( 'routing/hardware/status',
                                  'interface/config/all', 'interface/status/all',
                                  'l3/config',
                                  'routing/multicast/legacyconfig' ) )
def saveMcastCommonIpv4Config( mcastHostInjectConfig, root, sysdbRoot, options,
                               requireMounts ):

   saveMcastCommonConfig( mcastHostInjectConfig, root, sysdbRoot, options,
                          requireMounts, AddressFamily.ipv4 )

@CliSave.saver( 'HostInject::HostInjectConfig',
                'routing6/attached-host/mcast/config',
                requireMounts = ( 'routing6/hardware/status',
                                  'interface/config/all', 'interface/status/all',
                                  'l3/config',
                                  'routing/multicast/legacyconfig' ) )
def saveMcastCommonIpv6Config( mcastHostInjectConfig, root, sysdbRoot, options,
                               requireMounts ):

   saveMcastCommonConfig( mcastHostInjectConfig, root, sysdbRoot, options,
                          requireMounts, AddressFamily.ipv6 )
